/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.beehive.db;

import com.atlassian.beehive.core.ManagedClusterLock;
import com.atlassian.beehive.core.stats.StatisticsKey;
import com.atlassian.beehive.db.ClusterNodeHeartbeatService;
import com.atlassian.beehive.db.StatisticsHolder;
import com.atlassian.beehive.db.spi.ClusterLockDao;
import com.atlassian.beehive.db.spi.ClusterLockStatus;
import com.google.common.annotations.VisibleForTesting;
import java.lang.ref.WeakReference;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.Condition;
import javax.annotation.Nonnull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class DatabaseClusterLock
implements ManagedClusterLock {
    private static final long RECHECK_INTERVAL_MILLIS = 10000L;
    private static final int INITIAL_SLEEP_MILLIS = 100;
    private static final int MAX_SLEEP_MILLIS = 10000;
    private static final Logger log = LoggerFactory.getLogger(DatabaseClusterLock.class);
    private final String lockName;
    private final ClusterLockDao clusterLockDao;
    private final ClusterNodeHeartbeatService clusterNodeHeartbeatService;
    private final String nodeId;
    private final AtomicReference<Owner> ownerRef = new AtomicReference();
    private final AtomicInteger depth = new AtomicInteger();
    private final AtomicLong lastCheck = new AtomicLong();
    private final StatisticsHolder stats = new StatisticsHolder();

    public DatabaseClusterLock(String lockName, ClusterLockDao clusterLockDao, ClusterNodeHeartbeatService clusterNodeHeartbeatService) {
        this.lockName = lockName;
        this.clusterLockDao = clusterLockDao;
        this.clusterNodeHeartbeatService = clusterNodeHeartbeatService;
        this.nodeId = clusterNodeHeartbeatService.getNodeId();
    }

    @Nonnull
    public String getName() {
        return this.lockName;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void lock() {
        long startedAt = this.nowInMillis();
        boolean wasInterrupted = Thread.interrupted();
        if (!this.tryLock()) {
            this.stats.tallyWaitBegin();
            try {
                this.uninterruptibleWait();
            }
            finally {
                this.stats.tallyWaitEndAfter(this.nowInMillis() - startedAt);
            }
        }
        DatabaseClusterLock.interruptIf(wasInterrupted);
    }

    private void uninterruptibleWait() {
        boolean wasInterrupted = false;
        int sleepTimeMillis = 100;
        do {
            try {
                this.sleep(sleepTimeMillis);
            }
            catch (InterruptedException ie) {
                wasInterrupted = true;
            }
            sleepTimeMillis = Math.min(sleepTimeMillis * 2, 10000);
        } while (!this.tryLock());
        DatabaseClusterLock.interruptIf(wasInterrupted);
    }

    public void lockInterruptibly() throws InterruptedException {
        long startedAt = this.nowInMillis();
        if (Thread.interrupted()) {
            throw new InterruptedException();
        }
        if (this.tryLock()) {
            return;
        }
        this.stats.tallyWaitBegin();
        try {
            this.interruptibleWait();
        }
        finally {
            this.stats.tallyWaitEndAfter(this.nowInMillis() - startedAt);
        }
    }

    private void interruptibleWait() throws InterruptedException {
        int sleepTimeMillis = 100;
        do {
            this.sleep(sleepTimeMillis);
            sleepTimeMillis = Math.min(sleepTimeMillis * 2, 10000);
        } while (!this.tryLock());
    }

    public boolean tryLock() {
        Thread me = Thread.currentThread();
        log.debug("Attempt to get cluster lock '{}' by {}.", (Object)this.lockName, (Object)me);
        Owner owner = this.ownerRef.get();
        if (owner == null || !this.isLocalPermitValid(owner)) {
            return this.tryLockUsingDatabase();
        }
        if (this.tryReenter(owner, me)) {
            return true;
        }
        if (!this.hasDeadOwnerThread(owner)) {
            return false;
        }
        if (this.tryForceLocalUnlock(owner, me)) {
            return true;
        }
        log.debug("Cluster lock '{}' was stolen by another thread, so '{}' lost this race", (Object)this.lockName, (Object)me);
        this.stats.tallyFailLocal();
        return false;
    }

    private boolean tryReenter(Owner owner, Thread me) {
        if (owner.getThread() != me) {
            return false;
        }
        log.debug("Cluster lock '{}' reentered by '{}'", (Object)this.lockName, (Object)me);
        if (this.depth.incrementAndGet() < 0) {
            this.depth.decrementAndGet();
            throw new IllegalMonitorStateException("Maximum lock count exceeded");
        }
        return true;
    }

    private boolean hasDeadOwnerThread(Owner owner) {
        Thread ownerThread = owner.getThread();
        if (ownerThread == null || !ownerThread.isAlive()) {
            return true;
        }
        log.debug("Cluster lock '{}' currently held by another local thread '{}'.", (Object)this.lockName, (Object)ownerThread.getName());
        this.stats.tallyFailLocal();
        return false;
    }

    private boolean tryForceLocalUnlock(Owner owner, Thread me) {
        if (!this.ownerRef.compareAndSet(owner, new Owner(me))) {
            return false;
        }
        log.error("Cluster lock '{}' was not unlocked by '{}' before it terminated, so '{}' has stolen it", new Object[]{this.lockName, owner, me});
        this.stats.tallyForcedUnlock();
        return this.succeededAt(this.nowInMillis());
    }

    private boolean succeededAt(long currentTimeMillis) {
        this.depth.set(1);
        this.lastCheck.set(currentTimeMillis);
        this.stats.tallyLockedAt(currentTimeMillis);
        return true;
    }

    private boolean tryLockUsingDatabase() {
        long now;
        ClusterLockStatus clusterLockStatus = this.getClusterLockStatus();
        if (clusterLockStatus.getLockedByNode() != null && !clusterLockStatus.getLockedByNode().equals(this.nodeId)) {
            log.debug("Cluster lock '{}' currently held by node '{}'.", (Object)this.lockName, (Object)clusterLockStatus.getLockedByNode());
            long age = this.nowInMillis() - clusterLockStatus.getUpdateTime();
            if (age > 300000L) {
                clusterLockStatus = this.unlockIfDead(clusterLockStatus);
            }
        }
        Thread me = Thread.currentThread();
        if (clusterLockStatus.getLockedByNode() == null && this.clusterLockDao.tryUpdateAcquireLock(this.lockName, this.nodeId, now = this.nowInMillis())) {
            this.ownerRef.set(new Owner(me));
            log.debug("Cluster lock '{}' was acquired by {}.", (Object)this.lockName, (Object)me);
            return this.succeededAt(now);
        }
        log.debug("Acquisition of cluster lock '{}' by {} failed.", (Object)this.lockName, (Object)me);
        this.stats.tallyFailRemote();
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean tryLock(long waitTime, @Nonnull TimeUnit unit) throws InterruptedException {
        if (Thread.interrupted()) {
            throw new InterruptedException();
        }
        long startedAt = this.nowInMillis();
        if (this.tryLock()) {
            return true;
        }
        long deadline = startedAt + unit.toMillis(waitTime);
        this.stats.tallyWaitBegin();
        try {
            boolean bl = this.tryLockWaitWithTimeout(deadline);
            return bl;
        }
        finally {
            this.stats.tallyWaitEndAfter(this.nowInMillis() - startedAt);
        }
    }

    private boolean tryLockWaitWithTimeout(long deadline) throws InterruptedException {
        long sleepTimeMillis = 100L;
        do {
            long remainingWaitTime;
            if ((remainingWaitTime = deadline - this.nowInMillis()) <= 0L) {
                return false;
            }
            sleepTimeMillis = Math.min(sleepTimeMillis, remainingWaitTime);
            this.sleep(sleepTimeMillis);
            sleepTimeMillis = Math.min(sleepTimeMillis * 2L, 10000L);
        } while (!this.tryLock());
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void unlock() {
        Owner owner = this.ownerRef.get();
        Thread me = Thread.currentThread();
        if (owner == null || owner.getThread() != me || !this.isLocalPermitValid(owner)) {
            throw new IllegalMonitorStateException("Cluster lock '" + this.lockName + "' cannot be unlocked because it is not owned by this thread: " + me);
        }
        if (this.depth.decrementAndGet() > 0) {
            log.debug("Cluster lock '{}' re-entrance count decremented by '{}'", (Object)this.lockName, (Object)me);
            return;
        }
        long now = this.nowInMillis();
        log.debug("Cluster lock '{}' unlocked by '{}'", (Object)this.lockName, (Object)me);
        try {
            this.clusterLockDao.unlock(this.lockName, this.nodeId, now);
        }
        finally {
            this.stats.tallyUnlockedAt(now);
            this.ownerRef.compareAndSet(owner, null);
        }
    }

    public boolean isHeldByCurrentThread() {
        Owner owner = this.ownerRef.get();
        return owner != null && owner.getThread() == Thread.currentThread() && this.isLocalPermitValid(owner);
    }

    private boolean isLocalPermitValid(@Nonnull Owner owner) {
        long now = this.nowInMillis();
        if (now - this.lastCheck.get() < 10000L) {
            return true;
        }
        ClusterLockStatus clusterLockStatus = this.getClusterLockStatus();
        if (this.nodeId.equals(clusterLockStatus.getLockedByNode())) {
            return true;
        }
        if (this.ownerRef.compareAndSet(owner, null)) {
            log.error("Cluster lock '{}' was expected to already be held by this node, but its current owner is '{}'.", (Object)this.lockName, (Object)clusterLockStatus.getLockedByNode());
            this.stats.tallyForcedUnlock();
        }
        return false;
    }

    @Nonnull
    public Condition newCondition() {
        throw new UnsupportedOperationException("newCondition() not supported in ClusterLock");
    }

    private ClusterLockStatus getClusterLockStatus() {
        long now = this.nowInMillis();
        this.lastCheck.set(now);
        ClusterLockStatus lock = this.clusterLockDao.getClusterLockStatusByName(this.lockName);
        if (lock != null) {
            return lock;
        }
        this.clusterLockDao.insertEmptyClusterLock(this.lockName, now);
        return new ClusterLockStatus(this.lockName, null, now);
    }

    private ClusterLockStatus unlockIfDead(ClusterLockStatus lock) {
        if (this.clusterNodeHeartbeatService.isNodeLive(lock.getLockedByNode())) {
            return lock;
        }
        log.warn("Releasing lock '" + lock.getLockName() + "' from node '" + lock.getLockedByNode() + "' because the node has stopped heart-beating.");
        long updateTime = this.nowInMillis();
        this.clusterLockDao.unlock(lock.getLockName(), lock.getLockedByNode(), updateTime);
        this.stats.tallyForcedUnlock();
        return new ClusterLockStatus(this.lockName, null, updateTime);
    }

    @Nonnull
    public Map<StatisticsKey, Long> getStatistics() {
        return this.stats.getStatistics(this.lastCheck);
    }

    @VisibleForTesting
    long nowInMillis() {
        return System.currentTimeMillis();
    }

    @VisibleForTesting
    void sleep(long timeout) throws InterruptedException {
        Thread.sleep(timeout);
    }

    private static void interruptIf(boolean wasInterrupted) {
        if (wasInterrupted) {
            Thread.currentThread().interrupt();
        }
    }

    static class Owner {
        private final WeakReference<Thread> thd;
        private final String name;

        Owner(Thread me) {
            this.thd = new WeakReference<Thread>(me);
            this.name = me.getName();
        }

        Thread getThread() {
            return (Thread)this.thd.get();
        }

        public String toString() {
            return this.name;
        }
    }
}

