/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.asap.core.validator;

import com.atlassian.asap.api.Jwt;
import com.atlassian.asap.api.JwtClaims;
import com.atlassian.asap.api.exception.InvalidTokenException;
import com.atlassian.asap.core.exception.InvalidClaimException;
import com.atlassian.asap.core.exception.TokenExpiredException;
import com.atlassian.asap.core.exception.TokenTooEarlyException;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JwtClaimsValidator {
    public static final Duration TIME_CLAIM_LEEWAY = Duration.ofSeconds(Long.parseLong(System.getProperty("asap.resource.server.leeway.seconds", "30")));
    public static final Duration MAX_LIFETIME = Duration.ofHours(1L);
    private static final Logger logger = LoggerFactory.getLogger(JwtClaimsValidator.class);
    private final Clock clock;

    public JwtClaimsValidator(Clock clock) {
        this.clock = Objects.requireNonNull(clock);
    }

    public void validate(Jwt jwt, String resourceServerAudience) throws InvalidTokenException {
        JwtClaims claims = jwt.getClaims();
        String issuer = claims.getIssuer();
        Optional<String> subject = claims.getSubject();
        Set<String> audience = claims.getAudience();
        Instant issuedAt = claims.getIssuedAt();
        Instant expiry = claims.getExpiry();
        Optional<Instant> mayBeNotBefore = claims.getNotBefore();
        String keyId = jwt.getHeader().getKeyId();
        this.issuerAndSubjectValidation(issuer, subject, keyId);
        this.audienceValidation(audience, resourceServerAudience);
        this.formalTimeClaimsValidation(issuedAt, expiry, mayBeNotBefore);
        this.relativeTimeValidation(issuedAt, expiry, mayBeNotBefore);
    }

    private void issuerAndSubjectValidation(String issuer, Optional<String> subject, String keyId) throws InvalidClaimException {
        if (StringUtils.isBlank((CharSequence)issuer)) {
            logger.debug("Rejecting blank issuer");
            throw new InvalidClaimException("Issuer cannot be blank");
        }
        if (!keyId.startsWith(issuer + "/")) {
            logger.debug("The issuer {} does not match the key id {}", (Object)issuer, (Object)keyId);
            throw new InvalidClaimException("The issuer claim does not match the key id");
        }
    }

    private void audienceValidation(Set<String> audience, String resourceServerAudience) throws InvalidClaimException {
        if (!audience.contains(resourceServerAudience)) {
            logger.debug("Rejected unrecognised audience {}", audience);
            throw new InvalidClaimException("Unrecongised audience");
        }
    }

    private void formalTimeClaimsValidation(Instant issuedAt, Instant expiry, Optional<Instant> mayBeNotBefore) throws InvalidClaimException {
        if (expiry.isBefore(issuedAt)) {
            logger.debug("Expiry time {} set before issue time {}", (Object)expiry, (Object)issuedAt);
            throw new InvalidClaimException("Expiry time set before issue time");
        }
        if (issuedAt.plus(MAX_LIFETIME).isBefore(expiry)) {
            logger.debug("Token exceeds lifetime limit, issued at {} and expires at {}", (Object)issuedAt, (Object)expiry);
            throw new InvalidClaimException("Token exceeds lifetime limit");
        }
        if (mayBeNotBefore.isPresent()) {
            Instant nbf = mayBeNotBefore.get();
            if (nbf.isAfter(expiry)) {
                logger.debug("The expiry time {} must be after the not-before time {}", (Object)expiry, (Object)nbf);
                throw new InvalidClaimException("The expiry time must be after the not-before time");
            }
            if (nbf.isBefore(issuedAt)) {
                logger.debug("The token was valid since {} but was issued later at {}", (Object)nbf, (Object)issuedAt);
                throw new InvalidClaimException("The token must not be valid before it was issued");
            }
        }
    }

    private void relativeTimeValidation(Instant issuedAt, Instant expiry, Optional<Instant> mayBeNotBefore) throws TokenExpiredException, TokenTooEarlyException {
        Instant now = Instant.now(this.clock);
        Instant nowMinusLeeway = now.minus(TIME_CLAIM_LEEWAY);
        Instant nowPlusLeeway = now.plus(TIME_CLAIM_LEEWAY);
        if (expiry.isBefore(nowMinusLeeway)) {
            logger.info("Rejecting expired token, now={}, expiry={}, leeway={}", new Object[]{now, expiry, TIME_CLAIM_LEEWAY});
            throw new TokenExpiredException(expiry, now);
        }
        Instant effectiveNbf = mayBeNotBefore.orElse(issuedAt);
        if (effectiveNbf.isAfter(nowPlusLeeway)) {
            logger.info("Rejecting token that arrives too early, now={}, not before={}, leeway={}", new Object[]{now, effectiveNbf, TIME_CLAIM_LEEWAY});
            throw new TokenTooEarlyException(effectiveNbf, now);
        }
    }
}

