define("workflow-designer/transition-view", [
    "workflow-designer/dialogs/edit-transition-dialog-view",
    "workflow-designer/dialogs/delete-transition-dialog-view",
    "workflow-designer/policy/feedback/line-selection-feedback-policy",
    "workflow-designer/transition-connection",
    "workflow-designer/collection",
    "workflow-designer/selectable-figure-mixin",
    "workflow-designer/request-response-mixin",
    "workflow-designer/draw-2d-view",
    "workflow-designer/cocktail",
    "workflow-designer/underscore"
], function(
    EditTransitionDialogView,
    DeleteTransitionDialogView,
    LineSelectionFeedbackPolicy,
    TransitionConnection,
    Collection,
    SelectableFigureMixin,
    RequestResponseMixin,
    Draw2DView,
    Cocktail,
    _
) {
    var TransitionView = Draw2DView.extend(
    /** @lends JIRA.WorkflowDesigner.TransitionView# */
    {
        /**
         * Initialise the view.
         *
         * @classdesc The representation of a (non-global) transition on the designer canvas.
         * @constructs
         * @extends JIRA.WorkflowDesigner.Draw2DView
         * @mixes JIRA.WorkflowDesigner.SelectableFigureMixin
         *
         * @param {object} options
         * @param {draw2d.Canvas} options.canvas The canvas to render to.
         * @param {JIRA.WorkflowDesigner.CanvasModel} options.canvasModel The application's `CanvasModel`.
         * @param {boolean} options.immutable Whether the workflow is immutable.
         * @param {JIRA.WorkflowDesigner.TransitionModel} options.model The transition model.
         * @param {JIRA.WorkflowDesigner.StatusView} options.sourceView The view of the transition's source status.
         * @param {JIRA.WorkflowDesigner.StatusView} options.targetView The view of the transition's target status.
         * @param {JIRA.WorkflowDesigner.WorkflowModel} options.workflowModel The application's `WorkflowModel`.
         */
        initialize: function (options) {
            Draw2DView.prototype.initialize.apply(this, arguments);

            this._canvasModel = options.canvasModel;
            this._immutable = options.immutable;
            this._views = new Collection();
            this._views.reset([options.sourceView, options.targetView]);
            this._workflowModel = options.workflowModel;

            this.listenTo(this._workflowModel, "validation:transitions", this._updateValidation);

            this.listenTo(this._canvasModel, "change:showTransitionLabels", this._setTextVisibility);
            this.listenTo(this.model, "change:name", this.render);

            this.listenTo(this._views, {
                drag: this._onStatusDrag,
                dragEnd: this._onStatusDragEnd,
                highlight: this._onMouseEnter,
                unhighlight: this._onMouseLeave
            });
        },

        _updateValidation: function(validations) {
            var validation = validations[this.model.get("id")];
            this._connection.setValidation(validation);
            if (!this._isSelected()) {
                this._connection.unhighlight(); // this will redraw the line with proper color
            }
        },

        /**
         * Make the transition appear selected.
         */
        appearSelected: function () {
            this._canvas.getLayer("selected-transition").addFigure(this._connection);
            this._connection.appearSelected();
            this._connection.positionArrow();
            this.render();
        },

        /**
         * Create the view's Draw2D connection and add it to the canvas.
         *
         * @private
         */
        _createConnection: function () {
            var connection;

            connection = new TransitionConnection(this.isInitialTransition());
            connection.onDeselect = _.bind(this._onDeselect, this);
            connection.onMouseEnter = _.bind(this._onMouseEnter, this);
            connection.onMouseLeave = _.bind(this._onMouseLeave, this);
            connection.onReconnect = _.bind(this.trigger, this, "reconnect", this);
            connection.onSelect = _.bind(this._onSelect, this);
            connection.setSelectable(!this._immutable);

            if (!this._immutable) {
                connection.installEditPolicy(new LineSelectionFeedbackPolicy());
                connection.onDoubleClick = _.bind(this.edit, this);
            }

            this._canvas.getLayer("transitions").addFigure(connection);
            this._connection = this._figure = connection;
            this._setConnectionPorts();
            this._setTextVisibility();
        },

        /**
         * Destroy the transition on the server after confirming the operation with the user.
         *
         * The `TransitionModel` is also destroyed, which causes the canvas to update.
         */
        destroy: function () {
            new DeleteTransitionDialogView({
                transitionModel: this.model,
                workflowModel: this._workflowModel
            }).show();
        },

        /**
         * Show a dialog for editing the transition.
         */
        edit: function () {
            new EditTransitionDialogView({
                transitionModel: this.model,
                workflowModel: this._workflowModel
            }).show();
        },

        /**
         * @return {draw2d.Connection} The transition's Draw2D connection.
         */
        getConnection: function () {
            return this._connection;
        },

        /**
         * @private
         * @return {draw2d.Figure} The view's primary figure.
         * @see {@link JIRA.WorkflowDesigner.SelectableFigureMixin}
         */
        _getFigure: function () {
            return this._connection;
        },

        /**
         * @return {number} The transition's source angle in degrees, calculating it if necessary.
         * @private
         */
        _getSourceAngle: function () {
            var isToSelf = this._getSourceView() === this._getTargetView(),
                sourceAngle = this.model.get("sourceAngle");

            if (_.isNumber(sourceAngle)) {
                return sourceAngle;
            } else if (isToSelf) {
                return -90;
            } else {
                return this._getSourceView().getAngleToStatus(this._getTargetView());
            }
        },

        /**
         * @return {JIRA.WorkflowDesigner.BaseStatusView} The transition's source status view.
         * @private
         */
        _getSourceView: function () {
            return this._views.at(0);
        },

        /**
         * @return {number} The transition's target angle in degrees, calculating it if necessary.
         * @private
         */
        _getTargetAngle: function () {
            var isToSelf = this._getSourceView() === this._getTargetView(),
                targetAngle = this.model.get("targetAngle");

            if (_.isNumber(targetAngle)) {
                return targetAngle;
            } else if (isToSelf) {
                return 0;
            } else {
                return this._getTargetView().getAngleToStatus(this._getSourceView());
            }
        },

        /**
         * @return {JIRA.WorkflowDesigner.BaseStatusView} The transition's target status view.
         * @private
         */
        _getTargetView: function () {
            return this._views.at(1);
        },

        /**
         * Make the transition appear highlighted.
         *
         * @private
         */
        _highlight: function () {
            this._canvas.getLayer("highlighted-transition").addFigure(this._connection);
            this._connection.highlight();
            this._highlighted = true;
            this.render();
        },

        /**
         * @param {JIRA.WorkflowDesigner.BaseStatusView[]} statusViews Statuses to check
         * @returns {boolean} Whether this transition is connected to any of the given statuses.
         */
        isConnectedToAnyStatus: function(statusViews) {
            return _.contains(statusViews, this._getSourceView()) || _.contains(statusViews, this._getTargetView());
        },

        /**
         * Checks if the transition appears highlighted.
         *
         * @return {boolean} Whether the transition appears highlighted.
         */
        isHighlighted: function () {
            return this._highlighted;
        },

        /**
         * Checks if this is the initial transition.
         *
         * @return {boolean} Whether this is the initial transition.
         */
        isInitialTransition: function () {
            return this.model.get("source").get("initial");
        },

        /**
         * @private
         * @return {boolean} Whether the transition should be considered selected.
         */
        _isSelected: function () {
            return this.requestResponse.request("isSelected", this);
        },

        /**
         * Make the transition appear deselected and trigger a "deselected" event.
         *
         * @private
         * @see {@link JIRA.WorkflowDesigner.SelectableFigureMixin}
         */
        _onDeselect: function () {
            this.trigger("deselected", this);
            this.unhighlight();
        },

        /**
         * Highlight the transition if it's not selected.
         *
         * @private
         */
        _onMouseEnter: function () {
            this._isSelected() || this._highlight();
        },

        /**
         * Unhighlight the transition if it's not selected.
         *
         * @private
         */
        _onMouseLeave: function () {
            var statusIsSelected = _.some(this._views.invoke("isSelected"));
            statusIsSelected || this._isSelected() || this.unhighlight();
        },

        /**
         * Make the transition appear selected and trigger a "select" event.
         *
         * @private
         * @see {@link JIRA.WorkflowDesigner.SelectableFigureMixin}
         */
        _onSelect: function () {
            this.trigger("selected", this);
            this.appearSelected();
        },

        /**
         * Called when the user stopped dragging this transition's status.
         *
         * @private
         */
        _onStatusDragEnd: function () {
            this._connection.setTextVisible(true);
        },

        /**
         * Called when the user started dragging this transition's status.
         *
         * @private
         */
        _onStatusDrag: function () {
            this._connection.setTextVisible(false);
        },

        /**
         * Remove the view from the canvas.
         */
        remove: function () {
            Draw2DView.prototype.remove.apply(this, arguments);

            this._views.off({
                drag: this._onStatusDrag,
                dragEnd: this._onStatusDragEnd,
                highlight: this._onMousEnter,
                unhighlight: this._onMouseLeave
            });
        },

        /**
         * Render the transition
         *
         * @return {JIRA.WorkflowDesigner.TransitionView} `this`
         */
        render: function () {
            this._connection || this._createConnection();
            this._connection.setText(this.model.get("name"));
            this._setTextVisibility();
            return this;
        },

        /**
         * Reset the view's connection to match its model.
         */
        resetConnection: function() {
            this._canvas.removeFigure(this._connection);
            this._connection = null;
            this.render();
        },

        /**
         * Set the Draw2D connection's source and target ports and update the model accordingly.
         *
         * @private
         */
        _setConnectionPorts: function () {
            var sourcePort = this._getSourceView().getPortForAngle(this._getSourceAngle()),
                targetPort = this._getTargetView().getPortForAngle(this._getTargetAngle());

            this._connection.setSource(sourcePort);
            this._connection.setTarget(targetPort);
            this.model.set({
                sourceAngle: this._getSourceView().getAngleToPort(sourcePort),
                targetAngle: this._getTargetView().getAngleToPort(targetPort)
            });
        },

        /**
         * Shows the connection's text if necessary.
         *
         * @private
         */
        _setTextVisibility: function () {
            var hasName = !!this.model.get("name"),
                showLabels = this._canvasModel.get("showTransitionLabels"),
                visible = hasName && (showLabels || this.isHighlighted() || this._isSelected());

            this._connection.setTextVisible(visible);
        },

        /**
         * Set the transition's source and target views.
         *
         * @param {JIRA.WorkflowDesigner.StatusView} sourceView The transition's new source view.
         * @param {JIRA.WorkflowDesigner.StatusView} targetView The transition's new target view.
         */
        setViews: function (sourceView, targetView) {
            this._views.reset([sourceView, targetView]);
        },

        /**
         * Make the transition appear unhighlighted.
         */
        unhighlight: function () {
            this._canvas.getLayer("transitions").addFigure(this._connection);
            this._connection.unhighlight();
            this._highlighted = false;
            this.render();
        }
    });

    Cocktail.mixin(
        TransitionView,
        RequestResponseMixin,
        SelectableFigureMixin
    );

    return TransitionView;
});

AJS.namespace("JIRA.WorkflowDesigner.TransitionView", null, require("workflow-designer/transition-view"));