AJS.test.require([
    "com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer",
    "com.atlassian.jira.plugins.jira-workflow-designer:test-resources"
], function () {

    var DeleteTransitionDialogView = require("workflow-designer/dialogs/delete-transition-dialog-view");
    var StatusModel = require("workflow-designer/status-model");
    var StatusView = require("workflow-designer/status-view");
    var WorkflowModel = require("workflow-designer/workflow-model");
    var TransitionModel = require("workflow-designer/transition-model");
    var TransitionView = require("workflow-designer/transition-view");
    var CanvasModel = require("workflow-designer/canvas-model");
    var PanningSingleSelectionPolicy = require("workflow-designer/policy/canvas/panning-single-selection-policy");
    var TestUtilities = require("workflow-designer/test-utilities");
    var _ = require("workflow-designer/underscore");
    var draw2d = require("workflow-designer/draw-2d");



    module("TransitionView", {
        createStatusView: function (modelAttributes, viewOptions) {
            return new StatusView(_.defaults({}, viewOptions, {
                canvas: this.canvas,
                model: new StatusModel(modelAttributes),
                workflowModel: new WorkflowModel()
            })).render();
        },

        createTestData: function (options) {
            this.sourceView = new StatusView({
                canvas: this.canvas,
                model: new StatusModel({x: 10, y: 0}),
                workflowModel: new WorkflowModel()
            }).render();

            this.targetView = new StatusView({
                canvas: this.canvas,
                model: new StatusModel({x: 200, y: 0}),
                workflowModel: new WorkflowModel()
            }).render();

            this.transitionModel.set({
                source: this.sourceView.model,
                target: this.targetView.model
            });

            options = _.defaults({}, options, {
                canvas: this.canvas,
                canvasModel: this.canvasModel,
                model: this.transitionModel,
                sourceView: this.sourceView,
                targetView: this.targetView,
                workflowModel: this.workflowModel
            });

            this.transitionView = new TransitionView(options);
            this.transitionView.requestResponse.setHandler("isSelected", function () {
                return false;
            });

            this.transitionView.render();
        },

        createTransitionView: function (modelAttributes, viewOptions) {
            return new TransitionView(_.defaults({}, viewOptions, {
                canvas: this.canvas,
                canvasModel: new CanvasModel({}, {
                    workflowModel: this.workflowModel
                }),
                model: new TransitionModel(modelAttributes),
                workflowModel: this.workflowModel
            })).render();
        },

        deselectTransition: function () {
            this.canvas.selectFigure(null);
            this.transitionView.requestResponse.setHandler("isSelected", function () {
                return false;
            });
        },

        selectTransition: function () {
            this.canvas.selectFigure(this.transitionView._connection);
            this.transitionView.requestResponse.setHandler("isSelected", function () {
                return true;
            });
        },

        setup: function () {
            this.sandbox = sinon.sandbox.create();

            this.sandbox.stub(DeleteTransitionDialogView.prototype, "show");

            function getCanvasBoundingBox() {
                return new draw2d.geo.Rectangle(0, 0, 0, 0);
            }

            this.canvas = TestUtilities.testDraw2DCanvas();
            this.transitionModel = new TransitionModel({name: "Transition"});
            this.workflowModel = new WorkflowModel();
            this.canvasModel = new CanvasModel({}, {
                workflowModel: this.workflowModel
            });
            this.canvasModel.set("showTransitionLabels", false);

            this.canvas.installEditPolicy(new PanningSingleSelectionPolicy(getCanvasBoundingBox));
        },

        teardown: function () {
            this.sandbox.restore();
        },

        transitionIsHighlighted: function () {
            return this.transitionView.isHighlighted() && this.transitionView._connection.textIsVisible();
        },

        /**
         * @param {JIRA.WorkflowDesigner.TransitionView} transitionView A transition view.
         * @returns {boolean} Whether `transitionView` appears to be selected.
         */
        transitionIsSelected: function (transitionView) {
            return transitionView._connection.getColor().hash() === "#1A8CFF";
        }
    });

    test("appearSelected()", function () {
        this.createTestData();
        this.sandbox.spy(this.transitionView._connection, "appearSelected");

        this.transitionView.appearSelected();
        equal(this.transitionView._connection.appearSelected.callCount, 1,
            "The transition's connection appears selected");
    });

    test("destroy() displays a DeleteTransitionDialogView", function () {
        var dialogArguments,
            dialogStub = this.sandbox.stub(DeleteTransitionDialogView.prototype, "initialize");

        this.createTestData();
        this.transitionView.destroy();
        equal(dialogStub.callCount, 1, "A DeleteTransitionDialogView was created");

        dialogArguments = dialogStub.args[0];
        equal(dialogArguments.length, 1, "It was passed a single argument");
        ok(dialogArguments[0].transitionModel === this.transitionModel, "It was passed the correct TransitionModel");
        ok(dialogArguments[0].workflowModel === this.workflowModel, "It was passed the correct WorkflowModel");
    });

    test("Reconnection triggers a reconnect event", function () {
        var spy = sinon.spy();

        this.createTestData();
        this.transitionView.bind("reconnect", spy);
        this.transitionView._connection.onReconnect();

        ok(spy.called, "A reconnect event was triggered");
    });

    test("The label is not visible until the transition has a valid name", function () {
        this.createTestData();
        this.transitionModel.unset("name");
        sinon.stub(this.transitionView, "isSelected");

        ok(!this.transitionView._connection.textIsVisible(), "The transition's label isn't visible if it doesn't have a valid name");

        this.transitionView.isSelected.returns(true);
        this.transitionView.render();
        ok(!this.transitionView._connection.textIsVisible(), "The transition's label is not visible for selected transitions wihtout a valid name");
        this.transitionView.isSelected.returns(false);

        this.transitionView._highlight();
        ok(!this.transitionView._connection.textIsVisible(), "The transition's label is not visible for highlighted transitions wihtout a valid name");
        this.transitionView.unhighlight();

        this.transitionModel.set("name", "Valid Name");

        this.selectTransition();
        this.transitionView.render();
        ok(this.transitionView._connection.textIsVisible(), "The transition's label is visible for selected transitions with a valid name");
        this.deselectTransition();

        this.transitionView._highlight();
        ok(this.transitionView._connection.textIsVisible(), "The transition's label is visible for highlighted transitions with a valid name");
        this.transitionView.unhighlight();
    });

    test("The label is hidden while connected statuses are being dragged", function () {
        this.createTestData();

        this.sourceView.trigger("highlight");
        ok(this.transitionIsHighlighted(), "The transition is highlighted when its source status is hovered");

        this.sourceView.trigger("drag");
        ok(!this.transitionView._connection.textIsVisible(), "The transition's label is not visible when its source status is being dragged");

        this.sourceView.trigger("dragEnd");
        ok(this.transitionView._connection.textIsVisible(), "The transition's label is visible after the source status has finished being dragged");

        this.targetView.trigger("highlight");
        ok(this.transitionIsHighlighted(), "The transition is highlighted when its target status is hovered");

        this.targetView.trigger("drag");
        ok(!this.transitionView._connection.textIsVisible(), "The transition's label is not visible when its target status is being dragged");

        this.targetView.trigger("dragEnd");
        ok(this.transitionView._connection.textIsVisible(), "The transition's label is visible after the target status has finished being dragged");
    });

    test("The label is updated after changing the transition's name", function () {
        this.createTestData();

        ok(this.transitionView._connection._label.getText().indexOf("Transition") > -1, "The label's text is correct");

        this.transitionModel.set("name", "Updated Transition");
        ok(this.transitionView._connection._label.getText().indexOf("Updated Transition") > -1,
            "The label's text is correct after changing the transition's name");
    });

    test("The transition is highlighted on hover", function () {
        this.createTestData();
        ok(!this.transitionIsHighlighted(), "The transition is not highlighted by default");

        this.transitionView._connection.onMouseEnter();
        ok(this.transitionView.isHighlighted(), "The transition is highlighted on hover");

        this.transitionView._connection.onMouseLeave();
        ok(!this.transitionIsHighlighted(), "The transition is not highlighted after the mouse leaves");
    });

    test("The transition is highlighted when its status is hovered", function () {
        this.createTestData();
        ok(!this.transitionIsHighlighted(), "The transition is not highlighted by default");

        this.sourceView.trigger("highlight");
        ok(this.transitionIsHighlighted(), "The transition is highlighted when its source status is hovered");

        this.sourceView.trigger("unhighlight");
        ok(!this.transitionIsHighlighted(), "The transition is not highlighted when the mouse leaves its source status");

        this.targetView.trigger("highlight");
        ok(this.transitionIsHighlighted(), "The transition is highlighted when its target status is hovered");

        this.targetView.trigger("unhighlight");
        ok(!this.transitionIsHighlighted(), "The transition is not highlighted when the mouse leaves its target status");
    });

    test("The transition is highlighted when either of its statuses is selected", function () {
        this.createTestData();

        this.canvas.selectFigure(this.sourceView._getFigure());
        ok(this.transitionIsHighlighted(), "The transition is highlighted");

        this.transitionView._connection.onMouseLeave();
        ok(this.transitionIsHighlighted(), "The transition is still highlighted after the mouse leaves");
    });

    test("Selected transitions remain selected when the mouse leaves", function () {
        this.createTestData();

        ok(!this.transitionIsSelected(this.transitionView), "The transition does not appear selected");

        this.selectTransition();
        ok(this.transitionIsSelected(this.transitionView), "The transition appears selected after selection");

        this.transitionView._connection.onMouseLeave();
        ok(this.transitionIsSelected(this.transitionView), "The transition appears selected after mouse leaves");

        this.deselectTransition();
        ok(!this.transitionIsSelected(this.transitionView), "The transition does not appear selected after deselection");
    });

    test("The transition's sourceAngle and targetAngle attributes are automatically calculated", function () {
        this.createTestData();

        equal(this.transitionModel.get("sourceAngle"), this.sourceView.getAngleToPort(this.sourceView.getPortForAngle(0)),
            "The transition's sourceAngle attribute was set");
        equal(this.transitionModel.get("targetAngle"), this.targetView.getAngleToPort(this.targetView.getPortForAngle(180)),
            "The transition's targetAngle attribute was set");
    });

    test("The label is shown depending on the value of user preference 'showTransitionLabels'", function () {
        this.createTestData();

        this.canvasModel.set("showTransitionLabels", true);
        ok(this.transitionView._connection.textIsVisible(), "Label is visible");

        this.canvasModel.set("showTransitionLabels", false);
        ok(!this.transitionView._connection.textIsVisible(), "Label is not visible");
    });

    test("Double clicking a transition initiates editing", function () {
        var editStub = this.sandbox.stub(TransitionView.prototype, "edit");

        this.createTestData();
        this.transitionView._connection.onDoubleClick();
        equal(editStub.callCount, 1, "TransitionView#edit() was called after double clicking the connection");

        this.transitionView._connection._label.onDoubleClick();
        equal(editStub.callCount, 2, "TransitionView#edit() was called after double clicking the connection's label");
    });

    test("Double clicking a transition doesn't initiate editing when immutable", function () {
        var editStub = this.sandbox.stub(TransitionView.prototype, "edit"),
            transitionView;

        this.createTestData();
        transitionView = this.createTransitionView({
            source: this.sourceView.model,
            target: this.targetView.model
        }, {
            immutable: true,
            sourceView: this.sourceView,
            targetView: this.targetView
        });

        transitionView._connection.onDoubleClick();
        equal(editStub.callCount, 0, "TransitionView#edit() wasn't called");
    });

    test("The connection is added to the \"selected-transition\" layer on selection", function () {
        var selectedTransitionLayer = this.canvas.getLayer("selected-transition"),
            transitionsLayer = this.canvas.getLayer("transitions");

        this.createTestData();
        sinon.spy(selectedTransitionLayer, "addFigure");
        sinon.spy(transitionsLayer, "addFigure");

        this.transitionView.select();
        equal(selectedTransitionLayer.addFigure.callCount, 1, "The \"selected-transition\" layer's #addFigure() method was called");
        ok(selectedTransitionLayer.addFigure.args[0][0] === this.transitionView._connection, "It was passed the correct figure");

        this.transitionView.deselect();
        equal(transitionsLayer.addFigure.callCount, 1, "The \"transitions\" layer's #addFigure() method was called");
        ok(transitionsLayer.addFigure.args[0][0] === this.transitionView._connection, "It was passed the correct figure");
    });

    test("The connection is added to the \"transitions\" layer", function () {
        var layer = this.canvas.getLayer("transitions");

        sinon.spy(layer, "addFigure");
        this.createTestData();

        equal(layer.addFigure.callCount, 1, "The \"transitions\" layer's #addFigure() method was called");
        ok(layer.addFigure.args[0][0] === this.transitionView._connection, "It was passed the correct figure");
    });

    test("The connection's positionArrow method is called on selection", function () {
        this.createTestData();
        sinon.spy(this.transitionView._connection, "positionArrow");

        this.transitionView.select();
        equal(this.transitionView._connection.positionArrow.callCount, 1,
            "The connection's positionArrow() method was called");
    });

    test("resetConnection sets the connection based on the current model", function () {
        var originalSource, originalTarget;

        this.createTestData();
        originalSource = this.transitionView.getConnection().getSource();
        originalTarget = this.transitionView.getConnection().getTarget();

        this.transitionView.getConnection().setSource(null);
        this.transitionView.getConnection().setTarget(null);
        this.transitionView.resetConnection();

        ok(_.isEqual(this.transitionView.getConnection().getSource(), originalSource), "Original connection source was restored");
        ok(_.isEqual(this.transitionView.getConnection().getTarget(), originalTarget), "Original connection target was restored");
    });

    test("unhighlight()", function () {
        this.createTestData();
        this.sandbox.spy(this.transitionView._connection, "unhighlight");

        this.transitionView.unhighlight();
        equal(this.transitionView._connection.unhighlight.callCount, 1,
            "The transition's connection was unhighlighted");
    });

});
