define("workflow-designer/looped-transition-view", [
    "workflow-designer/dialogs/edit-looped-transition-dialog-view",
    "workflow-designer/dialogs/delete-transition-dialog-view",
    "workflow-designer/looped-transition-figure",
    "workflow-designer/inline-dialog-svg-positioner",
    "workflow-designer/draw-2d-view",
    "aui/inline-dialog",
    "workflow-designer/templates",
    "workflow-designer/underscore"
], function(
    EditLoopedTransitionDialogView,
    DeleteTransitionDialogView,
    LoopedTransitionFigure,
    InlineDialogSVGPositioner,
    Draw2DView,
    InlineDialog,
    Templates,
    _
) {
    return Draw2DView.extend(
    /** @lends JIRA.WorkflowDesigner.LoopedTransitionView# */
    {
        /**
         * Initialise the looped transition view.
         *
         * @classdesc The representation of a looped transition on the designer canvas.
         * @constructs
         * @extends JIRA.WorkflowDesigner.Draw2DView
         * @param {object} options
         * @param {draw2d.Canvas} options.canvas The canvas to render to.
         * @param {boolean} [options.immutable=false] Whether the workflow is immutable.
         * @param {JIRA.WorkflowDesigner.WorkflowModel} options.workflowModel The application's workflow model.
         */
        initialize: function (options) {
            Draw2DView.prototype.initialize.apply(this, arguments);

            options = _.defaults({}, options, {
                immutable: false
            });

            this._immutable = options.immutable;
            this._workflowModel = options.workflowModel;

            this.listenTo(this.model, "change:name", this.render);
            this.listenTo(this._workflowModel, "validation:transitions", this._updateValidation);
        },

        _updateValidation: function(validationResults) {
            var validation = validationResults[this.model.get("id")];
            this._figure.setValidation(validation);
        },

        /**
         * Create the view's information dialog.
         *
         * @private
         */
        _createDialog: function () {
            var hoverElement = this._figure.getHoverElement(),
                id = _.uniqueId("looped-transition-dialog-"),
                instance = this;

            function getArrowAttributes() {
                return {
                    fill : "#333333",
                    stroke : "#333333"
                };
            }

            function renderDialog(element, trigger, show) {
                var template = Templates.loopedTransitionDialogContent;

                element.html(template({
                    name: instance.model.get("name")
                }));

                show();
            }

            this._dialog = InlineDialog(hoverElement, id, renderDialog, {
                calculatePositions: InlineDialogSVGPositioner.calculatePosition,
                getArrowAttributes: getArrowAttributes,
                hideDelay: 50,
                onHover: true
            });
        },

        /**
         * Create the view's Draw2D figure.
         *
         * @private
         */
        _createFigure: function () {
            var figure;

            figure = this._figure = new LoopedTransitionFigure(this.model);
            this._immutable || (figure.onDoubleClick = _.bind(this.edit, this));

            this._canvas.getLayer("looped-transitions").addFigure(figure);
            this._createDialog();
        },

        /**
         * Show a dialog that allows the user to delete the looped transition.
         */
        destroy: function () {
            new DeleteTransitionDialogView({
                transitionModel: this.model,
                workflowModel: this._workflowModel
            }).show();
        },

        /**
         * Show a dialog for editing the looped transition.
         */
        edit: function () {
            new EditLoopedTransitionDialogView({
                transitionModel: this.model,
                workflowModel: this._workflowModel
            }).show();
        },

        /**
         * Reposition the view in response to its target status moving.
         *
         * @private
         * @return {draw2d.Figure} The view's primary figure.
         * @see {@link JIRA.WorkflowDesigner.SelectableFigureMixin}
         */
        getFigure: function () {
            return this._figure;
        },

        /**
         * Make the transition appear deselected and trigger a "deselected" event.
         *
         * @private
         * @see {@link JIRA.WorkflowDesigner.SelectableFigureMixin}
         */
        deselect: function () {
            if (this._immutable) {
                return;
            }
            this._figure.unselect();
            this._canvas.getLayer("looped-transitions").addFigure(this._figure);
            this.trigger("deselected", this);
        },

        /**
         * Make the transition appear selected and trigger a "selected" event.
         *
         * @private
         * @see {@link JIRA.WorkflowDesigner.SelectableFigureMixin}
         */
        select: function () {
            if (this._immutable) {
                return;
            }
            this._figure.select();
            this._canvas.getLayer("selected-transition").addFigure(this._figure);
            this.trigger("selected", this);
        },

        /**
         * Render the view.
         *
         * @return {JIRA.WorkflowDesigner.LoopedTransitionView} `this`
         */
        render: function () {
            this._figure || this._createFigure();
            this._figure.loadText();
            return this;
        },

        reset: function() {
            this._canvas.removeFigure(this._figure);
            this._figure = null;
            this.render();
        }
    });

});

AJS.namespace("JIRA.WorkflowDesigner.LoopedTransitionView", null, require("workflow-designer/looped-transition-view"));