AJS.test.require([
    "com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer",
    "com.atlassian.jira.plugins.jira-workflow-designer:test-resources"
], function () {

    var DeleteStatusDialogView = require("workflow-designer/dialogs/delete-status-dialog-view");
    var StatusLozengeColours = require("workflow-designer/status-lozenge-colours");
    var StatusModel = require("workflow-designer/status-model");
    var StatusView = require("workflow-designer/status-view");
    var WorkflowModel = require("workflow-designer/workflow-model");
    var TestUtilities = require("workflow-designer/test-utilities");
    var _ = require("workflow-designer/underscore");
    var draw2d = require("workflow-designer/draw-2d");

    module("StatusView", {
        /**
         * @param {JIRA.WorkflowDesigner.StatusView} statusView A status view.
         * @returns {boolean} Whether all of `statusView`'s ports are visible.
         */
        allPortsAreVisible: function (statusView) {
            return _.all(statusView.getPorts(), function (port) {
                return port.isVisible();
            });
        },

        /**
         * @param {object} [options] Options to pass to the view.
         * @returns {JIRA.WorkflowDesigner.StatusView} A new status view.
         */
        createStatusView: function (options) {
            var getAllTargetPorts = _.bind(this.canvasView._getAllTargetPorts, this.canvasView),
                isPortDragged = _.bind(this.canvasView._isPortDragged, this.canvasView),
                statusView;

            options = _.defaults({}, options, {
                canvas: this.canvas,
                model: new StatusModel(),
                getAllTargetPorts: getAllTargetPorts,
                isPortDragged: isPortDragged,
                workflowModel: this.workflowModel
            });

            statusView = new StatusView(options).render();
            this.canvasView.statusViews.add(statusView);
            return statusView;
        },

        /**
         * @param {JIRA.WorkflowDesigner.StatusView} statusView A status view.
         * @returns {boolean} Whether none of `statusView`'s ports are visible.
         */
        noPortsAreVisible: function (statusView) {
            return _.all(statusView.getPorts(), function (port) {
                return !port.isVisible();
            });
        },

        setup: function () {
            this.canvasView = TestUtilities.testCanvasView();
            this.canvas = this.canvasView.canvas;
            this.sandbox = sinon.sandbox.create();

            this.workflowModel = new WorkflowModel();
        },

        teardown: function () {
            TestUtilities.removeDialogs();
            this.sandbox.restore();
        }
    });

    test("All ports are visible while dragging", function () {
        var otherStatusView, port, statusView;

        otherStatusView = this.createStatusView();
        ok(this.noPortsAreVisible(otherStatusView), "Ports aren't visible initially");

        statusView = this.createStatusView();
        ok(this.noPortsAreVisible(statusView), "Ports aren't visible initially");

        port = statusView.getPorts()[0];
        port.onDragStart();

        ok(this.allPortsAreVisible(otherStatusView), "All ports are visible on the target status");
        ok(_.all(_.tail(statusView.getPorts()), function (port) {
            return port.isVisible();
        }), "All ports are visible on the source status except the one being dragged");

        port.onDragEnd();
        ok(this.noPortsAreVisible(otherStatusView), "No ports are visible on the target status");
        ok(this.allPortsAreVisible(statusView), "All ports are visible on the source status");
    });

    test("destroy() shows a DeleteStatusDialogView", function () {
        var dialogArguments,
            initializeShowStub = this.sandbox.stub(DeleteStatusDialogView.prototype, "initialize").returns(jQuery.noop),
            statusView = this.createStatusView();
        this.sandbox.stub(DeleteStatusDialogView.prototype, "show").returns(jQuery.noop),

        statusView.destroy();
        equal(initializeShowStub.callCount, 1, "A DeleteStatusDialogView was created");

        dialogArguments = initializeShowStub.args[0];
        equal(dialogArguments.length, 1, "It was passed a single argument");
        equal(dialogArguments[0].statusModel, statusView.model, "It was passed the correct StatusModel");
        equal(dialogArguments[0].workflowModel, this.workflowModel, "It was passed the correct WorkflowModel");
    });

    test("Double clicking a status doesn't initiate editing when immutable", function () {
        var editStub = this.sandbox.stub(StatusView.prototype, "edit"),
            statusView = this.createStatusView({immutable: true});

        statusView._figure.onDoubleClick();
        equal(editStub.callCount, 0, "StatusView#edit() wasn't called");
    });

    test("Double clicking a status initiates editing", function () {
        this.workflowModel.get("permissions").set("editStatus", true);

        var editStub = this.sandbox.stub(StatusView.prototype, "edit"),
            statusView = this.createStatusView();

        statusView._figure.onDoubleClick();
        equal(editStub.callCount, 1, "StatusView#edit() was called.");
    });

    test("Drag events are triggered", function () {
        var statusViews;

        function assertDragEvents(statusView) {
            var dragEndSpy = sinon.spy(),
                dragSpy = sinon.spy();

            statusView.bind("drag", dragSpy);
            statusView.bind("dragEnd", dragEndSpy);

            statusView._figure.onDrag();
            equal(dragSpy.callCount, 1, "A drag event was triggered");

            statusView._figure.command = statusView._figure.createCommand(new draw2d.command.CommandType(draw2d.command.CommandType.MOVE));
            statusView._figure.onDragEnd();
            equal(dragEndSpy.callCount, 1, "A dragEnd event was triggered");
        }

        statusViews = [
            this.createStatusView(),
            this.createStatusView({immutable: true})
        ];

        _.each(statusViews, assertDragEvents);
    });

    test("Dragging a port selects the associated status", function () {
        var statusView = this.createStatusView();
        ok(!statusView.isSelected(), "The status isn't selected");

        statusView.getPorts()[0].onDragStart();
        ok(statusView.isSelected(), "The status is selected");
    });

    test("Dragging the view updates model coordinates", function () {
        var statusView;

        statusView = this.createStatusView();
        statusView.getBoundingBox = function () {
            return new draw2d.geo.Rectangle(100, 200, 10, 10);
        };

        statusView._figure.onDragStart();
        statusView._figure.onDragEnd();

        equal(statusView.model.get("x"), 100, "The model's x coordinate was updated");
        equal(statusView.model.get("y"), 200, "The model's y coordinate was updated");
    });

    test("Highlight events are triggered on hover", function () {
        var statusViews;

        function assertHighlightEvents(statusView) {
            var highlightSpy = sinon.spy(),
                unhighlightSpy = sinon.spy();

            statusView.bind("highlight", highlightSpy);
            statusView.bind("unhighlight", unhighlightSpy);

            statusView._figure.onMouseEnter();
            equal(highlightSpy.callCount, 1, "A highlight event was fired");

            statusView._figure.onMouseLeave();
            equal(unhighlightSpy.callCount, 1, "An unhighlight event was fired");
        }

        statusViews = [
            this.createStatusView(),
            this.createStatusView({immutable: true})
        ];

        _.each(statusViews, assertHighlightEvents);
    });

    test("Statuses are the correct colour when status lozenges are available", function () {
        var expectedArguments,
            statusView;

        this.sandbox.stub(StatusLozengeColours, "getBackgroundColour").returns("#ff0000");
        this.sandbox.spy(StatusLozengeColours, "getBorderColour");
        this.sandbox.stub(StatusLozengeColours, "getColour").returns("#00ff00");

        statusView = this.createStatusView({
            model: new StatusModel({
                statusCategory: {
                    colourName: "green",
                    id: 1
                }
            })
        });

        expectedArguments = [{
            colourName: "green",
            isSubtle: false
        }];

        // Everything except the border colour should have been retrieved from StatusLozengeColours.
        equal(StatusLozengeColours.getBackgroundColour.callCount, 1, "StatusLozengeColours.getBackgroundColour was called");
        deepEqual(StatusLozengeColours.getBackgroundColour.args[0], expectedArguments, "It was passed the correct arguments");
        equal(StatusLozengeColours.getBorderColour.callCount, 0, "StatusLozengeColours.getBorderColour wasn't called");
        equal(StatusLozengeColours.getColour.callCount, 1, "StatusLozengeColours.getColour was called");
        deepEqual(StatusLozengeColours.getColour.args[0], expectedArguments, "It was passed the correct arguments");

        equal(statusView._figure.getBackgroundColor().hash(), "#FF0000", "The status has the correct background colour");
        equal(statusView._figure.getColor().hash(), "#FFFFFF", "The status has the correct border colour");
        equal(statusView._figure.getTextColor().hash(), "#00FF00", "The status has the correct text colour");
    });

    test("Statuses are the correct colour when status lozenges aren't available", function () {
        var statusView = this.createStatusView();

        equal(statusView._figure.getBackgroundColor().hash(), "#4A6785", "The status has the correct background colour");
        equal(statusView._figure.getColor().hash(), "#FFFFFF", "The status has the correct border colour");
        equal(statusView._figure.getTextColor().hash(), "#FFFFFF", "The status has the correct text colour");
    });

    test("Ports are shown on hover", function () {
        var statusView = this.createStatusView();

        ok(this.noPortsAreVisible(statusView), "Ports aren't visible initially");

        statusView._figure.onMouseEnter();
        ok(this.allPortsAreVisible(statusView), "Ports become visible on hover");

        statusView._figure.onMouseLeave();
        ok(this.noPortsAreVisible(statusView), "Ports are hidden on mouse leave");
    });

    test("Ports are shown on port hover", function () {
        var port, statusView;

        statusView = this.createStatusView();
        port = statusView.getPorts()[0];

        ok(this.noPortsAreVisible(statusView), "Ports aren't visible initially");

        port.onMouseEnter();
        ok(this.allPortsAreVisible(statusView), "Ports become visible on port hover");

        port.onMouseLeave();
        ok(this.noPortsAreVisible(statusView), "Ports are hidden on mouse leave");
    });

    test("Ports aren't hidden after moving from the status body to a port", function () {
        var port, statusView;

        statusView = this.createStatusView();
        port = statusView.getPorts()[0];

        ok(this.noPortsAreVisible(statusView), "Ports aren't visible initially");

        statusView._figure.onMouseEnter();
        ok(this.allPortsAreVisible(statusView), "Ports become visible on hover");

        statusView._figure.onMouseLeave(port);
        ok(this.allPortsAreVisible(statusView),
            "Ports are still visible as the new figure is a port");

        port.onMouseLeave(statusView._figure);
        ok(this.allPortsAreVisible(statusView),
            "Ports are still visible as the new figure is the status");
    });

    test("Ports aren't shown on hover when immutable", function () {
        var statusView = this.createStatusView({immutable: true});

        ok(this.noPortsAreVisible(statusView), "Ports aren't visible initially");

        statusView._figure.onMouseEnter();
        ok(this.noPortsAreVisible(statusView), "Ports aren't visible on hover");
    });

    test("Statuses are the correct colour when status lozenges are available in subtle mode", function () {
        var currentStatusView,
            expectedArguments,
            otherStatusView;

        this.sandbox.stub(StatusLozengeColours, "getBackgroundColour").returns("#ff0000");
        this.sandbox.stub(StatusLozengeColours, "getBorderColour").returns("#00ff00");
        this.sandbox.stub(StatusLozengeColours, "getColour").returns("#0000ff");

        this.canvasView = TestUtilities.testCanvasView({
            workflowModel: new WorkflowModel({currentStepId: 1})
        });

        currentStatusView = this.canvasView.addStatus(new StatusModel({
            statusCategory: {
                colourName: "green",
                id: 1
            },
            stepId: 1
        }));

        expectedArguments = [{
            colourName: "green",
            isSubtle: false
        }];

        // Everything except the border colour should have been retrieved from StatusLozengeColours.
        equal(StatusLozengeColours.getBackgroundColour.callCount, 1, "StatusLozengeColours.getBackgroundColour was called");
        deepEqual(StatusLozengeColours.getBackgroundColour.args[0], expectedArguments, "It was passed the correct arguments");
        equal(StatusLozengeColours.getBorderColour.callCount, 0, "StatusLozengeColours.getBorderColour wasn't called");
        equal(StatusLozengeColours.getColour.callCount, 1, "StatusLozengeColours.getColour was called");
        deepEqual(StatusLozengeColours.getColour.args[0], expectedArguments, "It was passed the correct arguments");

        equal(currentStatusView._figure.getBackgroundColor().hash(), "#FF0000", "The current status has the correct background colour");
        equal(currentStatusView._figure.getColor().hash(), "#FFFFFF", "The current status has the correct border colour");
        equal(currentStatusView._figure.getTextColor().hash(), "#0000FF", "The current status has the correct text colour");

        otherStatusView = this.canvasView.addStatus(new StatusModel({
            statusCategory: {
                colourName: "yellow",
                id: 2
            },
            stepId: 2
        }));

        expectedArguments = [{
            colourName: "yellow",
            isSubtle: true
        }];

        // Everything should have been retrieved from StatusLozengeColours.
        equal(StatusLozengeColours.getBackgroundColour.callCount, 2, "StatusLozengeColours.getBackgroundColour was called");
        deepEqual(StatusLozengeColours.getBackgroundColour.args[1], expectedArguments, "It was passed the correct arguments");
        equal(StatusLozengeColours.getBorderColour.callCount, 1, "StatusLozengeColours.getBorderColour was called");
        deepEqual(StatusLozengeColours.getBorderColour.args[0], expectedArguments, "It was passed the correct arguments");
        equal(StatusLozengeColours.getColour.callCount, 2, "StatusLozengeColours.getColour was called");
        deepEqual(StatusLozengeColours.getColour.args[1], expectedArguments, "It was passed the correct arguments");

        equal(otherStatusView._figure.getBackgroundColor().hash(), "#FF0000", "The current status has the correct background colour");
        equal(otherStatusView._figure.getColor().hash(), "#00FF00", "The current status has the correct border colour");
        equal(otherStatusView._figure.getTextColor().hash(), "#0000FF", "The current status has the correct text colour");
    });

    test("Statuses are the correct colour when status lozenges aren't available in subtle mode", function () {
        var currentStatusView, otherStatusView;

        this.canvasView = TestUtilities.testCanvasView({
            workflowModel: new WorkflowModel({currentStepId: 1})
        });

        currentStatusView = this.canvasView.addStatus(new StatusModel({stepId: 1}));
        equal(currentStatusView._figure.getBackgroundColor().hash(), "#4A6785", "The current status has the correct background colour");
        equal(currentStatusView._figure.getColor().hash(), "#FFFFFF", "The current status has the correct border colour");
        equal(currentStatusView._figure.getTextColor().hash(), "#FFFFFF", "The current status has the correct text colour");

        otherStatusView = this.canvasView.addStatus(new StatusModel({stepId: 2}));
        equal(otherStatusView._figure.getBackgroundColor().hash(), "#FFFFFF", "The other status has the correct background colour");
        equal(otherStatusView._figure.getColor().hash(), "#4A6785", "The other status has the correct border colour");
        equal(otherStatusView._figure.getTextColor().hash(), "#4A6785", "The other status has the correct text colour");
    });

    test("The figure is added to the \"selected-status\" layer on selection", function () {
        var selectedStatusLayer = this.canvas.getLayer("selected-status"),
            statusesLayer = this.canvas.getLayer("statuses"),
            view = this.createStatusView();

        sinon.spy(selectedStatusLayer, "addFigure");
        sinon.spy(statusesLayer, "addFigure");

        view.select();
        equal(selectedStatusLayer.addFigure.callCount, 1, "The \"selected-status\" layer's #addFigure() method was called");
        ok(selectedStatusLayer.addFigure.args[0][0] === view._figure, "It was passed the correct figure");

        view.deselect();
        equal(statusesLayer.addFigure.callCount, 1, "The \"statuses\" layer's #addFigure() method was called");
        ok(statusesLayer.addFigure.args[0][0] === view._figure, "It was passed the correct figure");
    });

    test("The figure is added to the \"statuses\" layer", function () {
        var layer = this.canvas.getLayer("statuses"),
            view;

        sinon.spy(layer, "addFigure");
        view = this.createStatusView();

        equal(layer.addFigure.callCount, 1, "The \"statuses\" layer's #addFigure() method was called");
        ok(layer.addFigure.args[0][0] === view._figure, "It was passed the correct figure");
    });

});
