define("workflow-designer/base-status-view", [
    "workflow-designer/status-port",
    "workflow-designer/draw-2d-view",
    "workflow-designer/draw-2d",
    "workflow-designer/backbone",
    "workflow-designer/underscore"
], function(
    StatusPort,
    Draw2DView,
    draw2d,
    Backbone,
    _
) {
    return Draw2DView.extend(
    /** @lends JIRA.WorkflowDesigner.BaseStatusView# */
    {
        /**
         * Initialise the view.
         *
         * @classdesc
         * The representation of a status on the designer canvas.
         *
         * This view is abstract; do not create instances of it, use those that extend from it.
         * @constructs
         * @extends JIRA.WorkflowDesigner.Draw2DView
         * @param {object} options
         * @param {draw2d.Canvas} options.canvas The canvas to render to.
         * @param {function} [options.getAllTargetPorts] Returns all possible target ports.
         * @param {boolean} [options.immutable=false] Whether the workflow is immutable.
         * @param {function} [options.isPortDragged] Returns true if a port is currently being dragged.
         * @param {JIRA.WorkflowDesigner.WorkflowModel} options.workflowModel The application's workflow model.
         */
        initialize: function (options) {
            Draw2DView.prototype.initialize.apply(this, arguments);

            options = _.defaults({}, options, {
                immutable: false
            });

            this.immutable = options.immutable;
            this.isPortDragged = options.isPortDragged;
            this.getAllTargetPorts = options.getAllTargetPorts;
            this.workflowModel = options.workflowModel;

            this.listenTo(this.model, "change", this.render);
        },

        /**
         * Creates and configures the view's Draw2D figures.
         *
         * @private
         */
        _createAndConfigureFigure: function() {
            var figure = this._createFigure(),
                instance = this;

            figure.onDrag = _.wrap(figure.onDrag, function (f) {
                f.apply(this, _.toArray(arguments).slice(1));
                instance.trigger("drag");
            });

            figure.onDragEnd = _.wrap(figure.onDragEnd, function (f) {
                f.apply(this, _.toArray(arguments).slice(1));
                instance._updatePosition();
                instance.trigger("dragEnd");
            });

            if (this.immutable) {
                figure.setDeleteable(false);
                figure.setResizeable(false);
            } else {
                figure.installEditPolicy(new draw2d.policy.figure.DragDropEditPolicy());
            }

            return figure;
        },

        /**
         * Creates a port
         *
         * @param {Object} options
         * @returns {JIRA.WorkflowDesigner.StatusPort}
         * @private
         */
        _createPort: function(options) {
            options = _.extend({}, options, {
                canvas: this._canvas,
                isPortDragged: this.isPortDragged,
                getAllTargetPorts: this.getAllTargetPorts
            });

            return new StatusPort(options);
        },

        /**
         * Calculate the angle between the view and another figure.
         *
         * Angles are measured in degrees from the figures' centres; 0 is to
         * the right, -90 is up, 90 is down, etc.
         *
         * @param {draw2d.Figure} figure The other figure.
         * @return {number} The angle between the view and `figure`.
         * @private
         */
        _getAngleToFigure: function (figure) {
            var figureCentre = figure.getBoundingBox().getCenter(),
                thisCentre = this._figure.getBoundingBox().getCenter();

            return Math.atan2(
                figureCentre.getY() - thisCentre.getY(),
                figureCentre.getX() - thisCentre.getX()
            ) / Math.PI * 180;
        },

        /**
         * @param {draw2d.Port} port A port on the status.
         * @return {number} `port`'s angle from the centre of the status.
         */
        getAngleToPort: function (port) {
            return this._getAngleToFigure(port);
        },

        /**
         * Calculate the angle from the view to another `BaseStatusView`.
         *
         * @param {JIRA.WorkflowDesigner.BaseStatusView} statusView The other view.
         * @return {number} The angle between this view and `statusView`.
         */
        getAngleToStatus: function (statusView) {
            return this._getAngleToFigure(statusView._figure);
        },

        /**
         * Get the port closest to a given angle.
         *
         * @param {number} angle The target angle.
         * @return {draw2d.Port} The port closest to `angle`.
         */
        getPortForAngle: function (angle) {
            function getAngleDifference(port) {
                var difference = Math.abs(angle - this._getAngleToFigure(port));
                return difference < 180 ? difference : Math.abs(difference - 360);
            }

            return _.min(this.getPorts(), getAngleDifference, this);
        },

        /**
         * Position the view's figure to match its model.
         *
         * @protected
         */
        _positionFigure: function () {
            if (this.model.hasCoordinates()) {
                this._figure.setPosition(
                    this.model.get("x"),
                    this.model.get("y")
                );
            }
        },

        /**
         * Render the status.
         *
         * @abstract
         * @return {JIRA.WorkflowDesigner.BaseStatusView} `this`
         */
        render: function () {
            this._figure || (this._figure = this._createAndConfigureFigure());
            return this;
        },

        /**
         * Set the status's position.
         *
         * @param {number} x The status's x position.
         * @param {number} y The status's y position.
         */
        setPosition: function (x, y) {
            this.model.set({
                x: x,
                y: y
            });
        },

        /**
         * Set the model's position in response to the rectangle moving.
         *
         * @private
         */
        _updatePosition: function () {
            var boundingBox = this.getBoundingBox();
            this.setPosition(boundingBox.getX(), boundingBox.getY());
        }
    });
});

AJS.namespace("JIRA.WorkflowDesigner.BaseStatusView", null, require("workflow-designer/base-status-view"));