AJS.test.require([
    "com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer",
    "com.atlassian.jira.plugins.jira-workflow-designer:test-resources"
], function () {

    var TransitionModel = require("workflow-designer/transition-model");
    var StatusModel = require("workflow-designer/status-model");
    var WorkflowModel = require("workflow-designer/workflow-model");
    var PermissionsModel = require("workflow-designer/permissions-model");
    var TransitionPropertiesView = require("workflow-designer/properties-panel/transition-properties-view");
    var contextPath = require("wrm/context-path");
    var PanelService = require("workflow-designer/internal-api/panel-service");

    var _ = require("underscore");

    module("TransitionPropertiesView", {

        setup: function () {
            this.sandbox = sinon.sandbox.create();

            this.sandbox
                .stub(PanelService, "renderPanels")
                .withArgs("workflow.transition.properties")
                .returns(["<div class='custom-panel'>My custom panel</div>"]);
        },

        teardown: function () {
            this.sandbox.restore();
        },

        createView: function (options) {
            options = options || {};


            this.propertiesTransitionOption = {
                name: "properties",
                count: "2",
                url: "/properties/url",
                weight: 10
            };

            this.conditionsTransitionOption = {
                name: "conditions",
                count: "1",
                url: "/conditions/url&descriptorTab=conditions",
                weight: 20
            };

            this.selectedView = {
                destroy: sinon.spy(),
                model: new TransitionModel({
                    actionId: 100,
                    screenId: options.screenId,
                    screenName: options.screenName,
                    source: new StatusModel({
                        stepId: 1,
                        initial: !!options.isInitialTransition
                    }),
                    transitionOptions: [this.propertiesTransitionOption, this.conditionsTransitionOption],
                    validations: {
                        mostSevere: {
                            "message": "problem with 'show conditions' fix",
                            "severity": "MAJOR",
                            "quickFixHtml": '<button class="show-conditions-quick-fix aui-button aui-button-link">Show conditions</button>'
                        }
                    }
                })
            };

            this.workflowModel = new WorkflowModel({
                name: "Workflow name",
                draft: !!options.isDraftWorkflow,
                permissions: this.permissionsModel(true, true, options.administrator)
            });

            this.transitionPropertiesView = new TransitionPropertiesView({
                selectedView: this.selectedView,
                workflowModel: this.workflowModel
            }).render();

            this.transitionPropertiesView._openConditions = sinon.spy();
        },

        permissionsModel: function (editPropertyOptions, createStatus, administrator) {
            return new PermissionsModel({
                editPropertyOptions: editPropertyOptions,
                createStatus: createStatus,
                administrator: _.isUndefined(administrator) ? true : administrator
            });
        },

        editScreenUrl: function (screenId) {
            return contextPath() + "/secure/admin/ConfigureFieldScreen.jspa?id=" + screenId;
        },

        viewPropertiesUrl: function () {
            return contextPath() + "/properties/url" + this._stepQueryString();
        },

        viewConditionsUrl: function () {
            return contextPath() + "/conditions/url&descriptorTab=conditions" + this._stepQueryString();
        },

        _stepQueryString: function () {
            return this.selectedView.model.isInitial() ? "&" : "&workflowStep=1";
        }
    });

    test("Clicking the delete button calls the selected view's destroy() method", function () {
        this.createView();

        this.transitionPropertiesView.$(".delete").click();
        equal(this.selectedView.destroy.callCount, 1, "The selected view's destroy() method was called");
    });

    test("Contains link to draft workflow transition properties", function () {
        this.createView({
            isDraftWorkflow: true
        });

        equal(this.transitionPropertiesView.$el.find("a[href='" + this.viewPropertiesUrl() + "']").length, 1, "Properties link is present");
    });

    test("Contains link to draft workflow initial transition properties", function () {
        this.createView({
            isDraftWorkflow: true,
            isInitialTransition: true
        });

        equal(this.transitionPropertiesView.$el.find("a[href='" + this.viewPropertiesUrl() + "']").length, 1, "Properties link is present");
    });

    test("Contains link to live workflow transition properties", function () {
        this.createView({
            isDraftWorkflow: false
        });

        equal(this.transitionPropertiesView.$el.find("a[href='" + this.viewPropertiesUrl() + "']").length, 1, "Properties link is present");
    });

    test("Contains link to draft workflow transition conditions", function () {
        this.createView({
            isDraftWorkflow: true
        });

        equal(this.transitionPropertiesView.$el.find("a[href='" + this.viewConditionsUrl() + "']").length, 1, "Conditions link is present");
    });

    test("Contains link to live workflow transition conditions", function () {
        this.createView({
            isDraftWorkflow: false
        });

        equal(this.transitionPropertiesView.$el.find("a[href='" + this.viewConditionsUrl() + "']").length, 1, "Conditions link is present");
    });

    test("Contains read-only screen information", function () {
        this.createView({
            isDraftWorkflow: false,
            screenId: 100,
            screenName: "test screen name",
            administrator: false
        });

        equal(this.transitionPropertiesView.$el.find("a[href='" + this.editScreenUrl(100) + "']").length, 0, "Edit screen link is not present");
        equal(this.transitionPropertiesView.$el.html().indexOf("test screen name") > 0, true, "Edit screen info is present");
    });

    test("Contains link to edit screen information", function () {
        this.createView({
            isDraftWorkflow: false,
            screenId: 100,
            screenName: "test screen name"
        });

        equal(this.transitionPropertiesView.$el.find("a[href='" + this.editScreenUrl(100) + "']").length, 1, "Edit screen link is present");
    });

    test("Doesn't contain link to edit screen information", function () {
        this.createView({
            isDraftWorkflow: false
        });

        equal(this.transitionPropertiesView.$el.find("a[href='" + this.editScreenUrl() + "']").length, 0, "Edit screen link is not present");
    });

    test("Clicking on show conditions quick fix opens a condition winow", function () {
        this.createView({
            isDraftWorkflow: false
        });
        this.transitionPropertiesView.$(".show-conditions-quick-fix").click();
        equal(this.transitionPropertiesView._openConditions.callCount, 1, "Conditions view is opened when a button with .show-conditions-quick-fix class is clicked");
    });

    test("Panels registered via the panel service get rendered", function () {
        this.createView();
        ok(PanelService.renderPanels.calledWith("workflow.transition.properties", {
            workflowName: "Workflow name",
            actionId: 100
        }), "Panel service is called with correct context");
        equal(this.transitionPropertiesView.$(".custom-panel").text(), "My custom panel");
    });

});
