AJS.test.require([
    "com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer",
    "com.atlassian.jira.plugins.jira-workflow-designer:test-resources"
], function () {

    var jQuery = require("jquery");
    var WorkflowModel = require("workflow-designer/workflow-model");
    var PermissionsModel = require("workflow-designer/permissions-model");
    var TransitionModel = require("workflow-designer/transition-model");
    var Messages = require("workflow-designer/messages");
    var CanvasModel = require("workflow-designer/canvas-model");
    var StatusModel = require("workflow-designer/status-model");
    var StatusView = require("workflow-designer/status-view");
    var WorkflowGlobalTransitionsAJAXManager = require("workflow-designer/io/ajax/workflow-global-transitions-ajax-manager");
    var StatusPropertiesView = require("workflow-designer/properties-panel/status-properties-view");
    var TestUtilities = require("workflow-designer/test-utilities");
    var contextPath = require("wrm/context-path");
    var PanelService = require("workflow-designer/internal-api/panel-service");

    module("StatusPropertiesView", {
        /**
         * Check the global transition checkbox in a `StatusPropertiesView`.
         *
         * @param {StatusPropertiesView} view A `StatusPropertiesView`.
         */
        checkGlobalTransitionCheckbox: function (view) {
            view.ui.createGlobalTransition.attr("checked", "");
            view.ui.createGlobalTransition.change();
        },

        /**
         * @param {object} [options] Options to pass to the view's constructor.
         * @returns {StatusPropertiesView} A `StatusPropertiesView`.
         */
        createView: function (options) {
            var statusModel, statusView, workflowModel;

            statusModel = new StatusModel({
                id: "status-id",
                name: "Status Name",
                statusId: "status-id",
                stepId: 1,
                validations: {
                    mostSevere: {
                        "message": "problem with 'add transition' fix",
                        "severity": "MAJOR",
                        "quickFixHtml": '<button class="add-transition-quick-fix aui-button aui-button-link">Add transition</button>'
                    }
                }
            });

            statusView = new StatusView({
                canvas: TestUtilities.testDraw2DCanvas(),
                model: statusModel
            });

            workflowModel = new WorkflowModel({
                name: "Workflow Name",
                permissions: this.permissionsModel(true, true, true)
            });

            options = _.defaults({}, options, {
                canvasModel: new CanvasModel({}, {
                    workflowModel: workflowModel
                }),
                selectedView: statusView,
                workflowModel: workflowModel
            });

            var view = new StatusPropertiesView(options).render();

            view._openAddTransitionDialog = sinon.spy();

            return view;
        },

        setup: function () {
            var sandbox = this.sandbox = sinon.sandbox.create();

            sandbox
                .stub(PanelService, "renderPanels")
                .withArgs("workflow.status.properties")
                .returns(["<div class='custom-panel'>My custom panel</div>"]);

            this.createGlobalTransitionDeferred = jQuery.Deferred();
            this.createGlobalTransitionStub = sandbox.stub(WorkflowGlobalTransitionsAJAXManager, "createGlobalTransition").returns(this.createGlobalTransitionDeferred);
            this.deleteGlobalTransitionDeferred = jQuery.Deferred();
            this.deleteGlobalTransitionStub = this.sandbox.stub(WorkflowGlobalTransitionsAJAXManager, "deleteGlobalTransition").returns(this.deleteGlobalTransitionDeferred);
            this.globalTransition = new TransitionModel({actionId: "action-id"});
            this.showErrorMessageSpy = sandbox.stub(Messages, "showErrorMessage");
            this.statusHasGlobalTransitionStub = sandbox.stub(WorkflowModel.prototype, "statusHasGlobalTransition").returns(false);
            this.sandbox.stub(WorkflowModel.prototype, "getGlobalTransitionForStatus").returns(this.globalTransition);
        },

        permissionsModel: function (editPropertyOptions, createStatus, editStatus) {
            return new PermissionsModel({
                editPropertyOptions: editPropertyOptions,
                createStatus: createStatus,
                editStatus: editStatus
            });
        },

        /**
         * @param {boolean} [isDraft=false] Whether the workflow is a draft.
         * @returns {string} The URL of the status properties page.
         */
        statusPropertiesUrl: function (isDraft) {
            return [
                contextPath(),
                "/secure/admin/workflows/ViewWorkflowStepMetaAttributes.jspa?workflowMode=",
                isDraft ? "draft" : "live",
                "&workflowName=Workflow+Name&workflowStep=1"
            ].join("");
        },

        teardown: function () {
            this.sandbox.restore();
        },

        /**
         * Uncheck the global transition checkbox in a `StatusPropertiesView`.
         *
         * @param {StatusPropertiesView} view A `StatusPropertiesView`.
         */
        uncheckGlobalTransitionCheckbox: function (view) {
            view.ui.createGlobalTransition.removeAttr("checked");
            view.ui.createGlobalTransition.change();
        }
    });

    test("Checking the global transition checkbox creates a global transition", function () {
        var canvasModelSetSpy,
            expectedArguments,
            layoutData = {},
            statusModel = new StatusModel(),
            transactionEndSpy = sinon.spy(),
            transactionStartSpy = sinon.spy(),
            view = this.createView(),
            workflowModelGetStatusSpy = this.sandbox.stub(WorkflowModel.prototype, "getStatus").returns(statusModel),
            workflowModelResetSpy = this.sandbox.spy(WorkflowModel.prototype, "reset");

        canvasModelSetSpy = this.sandbox.spy(CanvasModel.prototype, "set");
        expectedArguments = [{
            description: "",
            name: "Status Name",
            screenId: 0,
            statusId: "status-id",
            workflowName: "Workflow Name"
        }];

        view.on({
            "transaction:end": transactionEndSpy,
            "transaction:start": transactionStartSpy
        });

        this.checkGlobalTransitionCheckbox(view);
        equal(this.createGlobalTransitionStub.callCount, 1, "WorkflowGlobalTransitionsAJAXManager.createGlobalTransition was called");
        deepEqual(this.createGlobalTransitionStub.args[0], expectedArguments, "It was passed the correct arguments");
        equal(transactionStartSpy.callCount, 1, "A transaction:start event was triggered");

        this.createGlobalTransitionDeferred.resolve(layoutData);

        ok(canvasModelSetSpy.thisValues[0].get("selectedModel") === statusModel, "The status was re-selected");
        equal(transactionEndSpy.callCount, 1, "A transaction:end event was triggered");
        ok(workflowModelGetStatusSpy.calledWithExactly("status-id"), "The new StatusModel was requested from the WorkflowModel");
        equal(workflowModelResetSpy.callCount, 1, "WorkflowModel#reset was called");
        ok(workflowModelResetSpy.args[0][0] === layoutData, "It was passed the correct layout data");
    });

    test("Clicking the delete button calls StatusView#destroy", function () {
        var destroyStub = this.sandbox.stub(StatusView.prototype, "destroy"),
            view;

        view = this.createView();
        view.$(".delete").click();
        equal(destroyStub.callCount, 1, "StatusView#destroy was called");
    });

    test("Contains a link to draft workflow status properties", function () {
        var link, view;

        view = this.createView({
            workflowModel: new WorkflowModel({
                draft: true,
                name: "Workflow Name",
                permissions: this.permissionsModel(true, true, true)
            })
        });

        link = view.$(".status-properties");
        equal(link.attr("href"), this.statusPropertiesUrl(true), "The properties link is correct");
    });

    test("Contains a link to live workflow status properties", function () {
        var link,
            view = this.createView();

        link = view.$(".status-properties");
        equal(link.attr("href"), this.statusPropertiesUrl(), "The properties link is correct");
    });

    test("Failure to create a global transition is handled", function () {
        var errorMessage = "No global transition for you!",
            view = this.createView();

        this.checkGlobalTransitionCheckbox(view);
        this.createGlobalTransitionDeferred.reject(errorMessage);

        equal(this.showErrorMessageSpy.callCount, 1, "An error message was shown");
        equal(this.showErrorMessageSpy.args[0][0], errorMessage, "Its text is correct");
        ok(!view.ui.createGlobalTransition.is(":checked"), "The global transition checkbox isn't checked");
    });

    test("Failure to destroy a global transition is handled", function () {
        var errorMessage = "This global transition stays",
            view;

        this.statusHasGlobalTransitionStub.returns(true);
        view = this.createView();
        this.uncheckGlobalTransitionCheckbox(view);
        this.deleteGlobalTransitionDeferred.reject(errorMessage);

        equal(this.showErrorMessageSpy.callCount, 1, "An error message was shown");
        equal(this.showErrorMessageSpy.args[0][0], errorMessage, "Its text is correct");
        ok(view.ui.createGlobalTransition.is(":checked"), "The global transition checkbox is checked");
    });

    test("The global transition checkbox is checked if the status has a global transition", function () {
        var view;

        this.statusHasGlobalTransitionStub.returns(true);
        view = this.createView();

        ok(view.ui.createGlobalTransition.is(":checked"), "The global transition checkbox is checked");
    });

    test("The global transition checkbox isn't checked if the status doesn't have a global transition", function () {
        // statusHasGlobalTransitionStub returns false.
        ok(!this.createView().ui.createGlobalTransition.is(":checked"),
            "The global transition checkbox isn't checked");
    });

    test("Unchecking the global transition checkbox destroys the status's global transition", function () {
        var expectedArguments,
            removeTransitionSpy = this.sandbox.spy(WorkflowModel.prototype, "removeTransition"),
            transactionEndSpy = sinon.spy(),
            transactionStartSpy = sinon.spy(),
            view;

        expectedArguments = [{
            transitionId: "action-id",
            workflowName: "Workflow Name"
        }];

        this.statusHasGlobalTransitionStub.returns(true);
        view = this.createView();
        view.on({
            "transaction:start": transactionStartSpy,
            "transaction:end": transactionEndSpy
        });

        this.uncheckGlobalTransitionCheckbox(view);
        equal(this.deleteGlobalTransitionStub.callCount, 1, "WorkflowGlobalTransitionsAJAXManager.deleteGlobalTransition was called");
        deepEqual(this.deleteGlobalTransitionStub.args[0], expectedArguments, "It was passed the correct arguments");
        equal(transactionStartSpy.callCount, 1, "A transaction:start event was triggered");

        this.deleteGlobalTransitionDeferred.resolve();
        equal(removeTransitionSpy.callCount, 1, "WorkflowModel#removeTransition was called");
        ok(removeTransitionSpy.args[0][0] === this.globalTransition, "It was passed the correct global transition");
        equal(transactionEndSpy.callCount, 1, "A transaction:end event was triggered");
    });

    test("Clicking on add transition quick fix invokes add transition function", function () {
        var view = this.createView();
        view.$(".add-transition-quick-fix").click();
        equal(1, view._openAddTransitionDialog.callCount, "transition dialog is opened when a button with .add-transition-quick-fix class is clicked");
    });

    test("Panels registered via the panel service get rendered", function () {
        var view = this.createView();

        ok(PanelService.renderPanels.calledWith("workflow.status.properties", {
            statusId: "status-id",
            stepId: 1,
            workflowName: "Workflow Name"
        }), "Panel service is called with correct context");

        equal(view.$(".custom-panel").text(), "My custom panel");
    });
});
