define("workflow-designer/properties-panel/properties-view", [
    "workflow-designer/backbone",
    "workflow-designer/underscore"
], function(
    Backbone,
    _
) {
    return Backbone.Marionette.ItemView.extend(
    /** @lends JIRA.WorkflowDesigner.PropertiesPanel.PropertiesView# */
    {
        className: "properties-view",

        /**
         * Initialize the view.
         *
         * @classdesc
         * The contents of the workflow designer's properties panel.
         *
         * This view is abstract; do not create instances of it, use those that extend it.
         * @constructs
         * @extends Backbone.View
         * @param {object} options
         * @param {JIRA.WorkflowDesigner.CanvasModel} options.canvasModel The canvas model.
         * @param {Backbone.View} options.selectedView The workflow designer's currently selected view.
         * @param {JIRA.WorkflowDesigner.WorkflowModel} options.workflowModel The workflow model.
         */
        initialize: function (options) {
            this._canvasModel = options.canvasModel;
            this._selectedView = options.selectedView;
            this._workflowModel = options.workflowModel;

            this._selectedView && this.listenTo(this._selectedView.model, "change", this._onChange);
        },

        /**
         * Add or remove an AUI expander to the description, depending on its size.
         */
        applyDescriptionExpander: function () {
            var description = this.$("#properties-panel-description"),
                maxHeight = parseInt(description.css("max-height"), 10),
                needsExpander;

            if (description.length === 0) {
                return;
            }

            // Calculate the description's unrestricted height.
            description.css({height: "auto", maxHeight: "none"});
            needsExpander = description.height() > maxHeight;

            if (needsExpander) {
                description.css({height: "", minHeight: maxHeight});
            } else {
                description.find(".aui-expander-trigger").remove();
                description.removeClass("aui-expander-content");
            }
        },

        /**
         * Destroy the associated view.
         *
         * @protected
         */
        _destroy: function () {
            this._selectedView.destroy();
        },

        /**
         * Edit the selected view.
         *
         * @protected
         */
        _edit: function () {
            this._selectedView.edit();
        },

        /**
         * Re-render the view when its model changes.
         *
         * @private
         */
        _onChange: function () {
            this.render().applyDescriptionExpander();
        },

        /**
         * @return {object} The data to pass to the view's template.
         */
        serializeData: function () {
            var selectedView = this._selectedView && this._selectedView.model.toJSON() || {};
            return _.extend({}, selectedView, {
                optionsEditable: this._workflowModel.get("permissions").get("editPropertyOptions"),
                administrator: this._workflowModel.get("permissions").get("administrator")
            });
        }
    });
});

AJS.namespace("JIRA.WorkflowDesigner.PropertiesPanel.PropertiesView", null, require("workflow-designer/properties-panel/properties-view"));