define("workflow-designer/line-resize-handle-mixin", [
    "workflow-designer/status-port",
    "workflow-designer/draw-2d",
    "workflow-designer/underscore"
], function(
    StatusPort,
    draw2d,
    _
) {
    /**
     * Custom implementation of LineResizeHandler's. Features:
     *
     * * Adds a bigger actionable area around the handler
     * * Sets the handler style according to our design
     * * Displays/hides the ports of the target status
     *
     * To implement this mixin, the view must implement a `getGap()` that returns
     * the gap applied to the connection
     *
     * @mixin
     */
    return {
        /**
         * Initialises the `LineResizeHandle`.
         */
        init: function () {
            this._actionableArea = new draw2d.shape.basic.Circle(20);
            this._actionableArea.draggable = true;
            this.setActionableAreaAttributes({});
        },

        /**
         * Sets the position of this `LineResizeHandle` and its actionable area. This method will
         * move the `actionableArea` so the center is aligned with the LineResizeHandler center
         */
        setPosition: function () {
            this._actionableArea.setPosition(
                this.x - this._actionableArea.width / 2 + this.width / 2,
                this.y - this._actionableArea.height / 2 + this.height / 2
            );
        },

        /**
         * Shows this `LineResizeHandle` and its actionable area.
         *
         * @param {draw2d.Canvas} canvas Canvas where this figure is contained.
         */
        show: function (canvas) {
            this._actionableArea.setCanvas(canvas);
            this._actionableArea.sendToFront();
        },

        /**
         * Hides this `LineResizeHandle` and its actionable area.
         */
        hide: function () {
            this._actionableArea.setCanvas(null);
        },

        /**
         * Repaints this `LineResizeHandle` and its actionable area.
         */
        repaint: function (attributes) {
            attributes = _.extend({}, attributes, {
                cursor: "crosshair",
                fill: "#333333",
                stroke: "#333333"
            });

            var gap = this.getGap();
            this.x -= gap.x;
            this.y -= gap.y;

            if (this._actionableArea && this._actionableAreaAttributes) {
                this._actionableArea.repaint(this._actionableAreaAttributes);
                this._actionableArea.sendToFront();
            }

            // Cocktail will call draw2d.shape.basic.LineResizeHandle.prototype.repaint() before calling this method,
            // that's why we need to call it again with the proper attributes. This creates a double repaint, but it
            // should be fine.
            return draw2d.shape.basic.LineResizeHandle.prototype.repaint.call(this, attributes);
        },

        /**
         * Sets the actionable area attributes. Defaults are:
         *
         *  * "cursor": "crosshair",
         *  * "fill-opacity": 0,
         *  * "stroke-opacity": 0
         *
         * @param {object} attributes Attributes to apply.
         */
        setActionableAreaAttributes: function (attributes) {
            this._actionableAreaAttributes = _.defaults({}, attributes, {
                "cursor": "crosshair",
                "fill-opacity": 0,
                "stroke-opacity": 0
            });
        },

        /**
         * Tests if the x,y coordinates are inside the resize handle or the
         * actionable area.
         *
         * @param {number} x X-coord
         * @param {number} y Y-coord
         * @returns {Boolean} True if coordinates are inside the figures, false otherwise.
         */
        hitTest: function (x, y) {
            // As actionableArea will always overlap the LineResizeHandler, checking for
            // a hit in the actionableArea is enough.
            return this._actionableArea.hitTest(x, y);
        },

        /**
         * Sends the actionable area to the front.
         */
        sendActionableAreaToFront: function () {
            this._actionableArea.sendToFront();
        },

        /**
         * Actions to perform when the user starts the drag operation
         */
        onDragStart: function() {
            var port = this.getRelatedPort();
            port.triggerDragStart();

            _.chain(this._getDropTargets(port))
                .without(this.getOppositePortTo(port))
                .invoke("show");
        },

        /**
         * Actions to perform when the user ends the drag operation
         */
        onDragEnd: function() {
            var port = this.getRelatedPort();
            port.triggerDragEnd();

            _.invoke(this._getDropTargets(port), "hide");
        },

        /**
         * @returns {draw2d.Port|null} Drop target for the specified port at the specified coordinates.
         * @private
         */
        _getDropTarget: function(x, y, port) {
            return StatusPort.getHitPort(x, y, this._getDropTargets(port));
        }
    };

});

AJS.namespace("JIRA.WorkflowDesigner.LineResizeHandleMixin", null, require("workflow-designer/line-resize-handle-mixin"));