AJS.test.require([
    "com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer",
    "com.atlassian.jira.plugins.jira-workflow-designer:test-resources"
], function () {

    var _ = require("workflow-designer/underscore");
    var jQuery = require("jquery");
    var Analytics = require("workflow-designer/analytics");
    var WorkflowModel = require("workflow-designer/workflow-model");
    var TransitionModel = require("workflow-designer/transition-model");
    var WorkflowTransitionsAJAXManager = require("workflow-designer/io/ajax/workflow-transitions-ajax-manager");
    var ReuseTransitionFormView = require("workflow-designer/reuse-transition-form-view");
    var TestUtilities = require("workflow-designer/test-utilities");

    module("ReuseTransitionFormView", {
        /**
         * Create and render a `ReuseTransitionFormView`.
         *
         * The view and its dependencies are seeded with test data.
         *
         * @param {object} [options] Options to pass to the constructor.
         * @returns {JIRA.WorkflowDesigner.ReuseTransitionFormView} The view.
         */
        createView: function (options) {
            var view;

            options = _.defaults({}, options, {
                model: new TransitionModel(),
                sourceStatuses: [
                    {text: "Open", value: 1}
                ],
                targetStatuses: [
                    {text: "Closed", value: 2},
                    {text: "Open", value: 1}
                ],
                workflowModel: this.createWorkflowModel()
            });

            view = new ReuseTransitionFormView(options);
            this.sandbox.spy(view.model, "set");

            view.$el.appendTo("#qunit-fixture");
            view.render();
            view.onShow();
            return view;
        },

        /**
         * Create and render a `ReuseTransitionFormView` with no transitions.
         *
         * @returns {JIRA.WorkflowDesigner.ReuseTransitionFormView} The view.
         */
        createViewWithNoTransitions: function () {
            var workflowModel;

            workflowModel = this.createWorkflowModel();
            workflowModel.get("transitions").reset();

            return this.createView({
                workflowModel: workflowModel
            });
        },

        /**
         * Create a `WorkflowModel` seeded with test data.
         *
         * @returns {JIRA.WorkflowDesigner.WorkflowModel} A workflow model.
         */
        createWorkflowModel: function () {
            var statuses, workflowModel;

            workflowModel = new WorkflowModel({
                name: "Workflow"
            });

            statuses = workflowModel.get("statuses");
            statuses.add([{statusId: 1, stepId: 1}, {statusId: 2, stepId: 2}]);

            workflowModel.get("transitions").add([{
                actionId: 1,
                name: "Close",
                source: statuses.at(0),
                target: statuses.at(1)
            }]);

            return workflowModel;
        },

        setup: function () {
            var sandbox = this.sandbox = sinon.sandbox.create();

            this.addCommonTransitionDeferred = jQuery.Deferred();
            this.addCommonTransitionStub = sandbox.stub(WorkflowTransitionsAJAXManager, "addCommonTransition").returns(this.addCommonTransitionDeferred);
        },

        teardown: function () {
            this.sandbox.restore();
        }
    });

    test("Changing the target status updates the transition field", function () {
        var view;

        view = this.createView();
        ok(view.ui.noTransitionsMessage.is(":hidden"), "The \"no transitions\" message is hidden");

        view.ui.targetStepId.val("1");
        view.ui.targetStepId.trigger("selected");
        ok(view.ui.noTransitionsMessage.is(":visible"), "The \"no transitions\" message is visible");
    });

    test("Clears the transition's angles on status change", function () {
        var expectedArributes, view;

        expectedArributes = {
            sourceAngle: null,
            targetAngle: null
        };

        view = this.createView();
        view.ui.sourceStepId.trigger("selected");
        ok(view.model.set.getCall(0).calledWithExactly(expectedArributes),
            "The transition's angles were cleared after changing the source status");

        view.ui.targetStepId.trigger("selected");
        ok(view.model.set.getCall(0).calledWithExactly(expectedArributes),
            "The transition's angles were cleared after changing the target status");
    });

    test("Common transitions are squashed in the transition field", function () {
        var options, view, workflowModel;

        // Add a common transition.
        workflowModel = this.createWorkflowModel();
        workflowModel.get("transitions").add({
            actionId: 1,
            name: "Close",
            source: workflowModel.get("statuses").at(1),
            target: workflowModel.get("statuses").at(1)
        });

        view = this.createView({workflowModel: workflowModel});

        options = view.ui.transitionId.find("option");
        equal(options.length, 1, "The transition field has one option");
        equal(options.text(), "Close", "The option's text is correct");
        equal(options.val(), "1", "The option's value is correct");
    });

    test("Lists transitions in alphabetical order", function () {
        var expectedOptions, options, view, workflowModel;

        expectedOptions = [{
            text: "Close",
            value: "1"
        }, {
            text: "Stay Closed",
            value: "2"
        }];

        workflowModel = this.createWorkflowModel();
        workflowModel.get("transitions").add({
            actionId: 2,
            name: "Stay Closed",
            source: workflowModel.get("statuses").at(1),
            target: workflowModel.get("statuses").at(1)
        });

        view = this.createView({workflowModel: workflowModel});
        options = view.ui.transitionId.find("option").map(function () {
            var option = jQuery(this);

            return {
                text: option.text(),
                value: option.val()
            };
        }).toArray();

        deepEqual(options, expectedOptions, "The transitions are listed in alphabetical order");
    });

    test("Shows a message when there are no transitions to the selected target status", function () {
        var view = this.createViewWithNoTransitions();
        ok(view.$("#transition-id-single-select").is(":hidden"), "The transition ID field is not visible");
        ok(view.ui.noTransitionsMessage.is(":visible"), "The \"no transitions\" message is visible");
    });

    test("submit() rejects the returned deferred if there are no transitions to the target status", function () {
        var submitPromise = this.createViewWithNoTransitions().submit();
        equal(submitPromise.state(), "rejected", "The returned promise was rejected");
    });

    test("submit() rejects the returned deferred on failure", function () {
        var submitPromise;

        submitPromise = this.createView().submit();
        this.addCommonTransitionDeferred.reject("Error");

        equal(submitPromise.state(), "rejected", "The returned promise was rejected");
        submitPromise.fail(function (errorMessage) {
            equal(errorMessage, "Error", "It was rejected with the correct error message");
        });
    });

    test("submit() successfully reuses a transition", function () {
        var expectedAddCommonTransitionOptions, expectedAttributes, submitPromise, view, workflowModel;

        view = this.createView();
        workflowModel = view.options.workflowModel;
        this.sandbox.spy(Analytics, "triggerAddCommonTransition");

        expectedAddCommonTransitionOptions = {
            sourceStatusId: 1,
            transitionId: 1,
            workflowName: "Workflow"
        };

        expectedAttributes = workflowModel.get("transitions").at(0).omit("id", "sourceAngle", "targetAngle");
        expectedAttributes = _.extend(expectedAttributes, {
            source: workflowModel.get("statuses").at(0)
        });

        TestUtilities.fill(view.el, {
            sourceStepId: 1,
            targetStepId: 2,
            transitionId: 1
        });

        submitPromise = view.submit();

        equal(submitPromise.state(), "pending", "The returned promise is pending");
        ok(this.addCommonTransitionStub.calledWithExactly(expectedAddCommonTransitionOptions),
            "WorkflowTransitionsAJAXManager.addCommonTransition was called with the correct arguments");

        this.addCommonTransitionDeferred.resolve();
        equal(submitPromise.state(), "resolved", "The returned promise was resolved");
        equal(Analytics.triggerAddCommonTransition.callCount, 1, "An analytics event was triggered");
        sinon.assert.calledWith(Analytics.triggerAddCommonTransition, workflowModel.get('permissions'));
        ok(view.model.set.calledWithExactly(expectedAttributes),
            "The correct attributes were set on the TransitionModel");
    });

    test("The initial transition isn't shown in the transition field", function () {
        var view, workflowModel;

        workflowModel = this.createWorkflowModel();
        workflowModel.get("statuses").at(0).set("initial", true);

        view = this.createView({workflowModel: workflowModel});
        equal(view.ui.transitionId.find("option").length, 0,
            "The transitions field contains no options");
    });

    test("Triggers a \"createNewTransition\" event when the create new transition link is clicked", function () {
        var spy = sinon.spy(),
            view;

        this.sandbox.stub(AJS, "format", function () {
            return Array.prototype.join.call(arguments, "");
        });

        view = this.createViewWithNoTransitions();
        view.on("createNewTransition", spy);

        view.$(".create-new-transition").click();
        equal(spy.callCount, 1, "A createNewTransition event was triggered");
    });

});
