AJS.test.require([
    "com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer",
    "com.atlassian.jira.plugins.jira-workflow-designer:test-resources"
], function () {

    var jQuery = require("jquery");
    var TestUtilities = require("workflow-designer/test-utilities");
    var WorkflowModel = require("workflow-designer/workflow-model");
    var PermissionModel = require("workflow-designer/permissions-model");
    var TransitionModel = require("workflow-designer/transition-model");
    var NewTransitionFormView = require("workflow-designer/new-transition-form-view");
    var WorkflowTransitionsAJAXManager = require("workflow-designer/io/ajax/workflow-transitions-ajax-manager");
    var WorkflowLoopedTransitionsAJAXManager = require("workflow-designer/io/ajax/workflow-looped-transitions-ajax-manager");

    module("NewTransitionFormView", {
        /**
         * Create and render a `NewTransitionFormView`.
         *
         * The view and its dependencies are seeded with test data.
         *
         * @returns {NewTransitionFormView} The view.
         */
        createView: function (permissions) {
            var view;

            permissions = permissions || new PermissionModel({
                    editPropertyOptions: true,
                    selectScreenOnTransition: true
                });

            view = new NewTransitionFormView({
                model: new TransitionModel(),
                screens: [{text: "Screen", value: 1}],
                sourceStatuses: [{text: "Open", value: 2}],
                targetStatuses: [{text: "Closed", value: 3}],
                workflowModel: new WorkflowModel({
                    name: "Workflow",
                    permissions: permissions
                })
            });

            this.sandbox.spy(view.model, "set");
            view.options.workflowModel.get("statuses").add([{stepId: 2}, {stepId: 3}]);
            view.render();
            view.onShow();

            return view;
        },

        setup: function () {
            var sandbox = this.sandbox = sinon.sandbox.create();

            this.addTransitionDeferred = jQuery.Deferred();
            this.addLoopedTransitionDeferred = jQuery.Deferred();
            this.addTransitionStub = sandbox.stub(WorkflowTransitionsAJAXManager, "addTransition").returns(this.addTransitionDeferred);
            this.addLoopedTransitionStub = sandbox.stub(WorkflowLoopedTransitionsAJAXManager, "addLoopedTransition").returns(this.addLoopedTransitionDeferred);
        },

        teardown: function () {
            this.sandbox.restore();
        }
    });

    test("Clears the transition's angles on status change", function () {
        var expectedArributes, view;

        expectedArributes = {
            sourceAngle: null,
            targetAngle: null
        };

        view = this.createView();
        view.ui.sourceStepId.trigger("selected");
        ok(view.model.set.getCall(0).calledWithExactly(expectedArributes),
            "The transition's angles were cleared after changing the source status");

        view.ui.targetStepId.trigger("selected");
        ok(view.model.set.getCall(0).calledWithExactly(expectedArributes),
            "The transition's angles were cleared after changing the target status");
    });

    test("submit() rejects the returned deferred on failure", function () {
        var submitPromise;

        submitPromise = this.createView().submit();
        this.addTransitionDeferred.reject("Error");

        equal(submitPromise.state(), "rejected", "The returned promise was rejected");
        submitPromise.fail(function (errorMessage) {
            equal(errorMessage, "Error", "It was rejected with the correct error message");
        });
    });

    test("submit() successfully creates a new transition", function () {
        var expectedAddTransitionOptions, expectedAttributes, submitPromise, view;

        view = this.createView();
        expectedAddTransitionOptions = {
            description: "Close the issue.",
            name: "Close",
            screenId: 1,
            sourceStepId: 2,
            targetStepId: 3,
            workflowName: "Workflow"
        };

        expectedAttributes = {
            description: "Close the issue.",
            name: "Close",
            screenId: 1,
            screenName: "Screen",
            global: false,
            globalTransition: false,
            looped: false,
            loopedTransition: false,
            source: view.options.workflowModel.get("statuses").at(0),
            target: view.options.workflowModel.get("statuses").at(1)
        };

        TestUtilities.fill(view.el, {
            "#transition-description": "Close the issue.",
            "#transition-name": "Close",
            "#transition-screen-id": 1,
            "#transition-source-step-id": 2,
            "#transition-target-step-id": 3
        });

        submitPromise = view.submit();

        equal(submitPromise.state(), "pending", "The returned promise is pending");
        ok(this.addTransitionStub.calledWithExactly(expectedAddTransitionOptions),
            "WorkflowTransitionsAJAXManager.addTransition was called with the correct arguments");

        this.addTransitionDeferred.resolve({global: false, looped: false});
        equal(submitPromise.state(), "resolved", "The returned promise was resolved");
        ok(view.model.set.calledWithExactly(expectedAttributes),
            "The correct attributes were set on the TransitionModel");
    });

    test("submit() successfully creates a new looped transition", function () {
        var expectedAddTransitionOptions, expectedAttributes, submitPromise, view;

        view = this.createView();

        expectedAddTransitionOptions = {
            description: "Close the issue.",
            name: "Close",
            screenId: 1,
            sourceStepId: -65536,
            targetStepId: -65537,
            workflowName: "Workflow"
        };

        expectedAttributes = {
            description: "Close the issue.",
            global: false,
            globalTransition: false,
            looped: true,
            loopedTransition: true,
            name: "Close",
            screenId: 1,
            screenName: "Screen",
            source: undefined,
            target: undefined
        };

        TestUtilities.fill(view.el, {
            "#transition-description": "Close the issue.",
            "#transition-name": "Close",
            "#transition-screen-id": 1,
            "#transition-source-step-id": -65536,
            "#transition-target-step-id": -65537
        });

        submitPromise = view.submit();

        equal(submitPromise.state(), "pending", "The returned promise is pending");
        ok(this.addLoopedTransitionStub.calledWithExactly(expectedAddTransitionOptions),
            "WorkflowTransitionsAJAXManager.addTransition was called with the correct arguments");

        this.addLoopedTransitionDeferred.resolve({global: false, looped: true});
        equal(submitPromise.state(), "resolved", "The returned promise was resolved");
        ok(view.model.set.calledWithExactly(expectedAttributes),
            "The correct attributes were set on the TransitionModel for a looped transition");
    });

    test("When user has screen permission then Screen select is present", function () {
        var view = this.createView();

        equal(view.$el.find('#transition-screen-id').length, 1, "Screen select should be present");
    });

    test("When user doesn't have screen permission then no Screen select is present", function () {
        var view = this.createView(new PermissionModel({selectScreenOnTransition: false}));

        equal(view.$el.find('#transition-screen-id').length, 0, "No screen select should be present");
    });

    test("When user doesn't have screen permission 0 is submitted as screenid", function () {
        var expectedAddTransitionOptions, expectedAttributes, submitPromise, view;

        view = this.createView(new PermissionModel({selectScreenOnTransition: false}));
        expectedAddTransitionOptions = {
            description: "Close the issue.",
            name: "Close",
            screenId: 0,
            sourceStepId: 2,
            targetStepId: 3,
            workflowName: "Workflow"
        };

        expectedAttributes = {
            description: "Close the issue.",
            name: "Close",
            screenId: 0,
            screenName: "",
            global: false,
            globalTransition: false,
            looped: false,
            loopedTransition: false,
            source: view.options.workflowModel.get("statuses").at(0),
            target: view.options.workflowModel.get("statuses").at(1)
        };

        TestUtilities.fill(view.el, {
            "#transition-description": "Close the issue.",
            "#transition-name": "Close",
            "#transition-screen-id": 1,
            "#transition-source-step-id": 2,
            "#transition-target-step-id": 3
        });

        submitPromise = view.submit();

        equal(submitPromise.state(), "pending", "The returned promise is pending");
        ok(this.addTransitionStub.calledWithExactly(expectedAddTransitionOptions),
            "WorkflowTransitionsAJAXManager.addTransition was called with the correct arguments");

        this.addTransitionDeferred.resolve({global: false, looped: false});
        equal(submitPromise.state(), "resolved", "The returned promise was resolved");
        ok(view.model.set.calledWithExactly(expectedAttributes),
            "The correct attributes were set on the TransitionModel");
    });
});
