define("workflow-designer/dialogs/add-transition-dialog-view", [
    "workflow-designer/dialogs/transition-dialog-view",
    "workflow-designer/reuse-transition-form-view",
    "workflow-designer/new-transition-form-view",
    "workflow-designer/analytics",
    "workflow-designer/templates",
    "aui/tabs",
    "workflow-designer/backbone",
    "workflow-designer/underscore"
], function(
    TransitionDialogView,
    ReuseTransitionFormView,
    NewTransitionFormView,
    Analytics,
    Templates,
    AuiTabs,
    Backbone,
    _
) {

    /**
     * @class JIRA.WorkflowDesigner.Dialogs.AddTransitionDialogView
     * @classdesc The workflow designer's "Add Transition" dialog.
     * @extends JIRA.WorkflowDesigner.Dialogs.TransitionDialogView
     */
    return TransitionDialogView.extend(
    /** @lends JIRA.WorkflowDesigner.Dialogs.AddTransitionDialogView# */
    {
        events: {
            tabSelect: "_onTabChange"
        },

        id: "add-transition-dialog",

        regions: {
            newTransitionPane: "#new-transition-pane",
            reuseTransitionPane: "#reuse-transition-pane"
        },

        tabsTemplate: Templates.addTransitionDialogTabs,
        template: Templates.addTransitionDialog,

        /**
         * Create and show a `NewTransitionFormView`.
         *
         * @param {object} [options] Options to pass to the view's constructor.
         * @private
         */
        _createNewTransitionForm: function (options) {
            var newTransitionForm;

            options = _.extend({}, options, {
                screens: this._screenOptions
            });

            newTransitionForm = new NewTransitionFormView(options);
            this.newTransitionPane.show(newTransitionForm);
        },

        /**
         * Create and show a `ReuseTransitionFormView`.
         *
         * @param {object} [options] Options to pass to the view's constructor.
         * @private
         */
        _createReuseTransitionForm: function (options) {
            var reuseTransitionForm;

            reuseTransitionForm = new ReuseTransitionFormView(options);
            this.listenTo(reuseTransitionForm, "createNewTransition", this._showNewTransitionPane);
            this.reuseTransitionPane.show(reuseTransitionForm);
        },

        /**
         * Create and show the "New transition" and "Reuse transition" tabs.
         *
         * @private
         */
        _createTabs: function () {
            this.dialog.get$popupHeading().append(this.tabsTemplate());
            AuiTabs.setup();
            this.dialog._positionInCenter();
        },

        /**
         * @returns {function} A function that retrieves the dialog's content.
         * @private
         */
        _getContent: function () {
            var instance = this,
                request;

            return function (callback) {
                request = instance._getScreenOptions();
                request.done(function (screenOptions) {
                    instance._screenOptions = screenOptions;
                    callback(instance.template());
                });

                request.fail(instance._onLoadError);
            };
        },

        /**
         * Build a list of status options, optionally selecting one.
         *
         * Suitable to be passed to the `aui.forms.selectField` soy template.
         *
         * @param {number} [selectedStatusId] The ID of the status that should be selected.
         * @private
         */
        _getStatusOptions: function (selectedStatusId) {
            var statuses = this._workflowModel.get("statuses");

            function getName(status) {
                return status.get("name").toLowerCase();
            }

            function isInitial(status) {
                return status.get("initial");
            }

            function makeOption(status) {
                return {
                    text: status.get("name"),
                    value: status.get("stepId"),
                    selected: status.get("stepId") === selectedStatusId
                };
            }

            return statuses
                .chain()
                .reject(isInitial)
                .sortBy(getName)
                .map(makeOption)
                .value();
        },

        /**
         * @returns {Backbone.View} The currently visible view.
         * @private
         */
        _getVisibleView: function () {
            var views;

            views = [
                this.newTransitionPane.currentView,
                this.reuseTransitionPane.currentView
            ];

            return _.find(views, function (view) {
                return view.$el.closest(".active-pane").length;
            });
        },

        /**
         * @returns {string} The window title to use when the dialog is visible.
         */
        _getWindowTitle: function () {
            return AJS.I18n.getText("workflow.designer.add.transition");
        },

        /**
         * Clear the transition's angles if the source or target fields change.
         *
         * @private
         */
        _onContentRefresh: function () {
            var options;

            options = {
                model: this._transitionModel,
                sourceStatuses: this._getStatusOptions(this._transitionModel.getSourceStepId()),
                targetStatuses: this._getStatusOptions(this._transitionModel.getTargetStepId()),
                workflowModel: this._workflowModel
            };

            this._createNewTransitionForm(options);
            this._createReuseTransitionForm(options);
            this._createTabs();
        },

        /**
         * @param {object} e The submit event.
         * @param {function} callback A function to call after submission fails.
         * @private
         */
        _onSubmit: function (e, callback) {
            var request;

            TransitionDialogView.prototype._onSubmit.apply(this, arguments);
            request = this._getVisibleView().submit();
            request.done(this._onSuccess).fail(callback).fail(this._onError);
        },

        /**
         * Handle successful submission by updating the transition model.
         *
         * @private
         */
        _onSuccess: function () {
            var duration = new Date() - this._workflowModel.get("loadedAt");
            Analytics.triggerAddTransition(this._workflowModel.get("permissions"));
            Analytics.triggerFirstAddTransition(duration, this._workflowModel.get("permissions"));

            this.hide();
            this._workflowModel.addTransition(this._transitionModel);
            this._canvasModel.set("selectedModel", this._transitionModel);
        },

        _onTabChange: function () {
            this.$(":input:visible:first").focus();
            this.reposition();
        },

        /**
         * Show an error message at the top of the dialog.
         *
         * We override this method to prepend the message to the currently
         * visible view; that way, it disappears after switching tabs.
         *
         * @param {string} message The error message to show.
         */
        showErrorMessage: function (message) {
            var error = aui.message.error({content: message}),
                visibleView = this._getVisibleView();

            this.removeErrorMessages();
            visibleView.$el.prepend(error);
            this.reposition();
        },

        /**
         * Show the "New transition" pane.
         *
         * @private
         */
        _showNewTransitionPane: function () {
            var newTransitionLink = this.$(".aui-tabs .new-transition > a");
            AuiTabs.change(newTransitionLink);
        }
    });
});

AJS.namespace("JIRA.WorkflowDesigner.Dialogs.AddTransitionDialogView", null, require("workflow-designer/dialogs/add-transition-dialog-view"));