AJS.test.require([
    "com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer",
    "com.atlassian.jira.plugins.jira-workflow-designer:test-resources"
], function () {

    var _;
    var jQuery;
    var TestUtilities;
    var Messages;
    var WorkflowModel;
    var PermissionModel;
    var CanvasModel;
    var TransitionModel;
    var ScreensAJAXManager;
    var FormDialogView;
    var AddTransitionDialogView;
    var ReuseTransitionFormView;
    var NewTransitionFormView;
    var AuiTabs = require("aui/tabs");
    var Analytics;

    module("AddTransitionDialogView", {
        /**
         * Create a `WorkflowModel` seeded with test data.
         *
         * @returns {WorkflowModel} The model.
         */
        createWorkflowModel: function (permissions) {
            var workflowModel;

            workflowModel = new WorkflowModel({
                permissions: new PermissionModel({
                    editPropertyOptions: true,
                    selectScreenOnTransition: true
                })
            });
            workflowModel.get("statuses").add([{
                name: "Open",
                stepId: 1
            }, {
                name: "In Progress",
                stepId: 2
            }, {
                name: "Closed",
                stepId: 3
            }]);

            if (permissions) {
                workflowModel.set('permissions', permissions);
            }

            return workflowModel;
        },

        setup: function () {
            this.sandbox = sinon.sandbox.create();
            this.context = AJS.test.mockableModuleContext();

            this.context.mock("workflow-designer/analytics", require("workflow-designer/analytics"));
            this.newTransitionFormViewSpy = this.sandbox.spy(require("workflow-designer/new-transition-form-view"));
            this.context.mock("workflow-designer/new-transition-form-view", this.newTransitionFormViewSpy);
            this.reuseTransitionFormViewSpy = this.sandbox.spy(require("workflow-designer/reuse-transition-form-view"));
            this.context.mock("workflow-designer/reuse-transition-form-view", this.reuseTransitionFormViewSpy);

            this.context.mock("workflow-designer/dialogs/form-dialog-view", require("workflow-designer/dialogs/form-dialog-view"));
            this.context.mock("workflow-designer/io/ajax/screens-ajax-manager", require("workflow-designer/io/ajax/screens-ajax-manager"));
            this.context.mock("workflow-designer/transition-model", require("workflow-designer/transition-model"));
            this.context.mock("workflow-designer/messages", require("workflow-designer/messages"));
            this.context.mock("workflow-designer/test-utilities", require("workflow-designer/test-utilities"));

            FormDialogView = this.context.require("workflow-designer/dialogs/form-dialog-view");
            ScreensAJAXManager = this.context.require("workflow-designer/io/ajax/screens-ajax-manager");
            TransitionModel = this.context.require("workflow-designer/transition-model");
            CanvasModel = this.context.require("workflow-designer/canvas-model");
            WorkflowModel = this.context.require("workflow-designer/workflow-model");
            PermissionModel = this.context.require("workflow-designer/permissions-model");
            Messages = this.context.require("workflow-designer/messages");
            TestUtilities = this.context.require("workflow-designer/test-utilities");
            jQuery = this.context.require("jquery");
            _ = this.context.require("workflow-designer/underscore");
            Analytics = this.context.require("workflow-designer/analytics");

            NewTransitionFormView = this.context.require("workflow-designer/new-transition-form-view");
            ReuseTransitionFormView = this.context.require("workflow-designer/reuse-transition-form-view");
            AddTransitionDialogView = this.context.require("workflow-designer/dialogs/add-transition-dialog-view");

            this.getScreensDeferred = jQuery.Deferred().resolve([]);
            this.getScreensStub = this.sandbox.stub(ScreensAJAXManager, "getScreens").returns(this.getScreensDeferred);
            this.newTransitionFormViewSubmitDeferred = jQuery.Deferred();
            this.newTransitionFormViewSubmitStub = this.sandbox.stub(NewTransitionFormView.prototype, "submit").returns(this.newTransitionFormViewSubmitDeferred);
            this.reuseTransitionFormViewSubmitDeferred = jQuery.Deferred();
            this.reuseTransitionFormViewSubmitStub = this.sandbox.stub(ReuseTransitionFormView.prototype, "submit").returns(this.reuseTransitionFormViewSubmitDeferred);
            this.showErrorMessageSpy = this.sandbox.spy(AddTransitionDialogView.prototype, "showErrorMessage");

            this.workflowModel = this.createWorkflowModel();
        },

        /**
         * Create and show an `AddTransitionDialogView`.
         *
         * @param {object} [options] Options to pass to the constructor.
         * @returns {JIRA.WorkflowDesigner.Dialogs.AddTransitionDialogView} The view.
         */
        showDialog: function (options) {
            options = _.defaults({}, options, {
                transitionModel: new TransitionModel(),
                workflowModel: this.workflowModel
            });

            options.canvasModel || (options.canvasModel = new CanvasModel({}, {
                workflowModel: options.workflowModel
            }));

            return new AddTransitionDialogView(options).show();
        },

        teardown: function () {
            TestUtilities.removeDialogs();
            this.sandbox.restore();
        }
    });

    test("Automatically selects the transition's source and target statuses", function () {
        var newTransitionFormView, reuseTransitionFormView, statuses, workflowModel;

        workflowModel = this.createWorkflowModel();
        statuses = workflowModel.get("statuses");

        this.showDialog({
            transitionModel: new TransitionModel({
                source: statuses.at(1),
                target: statuses.at(2)
            })
        });

        newTransitionFormView = this.newTransitionFormViewSpy.thisValues[0];
        equal(newTransitionFormView.ui.sourceStepId.val(), 2, "The transition's source is selected");
        equal(newTransitionFormView.ui.targetStepId.val(), 3, "The transition's target is selected");

        reuseTransitionFormView = this.reuseTransitionFormViewSpy.thisValues[0];
        equal(reuseTransitionFormView.ui.sourceStepId.val(), 2, "The transition's source is selected");
        equal(reuseTransitionFormView.ui.targetStepId.val(), 3, "The transition's target is selected");
    });

    test("Creates and shows a NewTransitionFormView and a ReuseTransitionFormView", function () {
        var dialog = this.showDialog(),
            expectedNewOptions,
            expectedReuseOptions,
            expectedStatusOptions;

        expectedStatusOptions = [{
            selected: false,
            text: "Closed",
            value: 3
        }, {
            selected: false,
            text: "In Progress",
            value: 2
        }, {
            selected: false,
            text: "Open",
            value: 1
        }];

        expectedNewOptions = {
            model: dialog.options.transitionModel,
            screens: [],
            sourceStatuses: expectedStatusOptions,
            targetStatuses: expectedStatusOptions,
            workflowModel: dialog.options.workflowModel
        };

        ok(this.newTransitionFormViewSpy.calledWithExactly(expectedNewOptions), "A NewTransitionFormView was created");
        ok(dialog.newTransitionPane.currentView === this.newTransitionFormViewSpy.thisValues[0],
            "It was shown in the newTransitionPane region");

        expectedReuseOptions = _.omit(expectedNewOptions, "screens", "screensPermission");
        ok(this.reuseTransitionFormViewSpy.calledWithExactly(expectedReuseOptions), "A ReuseTransitionFormView was created");
        ok(dialog.reuseTransitionPane.currentView === this.reuseTransitionFormViewSpy.thisValues[0],
            "It was shown in the reuseTransitionPane region");
    });

    test("Doesn't show the initial status in the status fields", function () {
        var workflowModel;

        workflowModel = this.createWorkflowModel();
        workflowModel.get("statuses").reset([{initial: true}]);

        this.showDialog({workflowModel: workflowModel});

        deepEqual(this.newTransitionFormViewSpy.args[0][0].sourceStatuses, [], "No source statuses were passed to NewTransitionFormView");
        deepEqual(this.newTransitionFormViewSpy.args[0][0].targetStatuses, [], "No target statuses were passed to NewTransitionFormView");
        deepEqual(this.reuseTransitionFormViewSpy.args[0][0].sourceStatuses, [], "No source statuses were passed to ReuseTransitionFormView");
        deepEqual(this.reuseTransitionFormViewSpy.args[0][0].targetStatuses, [], "No target statuses were passed to ReuseTransitionFormView");
    });

    test("Failing to create a new transition", function () {
        var dialog;

        dialog = this.showDialog();
        dialog.$("form").submit();
        this.newTransitionFormViewSubmitDeferred.reject("Nope");

        equal(dialog.$(":input:disabled").length, 0, "All inputs are enabled");
        ok(this.showErrorMessageSpy.calledWithExactly("Nope"), "The correct error message was shown");
    });

    test("Failing to reuse a transition", function () {
        var dialog;

        dialog = this.showDialog();
        AuiTabs.change(dialog.$(".aui-tabs a:last"));
        dialog.$("form").submit();
        this.reuseTransitionFormViewSubmitDeferred.reject("Nope");

        equal(dialog.$(":input:disabled").length, 0, "All inputs are enabled");
        ok(this.showErrorMessageSpy.calledWithExactly("Nope"), "The correct error message was shown");
    });

    test("Hides and shows an error message when loading screen information fails", function () {
        var dialog, getScreensDeferred, showErrorMessageStub;

        getScreensDeferred = jQuery.Deferred();
        showErrorMessageStub = this.sandbox.stub(Messages, "showErrorMessage");
        this.getScreensStub.returns(getScreensDeferred);

        dialog = this.showDialog();
        getScreensDeferred.reject("Error");

        ok(dialog.$el.is(":hidden"), "The dialog isn't visible");
        ok(showErrorMessageStub.calledWithExactly("Error"),
            "showErrorMessage was called with the correct error message");
    });

    test("Repositions on tab change", function () {
        var dialog, repositionSpy;

        dialog = this.showDialog();
        repositionSpy = this.sandbox.spy(FormDialogView.prototype, "reposition");

        dialog.$(".aui-tabs a:last").click();
        equal(repositionSpy.callCount, 1, "FormDialogView#reposition was called");
    });

    test("Requests screen information on show", function () {
        var dialog, getScreensDeferred;

        getScreensDeferred = jQuery.Deferred();
        this.getScreensStub.returns(getScreensDeferred);

        dialog = this.showDialog();
        ok(dialog.$el.is(":hidden"), "The dialog isn't visible");

        getScreensDeferred.resolve([]);
        ok(dialog.$el.is(":visible"), "The dialog is visible");
    });

    test("Shows the new transition pane when ReuseTransitionFormView triggers a createNewTransition event", function () {
        var dialog, newTransitionLink, reuseTransitionFormView, tabsChangeSpy;

        dialog = this.showDialog();
        newTransitionLink = dialog.$(".aui-tabs .new-transition > a");
        tabsChangeSpy = this.sandbox.spy(AuiTabs, "change");

        reuseTransitionFormView = this.reuseTransitionFormViewSpy.thisValues[0];
        reuseTransitionFormView.trigger("createNewTransition");
        ok(tabsChangeSpy.calledWithExactly(newTransitionLink),
            "The new transition pane was shown");
    });

    test("Successfully creates a new transition", function () {
        this.sandbox.stub(Analytics, "triggerAddTransition");
        this.sandbox.stub(Analytics, "triggerFirstAddTransition");

        var dialog = this.showDialog(),
            transitionModel = dialog.options.transitionModel;

        dialog.$("form").submit();
        equal(dialog.$(":input:disabled").length, dialog.$(":input").length, "All inputs are disabled");
        equal(this.newTransitionFormViewSubmitStub.callCount, 1, "NewTransitionFormView#submit was called");

        this.newTransitionFormViewSubmitDeferred.resolve();
        ok(dialog.options.canvasModel.get("selectedModel") === transitionModel, "The TransitionModel is selected");
        ok(dialog.options.workflowModel.get("transitions").contains(transitionModel),
            "The TransitionModel was added to the WorkflowModel's transitions collection");
        ok(!TestUtilities.dialogIsVisible(), "The dialog was hidden");
        ok(Analytics.triggerAddTransition.calledOnce, "Analytics event for add/transition has been triggered");
        sinon.assert.calledWith(Analytics.triggerAddTransition, this.workflowModel.get('permissions'));
        ok(Analytics.triggerFirstAddTransition.calledOnce, "Analytics event for add/transition has been triggered");
        equal(Analytics.triggerFirstAddTransition.getCall(0).args[1], this.workflowModel.get('permissions'));
    });

    test("Successfully reuses an existing transition", function () {
        var dialog = this.showDialog(),
            transitionModel = dialog.options.transitionModel;

        AuiTabs.change(dialog.$(".aui-tabs a:last"));
        dialog.$("form").submit();
        equal(dialog.$(":input:disabled").length, dialog.$(":input").length, "All inputs are disabled");
        equal(this.reuseTransitionFormViewSubmitStub.callCount, 1, "ReuseTransitionFormView#submit was called");

        this.reuseTransitionFormViewSubmitDeferred.resolve();
        ok(dialog.options.canvasModel.get("selectedModel") === transitionModel, "The TransitionModel is selected");
        ok(dialog.options.workflowModel.get("transitions").contains(transitionModel),
            "The TransitionModel was added to the WorkflowModel's transitions collection");
        ok(!TestUtilities.dialogIsVisible(), "The dialog was hidden");
    });

    test("When doesn't have screen permissions no rest call is made to screens", function () {
        var workflowModel = this.createWorkflowModel();
        workflowModel.get("permissions").set({selectScreenOnTransition: false});

        this.showDialog({workflowModel: workflowModel});

        sinon.assert.notCalled(this.getScreensStub, "No rest call is made for screens");
    });

    test("When have screen permissions rest call is made to screens", function () {
        this.showDialog();

        sinon.assert.calledOnce(this.getScreensStub, "Rest call is made for screens");
    });

    test("When doesn't have screen permissions view should not contain screens select", function () {
        var workflowModel = this.createWorkflowModel();
        workflowModel.get("permissions").set({selectScreenOnTransition: false});

        var view = this.showDialog({workflowModel: workflowModel});

        var $dialog = jQuery('#' + view.id);
        equal($dialog.find('#transition-screen-id').length, 0, "No screen select should be present");
    });

    test("When have screen permissions view should contain screens select", function () {
        var view = this.showDialog();

        var $dialog = jQuery('#' + view.id);
        equal($dialog.find('#transition-screen-id').length, 1, "Screen select should be present");
    });
});
