define("workflow-designer/dialogs/add-status-suggestion-handler", [
    "jira/ajs/list/group-descriptor",
    "jira/ajs/list/item-descriptor",
    "jira/ajs/select/suggestions/select-suggest-handler",
    "workflow-designer/underscore"
], function(
    GroupDescriptor,
    ItemDescriptor,
    SelectSuggestHandler,
    _
) {
    /**
     * Make a "new status" option from a given query.
     *
     * @param {string} query The query entered by the user.
     * @return {AJS.GroupDescriptor} An option group containing the create option.
     */
    function makeNewStatusOption(query) {
        return new GroupDescriptor({
            items: [new ItemDescriptor({
                fieldText: query,
                label: AJS.I18n.getText("workflow.designer.add.status.new.option", query),
                value: ""
            })]
        });
    }

    /**
     * @class
     * @classdesc The add status dialog's status name suggestion handler.
     * @extends AJS.SelectSuggestHandler
     */
    return SelectSuggestHandler.extend({
        /**
         * Format suggestions for display.
         *
         * @param {AJS.ItemDescriptor[]} descriptors The suggestions to display.
         * @param {string} query The query entered by the user.
         * @returns {AJS.GroupDescriptor[]} The suggestions to display.
         */
        formatSuggestions: function (descriptors, query) {
            var suggestions;

            descriptors = this._super.apply(this, arguments);
            suggestions = [];

            if (descriptors.length) {
                suggestions.push(new GroupDescriptor({
                    items: descriptors
                }));
            }

            if (this._shouldShowNewStatusOption(query)) {
                suggestions.push(makeNewStatusOption(query));
            }

            return suggestions;
        },

        /**
         * @param {string} query The query entered by the user.
         * @private
         * @return {boolean} Whether the "new status" option should be shown for the given query.
         */
        _shouldShowNewStatusOption: function (query) {
            var isEmpty;
            var isNew;

            if (!this.model.options.canCreateValues) {
                return false;
            }

            query = query.toLowerCase().trim();

            isEmpty = query.length === 0;
            isNew = _.every(this.options.statuses, function (status) {
                return query !== status.name.toLowerCase().trim();
            });

            return !isEmpty && isNew;
        }
    });
});

AJS.namespace("JIRA.WorkflowDesigner.Dialogs.AddStatusSuggestionHandler", null, require("workflow-designer/dialogs/add-status-suggestion-handler"));