AJS.test.require([
    "com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer",
    "com.atlassian.jira.plugins.jira-workflow-designer:test-resources"
], function () {

    var AddStatusSuggestionHandler = require("workflow-designer/dialogs/add-status-suggestion-handler");
    var SelectModel = require("jira/ajs/select/select-model");
    var _ = require("workflow-designer/underscore");
    var jQuery = require("jquery");

    /**
     * Create an `AJS.SelectModel` containing the given statuses.
     *
     * @param {object[]} statuses The status objects to be contained in the `SelectModel`.
     * @param {boolean=false} createStatus whether the user is able to create status
     * @returns {AJS.SelectModel} An `AJS.SelectModel` containing `statuses`.
     */
    function createSelectModel(statuses, createStatus) {
        var select = jQuery("<select>");
        select.get(0).dataset.canCreateValues = '' + !!createStatus;

        select = _.reduce(statuses, function (select, status) {
            return select.append(jQuery("<option>").text(status.name).val(status.id));
        }, select);

        return new SelectModel({
            element: select
        });
    }

    /**
     * Create an `AddStatusSuggestionHandler`.
     *
     * @param {object} [options] Options to pass to the constructor.
     * @param {AJS.SelectModel} [options.selectModel] The model to pass to the constructor.
     * @param {boolean=false} [options.createStatus] whether the user is able to create status
     * @returns {JIRA.WorkflowDesigner.Dialogs.AddStatusSuggestionHandler} A selection handler.
     */
    function createSuggestionHandler(options) {
        var statuses;

        statuses = [{
            id: "closed",
            name: "Closed"
        }, {
            id: "in-progress",
            name: "In Progress"
        }, {
            id: "open",
            name: "Open"
        }];

        options = _.defaults({}, options, {
            selectModel: createSelectModel(statuses.slice(0, -1), options.createStatus),
            statuses: statuses
        });

        return new AddStatusSuggestionHandler(options, options.selectModel);
    }

    module("AddStatusSuggestionHandler with statuses and create option available", {
        setup: function () {
            this.suggestionHandler = createSuggestionHandler({
                createStatus: true
            });
        }
    });

    test("A \"create status\" option is shown in a separate group", function () {
        this.suggestionHandler.execute("New Status").done(function (descriptors) {
            equal(descriptors.length, 2, "Two groups were returned");
            equal(descriptors[1].items().length, 1, "The second group contains a single item");
            ok(!descriptors[1].items()[0].value(), "The item represents a new status");
        });
    });

    test("No \"create status\" option is shown if the query is empty", function () {
        this.suggestionHandler.execute("").done(function (descriptors) {
            equal(descriptors.length, 1, "A single group was returned");
            equal(descriptors[0].items().length, 1, "It contains a single item");
            equal(descriptors[0].items()[0].value(), "in-progress", "The item's value is correct");
        });
    });

    test("No \"create status\" option is shown if the query matches an existing status", function () {
        // The check should be case and whitespace insensitive.
        var queries = ["open", "Open", "In Progress", " In Progress "];

        _.each(queries, function (query) {
            this.suggestionHandler.execute(query).done(function (descriptors) {
                equal(descriptors.length, 1, "A single group was returned");
                equal(descriptors[0].items().length, 1, "It contains a single item");
                equal(descriptors[0].items()[0].value(), "in-progress", "The item's value is correct");
            });
        }, this);
    });


    module("AddStatusSuggestionHandler without statuses but with create option available", {
        setup: function () {
            this.suggestionHandler = createSuggestionHandler({
                selectModel: createSelectModel([], true)
            });
        }
    });

    test("Only the \"create status\" group is returned if there are no existing statuses", function () {
        this.suggestionHandler.execute("New Status").done(function (descriptors) {
            equal(descriptors.length, 1, "A single group was returned");
            equal(descriptors[0].items().length, 1, "It contains a single item");
            ok(!descriptors[0].items()[0].value(), "The item represents a new status");
        });
    });


    module("AddStatusSuggestionHandler with statuses but without create option available", {
        setup: function () {
            this.suggestionHandler = createSuggestionHandler({
                createStatus: false
            });
        }
    });

    test("No \"create status\" option is shown when the user cannot create the status", function () {
        this.suggestionHandler.execute("New Status").done(function (descriptors) {
            equal(descriptors.length, 1, "A single group was returned");
            equal(descriptors[0].items().length, 1, "It contains a single item");
            equal(descriptors[0].items()[0].value(), "in-progress", "The item's value is correct");
        });
    });

    module("AddStatusSuggestionHandler without statuses nor create option available", {
        setup: function () {
            this.suggestionHandler = createSuggestionHandler({
                selectModel: createSelectModel([], false)
            });
        }
    });

    test("Only the \"No Matches\" group is returned if there are no existing statuses and the user cannot create the status", function () {
        this.suggestionHandler.execute("New Status").done(function (descriptors) {
            equal(descriptors.length, 0, "No groups were returned");
        });
    });
});
