define("workflow-designer/dialogs/add-status-inline-dialog-view", [
    "workflow-designer/io/ajax/workflow-statuses-ajax-manager",
    "workflow-designer/dialogs/inline-dialog-view",
    "workflow-designer/dialogs/create-status-dialog-view",
    "workflow-designer/status-model",
    "workflow-designer/dialogs/add-status-suggestion-handler",
    "workflow-designer/messages",
    "workflow-designer/analytics",
    "jira/ajs/select/single-select",
    "workflow-designer/templates",
    "workflow-designer/backbone",
    "workflow-designer/underscore",
    "jquery"
], function(
    WorkflowStatusesAJAXManager,
    InlineDialogView,
    CreateStatusDialogView,
    StatusModel,
    AddStatusSuggestionHandler,
    Messages,
    Analytics,
    SingleSelect,
    Templates,
    Backbone,
    _,
    jQuery
) {
    return InlineDialogView.extend(
    /** @lends JIRA.WorkflowDesigner.Dialogs.AddStatusInlineDialogView# */
    {
        events: {
            "change #create-global-transition-add": "_storeCreateGlobalTransition",
            "click .cancel": "_cancel",
            "selected #status-name": "_onSelection",
            "submit form": "_submit",
            "unselect #status-name": "_clearSelection"
        },

        // The -watchers suffix is required to prevent JIRA's LayerManager.js
        // from closing the inline dialog when the status dropdown appears.
        id: "add-status-dialog-view-watchers",

        template: Templates.addStatusInlineDialog,

        ui: {
            createGlobalTransition: "#create-global-transition-add",
            form: ".contents form",
            statusName: "#status-name",
            submit: ":submit"
        },

        /**
         * Initialize the view.
         *
         * @class
         * @classdesc The "add status" inline dialog.
         * @constructs
         * @extends JIRA.WorkflowDesigner.Dialogs.InlineDialogView
         * @param {object} options
         * @param {element} options.trigger The element under which the dialog is to appear.
         * @param {JIRA.WorkflowDesigner.WorkflowModel} options.workflowModel The application's workflow model.
         */
        initialize: function (options) {
            _.bindAll(this, "_onError", "_onSelection", "_onSuccess");

            InlineDialogView.prototype.initialize.apply(this, arguments);
            this._workflowModel = options.workflowModel;
            this._createStatus = this._workflowModel.get("permissions").get("createStatus");
        },

        /**
         * Add the selected status to the workflow.
         *
         * @private
         */
        _addExistingStatus: function () {
            var options,
                triggerDone = _.bind(this.trigger, this, "done");

            this._disable();
            this._removeErrorMessages();
            this.trigger("submit");

            options = {
                createGlobalTransition: this.ui.createGlobalTransition.is(":checked"),
                statusId: this._selectedOption.value(),
                workflowName: this._workflowModel.get("name")
            };

            this._request = WorkflowStatusesAJAXManager.addStatus(options);
            this._request.always(triggerDone).done(this._onSuccess).fail(this._onError);
        },

        /**
         * Clear and hide the dialog.
         *
         * @param {object} [e] A click event.
         * @private
         */
        _cancel: function (e) {
            e && e.preventDefault();
            this._clearSelection();
            this._createGlobalTransition = false;
            this.hide();
        },

        /**
         * Clear the currently selected option and disable the submit button.
         *
         * @private
         */
        _clearSelection: function () {
            this._selectedOption = undefined;
            this.ui.submit.attr("disabled", "");
        },

        /**
         * Determine if the inline dialog should hide in response to a click.
         *
         * @param {object} e A click event.
         * @private
         * @returns {boolean} Whether the inline dialog should hide.
         */
        _clickShouldHide: function (e) {
            var shouldHide = InlineDialogView.prototype._clickShouldHide.apply(this, arguments);
            return !this._isStatusNameDropdown(e.target) && shouldHide;
        },

        /**
         * Show a create status dialog, pre-populated with the selected name.
         *
         * @private
         */
        _createNewStatus: function () {
            var options;

            options = {
                createGlobalTransition: this.ui.createGlobalTransition.is(":checked"),
                statusModel: new StatusModel({
                    name: this._selectedOption.fieldText()
                }),
                workflowModel: this._workflowModel
            };

            new CreateStatusDialogView(options).show();
            this._cancel();
        },

        /**
         * Disable all buttons/inputs in the inline dialog.
         *
         * @private
         */
        _disable: function () {
            this.$(":input").attr("disabled", "disabled");
        },

        /**
         * Enable all buttons/inputs in the inline dialog.
         *
         * @private
         */
        _enable: function () {
            this.$(":input").removeAttr("disabled");
        },

        /**
         * @param {element} element The inline dialog element.
         * @param {function} showDialog A function that shows the inline dialog.
         * @private
         */
        _getContent: function (element, showDialog) {
            var instance = this,
                request = WorkflowStatusesAJAXManager.getStatuses({workflowName: this._workflowModel.get("name")});

            request.done(function (statuses) {
                var content;

                content = instance.template({
                    createGlobalTransition: instance._createGlobalTransition,
                    statusOptions: instance._getStatusOptions(statuses),
                    createStatus: instance._createStatus
                });

                element.html(content);
                instance._statuses = statuses;
                showDialog();
            });

            request.fail(function (errorMessage) {
                Messages.showErrorMessage(errorMessage);
                instance._reset();
            });
        },

        /**
         * Extract valid status options.
         *
         * A status is valid if it isn't already in the workflow.
         *
         * @param {object[]} statuses The status data to consider.
         * @private
         * @returns {object[]} Status options to be passed to the template.
         */
        _getStatusOptions: function (statuses) {
            var statusIds;

            function getLowerCaseName(status) {
                return status.name.toLowerCase();
            }

            function inWorkflow(status) {
                return _.contains(statusIds, status.id);
            }

            function makeOption(status) {
                return {
                    text: status.name,
                    value: status.id
                };
            }

            statusIds = this._workflowModel.get("statuses").pluck("statusId");

            return _.chain(statuses)
                    .reject(inWorkflow)
                    .sortBy(getLowerCaseName)
                    .map(makeOption)
                    .value();
        },

        /**
         * @param {element} element An element.
         * @private
         * @returns {boolean} Whether `element` is the status name dropdown.
         */
        _isStatusNameDropdown: function (element) {
            var dropdownController = this._statusName.dropdownController,
                layer = dropdownController && dropdownController.layer()[0];

            return !!jQuery(element).closest(layer).length;
        },

        /**
         * Initialise the status name single select before the dialog is shown.
         *
         * @private
         */
        _onBeforeShow: function () {
            // AJS.SingleSelect triggers a "selected" event on initialisation;
            // store the previously selected option here so we can restore it.
            var selectedOption = this._selectedOption;

            this._statusName = new SingleSelect({
                element: this.ui.statusName,
                errorMessage: "",
                statuses: this._statuses,
                suggestionsHandler: AddStatusSuggestionHandler
            });

            selectedOption && this._statusName.setSelection(selectedOption);
        },

        /**
         * Show an error message in the inline dialog.
         *
         * Called when adding a status to the workflow fails.
         *
         * @param {string} message A message describing the error.
         * @private
         */
        _onError: function (message) {
            this._enable();
            message && this.ui.form.prepend(aui.message.error({
                content: message
            }));
        },

        /**
         * Store the selected status.
         *
         * Called when a status is selected.
         *
         * @param {object} e The selection event.
         * @param {AJS.ItemDescriptor} descriptor The option that was selected.
         * @private
         */
        _onSelection: function (e, descriptor) {
            this._selectedOption = descriptor;
            this.ui.submit.removeAttr("disabled");
        },

        /**
         * Focus the status name single select after the dialog has been shown.
         *
         * @private
         */
        _onShow: function () {
            this.$("#status-name-field").focus();
        },

        /**
         * Hide the dialog and reset the `WorkflowModel`.
         *
         * @param {object} layoutData The workflow's updated layout data.
         * @private
         */
        _onSuccess: function (layoutData) {
            var duration = new Date() - this._workflowModel.get("loadedAt");

            Analytics.triggerAddStep(this._workflowModel.get("permissions"));
            Analytics.triggerFirstAddStatus(duration, this._workflowModel.get("permissions"));
            WorkflowStatusesAJAXManager.reset();
            this._clearSelection();
            this._createGlobalTransition = false;
            this.hide();
            this._workflowModel.reset(layoutData);
        },

        /**
         * Remove all error messages from the inline dialog.
         *
         * @private
         */
        _removeErrorMessages: function () {
            this.$(".aui-message.error").remove();
        },

        /**
         * @private
         * @returns {boolean} Whether the inline dialog should hide.
         */
        _shouldHide: function () {
            // Don't hide if we're submitting.
            return !(this._request && this._request.state() === "pending");
        },

        /**
         * Store the state of the create global transition checkbox.
         *
         * @private
         */
        _storeCreateGlobalTransition: function () {
            this._createGlobalTransition = this.ui.createGlobalTransition.is(":checked");
        },

        /**
         * Submit the form, adding a status to the workflow.
         *
         * @param {object} e The submit event.
         * @private
         */
        _submit: function (e) {
            var hasId = !!(this._selectedOption && this._selectedOption.value());
            var hasStatus = !!this._selectedOption;

            e.preventDefault();

            if (hasId) {
                this._addExistingStatus();
            } else if (hasStatus && this._createStatus) {
                this._createNewStatus();
            }
        }
    });
});

AJS.namespace("JIRA.WorkflowDesigner.Dialogs.AddStatusInlineDialogView", null, require("workflow-designer/dialogs/add-status-inline-dialog-view"));