AJS.test.require([
    "com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer",
    "com.atlassian.jira.plugins.jira-workflow-designer:test-resources"
], function () {

    var AddStatusInlineDialogView; // required in setup and mockDialogView
    var ContentRetriever = require("jira/ajs/contentretriever/content-retriever");
    var WorkflowStatusesAJAXManager = require("workflow-designer/io/ajax/workflow-statuses-ajax-manager");
    var WorkflowModel = require("workflow-designer/workflow-model");
    var TestUtilities = require("workflow-designer/test-utilities");
    var Messages = require("workflow-designer/messages");
    var Analytics = require("workflow-designer/analytics");
    var ItemDescriptor = require("jira/ajs/list/item-descriptor");
    var _ = require("workflow-designer/underscore");
    var jQuery = require("jquery");

    module("AddStatusInlineDialogView", {
        /**
         * Assert that an option has the correct text and value.
         *
         * @param {element} option An `option` element.
         * @param {string} text The option's expected text.
         * @param {string} value The options' expected value.
         */
        assertOption: function (option, text, value) {
            option = jQuery(option);
            equal(option.text(), text, "The option contains the correct text");
            equal(option.val(), value, "The option contains the correct value");
        },

        /**
         * Check the "create global transition" checkbox in an add status dialog.
         *
         * @param {JIRA.WorkflowDesigner.Dialogs.AddStatusInlineDialogView} dialog An add status dialog.
         */
        checkCreateGlobalTransition: function (dialog) {
            dialog.ui.createGlobalTransition.attr("checked", "");
            dialog.ui.createGlobalTransition.change();
        },

        mockStatusView: function() {
            this.context.mock("jira/ajs/contentretriever/content-retriever", ContentRetriever);
            this.context.mock("workflow-designer/io/ajax/workflow-statuses-ajax-manager", WorkflowStatusesAJAXManager);
            AddStatusInlineDialogView = this.context.require("workflow-designer/dialogs/add-status-inline-dialog-view");
        },

        /**
         * @param {object} [options] Options to pass to the dialog's constructor.
         * @returns {JIRA.WorkflowDesigner.Dialogs.AddStatusInlineDialogView} An add status dialog.
         */
        createDialog: function (options) {
            options = _.defaults({}, options, {
                trigger: jQuery("<div>"),
                workflowModel: this.workflowModel
            });

            return new AddStatusInlineDialogView(options);
        },

        /**
         * Default WorkflowModel used in tests
         * @returns {JIRA.WorkflowDesigner.WorkflowModel}
         */
        createWorkflowModel: function(createStatus) {
            var model = new WorkflowModel();
            model.get("permissions").set("createStatus", _.isUndefined(createStatus) ? true : createStatus);
            return model;
        },

        /**
         * Select a status in a dialog.
         *
         * @param {JIRA.WorkflowDesigner.Dialogs.AddStatusInlineDialogView} dialog An add status dialog.
         * @param {string} statusId The ID of the status to select.
         * @param {string} statusName The name of the status to select.
         */
        selectStatus: function (dialog, statusId, statusName) {
            dialog.ui.statusName.trigger("selected", new ItemDescriptor({
                fieldText: statusName,
                label: statusName,
                value: statusId
            }));
        },

        setup: function () {
            var getStatusesDeferred = jQuery.Deferred(),
                sandbox = this.sandbox = sinon.sandbox.create();

            this.context = AJS.test.mockableModuleContext();
            AddStatusInlineDialogView = require("workflow-designer/dialogs/add-status-inline-dialog-view");

            getStatusesDeferred.resolve([{
                id: "closed",
                name: "Closed"
            }, {
                id: "open",
                name: "Open"
            }]);

            this.addStatusDeferred = jQuery.Deferred();
            this.addStatusStub = sandbox.stub(WorkflowStatusesAJAXManager, "addStatus").returns(this.addStatusDeferred);
            this.getStatusesStub = sandbox.stub(WorkflowStatusesAJAXManager, "getStatuses").returns(getStatusesDeferred);
            this.workflowModel = this.createWorkflowModel();
        },

        teardown: function () {
            TestUtilities.removeDialogs();
            this.sandbox.restore();
        }
    });

    asyncTest("A CreateStatusDialogView is shown if the new status option was selected", function () {
        var createStatusDialog = {show: sinon.spy()},
            createStatusDialogStub = this.sandbox.stub().returns(createStatusDialog),
            dialog,
            instance = this,
            workflowModel = this.createWorkflowModel();

        this.context.mock("workflow-designer/dialogs/create-status-dialog-view", createStatusDialogStub);
        this.mockStatusView();
        dialog = this.createDialog({workflowModel: workflowModel});
        this.sandbox.spy(dialog, "_cancel");

        dialog.show().done(function () {
            instance.selectStatus(dialog, "", "New Status");
            dialog.$(":submit").click();

            equal(createStatusDialogStub.callCount, 1, "A CreateStatusDialogView was created");
            equal(createStatusDialogStub.args[0][0].statusModel.get("name"), "New Status", "It was passed a StatusModel with the correct name");
            ok(createStatusDialogStub.args[0][0].workflowModel === workflowModel, "It was passed the correct WorkflowModel");
            equal(createStatusDialog.show.callCount, 1, "It was shown");
            equal(dialog._cancel.callCount, 1, "The AddStatusInlineDialogView was hidden and reset");

            start();
        });
    });

    asyncTest("A CreateStatusDialogView is not shown if the new status option was selected", function () {
        var createStatusDialog = {show: sinon.spy()},
            createStatusDialogStub = this.sandbox.stub().returns(createStatusDialog),
            dialog,
            instance = this,
            workflowModel = this.createWorkflowModel();

        this.context.mock("workflow-designer/dialogs/create-status-dialog-view", createStatusDialogStub);
        this.mockStatusView();
        dialog = this.createDialog({workflowModel: workflowModel});
        this.sandbox.spy(dialog, "_cancel");

        dialog.show().done(function () {
            instance.selectStatus(dialog, "", "New Status");
            dialog.$(":submit").click();

            equal(createStatusDialogStub.callCount, 1, "A CreateStatusDialogView was created");
            equal(createStatusDialogStub.args[0][0].statusModel.get("name"), "New Status", "It was passed a StatusModel with the correct name");
            ok(createStatusDialogStub.args[0][0].workflowModel === workflowModel, "It was passed the correct WorkflowModel");
            equal(createStatusDialog.show.callCount, 1, "It was shown");
            equal(dialog._cancel.callCount, 1, "The AddStatusInlineDialogView was hidden and reset");

            start();
        });
    });

    asyncTest("An error message is shown if adding a status fails", function () {
        var dialog = this.createDialog(),
            doneSpy = sinon.spy(),
            instance = this;

        dialog.on("done", doneSpy);
        dialog.show().done(function () {
            var deferred = jQuery.Deferred(),
                errorElement,
                errorMessage = "No status for you!";

            dialog.$(":submit").click();
            instance.addStatusDeferred.reject(errorMessage);
            errorElement = dialog.$(".aui-message.error");
            equal(doneSpy.callCount, 1, "A done event was triggered");
            equal(errorElement.length, 1, "An error message is shown in the inline dialog");
            equal(errorElement.text(), errorMessage, "It contains the correct message");
            equal(dialog.$(":disabled").length, 0, "No inputs are disabled");

            instance.addStatusStub.returns(deferred);
            dialog.$(":submit").click();
            equal(dialog.$(".aui-message.error").length, 0, "Error messages are removed while submitting");

            // Resolve the deferred so the dialog can close.
            deferred.resolve({});
            start();
        });
    });

    test("An error message is shown if statuses fail to load", function () {
        var dialog = this.createDialog(),
            errorMessage = "No statuses for you!",
            showErrorMessageStub = this.sandbox.stub(Messages, "showErrorMessage");

        this.getStatusesStub.returns(jQuery.Deferred().reject(errorMessage));

        dialog.show();
        equal(showErrorMessageStub.callCount, 1, "An error message was shown");
        equal(showErrorMessageStub.args[0][0], errorMessage, "The correct error message was shown");
    });

    asyncTest("Clicking a status name suggestion doesn't close the dialog", function () {
        var dialog = this.createDialog();
        this.sandbox.spy(dialog, "hide");

        dialog.show().done(function () {
            dialog._statusName._handleCharacterInput(true);
            dialog._statusName.dropdownController.layer().click();
            equal(dialog.hide.callCount, 0, "The dialog wasn't hidden");

            start();
        });
    });

    asyncTest("Clicking cancel closes and resets the dialog", function () {
        var dialog = this.createDialog(),
            instance = this;

        this.sandbox.spy(dialog, "hide");

        dialog.show().done(function () {
            instance.checkCreateGlobalTransition(dialog);
            instance.selectStatus(dialog, "open", "Open");
            dialog.$(".cancel").click();
            equal(dialog.hide.callCount, 1, "The dialog was hidden");

            dialog.show().done(function () {
                equal(dialog.$(":selected").val(), "closed", "The selected status was cleared");
                ok(!dialog.ui.createGlobalTransition.is(":checked"), "The create global transition checkbox was unchecked");

                start();
            });
        });
    });

    asyncTest("Clicking the document doesn't close the dialog while it's submitting", function () {
        var dialog = this.createDialog(),
            instance = this,
            willClose = dialog._inlineDialog.getOptions().preHideCallback;

        dialog.show().done(function () {
            ok(willClose(), "The inline dialog will close before submission");
            dialog.$(":submit").click();
            ok(!willClose(), "The inline dialog won't close while submitting");

            // Resolve the deferred so the dialog can close.
            instance.addStatusDeferred.resolve({});

            start();
        });
    });

    asyncTest("No error appears if the message is empty", function () {
        var dialog = this.createDialog();

        this.addStatusDeferred.reject();

        dialog.show().done(function () {
            dialog.ui.submit.click();
            equal(dialog.$(".aui-message.error").length, 0, "No error message appeared");

            start();
        });
    });

    asyncTest("Submission is ignored if no status is selected", function () {
        var createStatusDialogSpy = this.sandbox.spy();
        this.context.mock("workflow-designer/dialogs/create-status-dialog-view", createStatusDialogSpy);
        this.mockStatusView();
        var dialog = this.createDialog(),
            instance = this;

        this.getStatusesStub.returns(jQuery.Deferred().resolve([]));

        dialog.show().done(function () {
            dialog.ui.form.submit();
            equal(createStatusDialogSpy.callCount, 0, "No CreateStatusDialogView was created");
            equal(instance.addStatusStub.callCount, 0, "WorkflowStatusesAJAXManager.addStatus wasn't called");

            start();
        });
    });

    asyncTest("Submitting the form adds a status to the workflow", function () {
        var dialog,
            doneSpy = sinon.spy(),
            expectedArguments,
            instance = this,
            layoutData = {},
            resetWorkflowStatusesAJAXManagerSpy = this.sandbox.spy(WorkflowStatusesAJAXManager, "reset"),
            submitSpy = sinon.spy(),
            workflowModel;

        workflowModel = new WorkflowModel({name: "Workflow"});
        this.sandbox.stub(workflowModel, "reset");

        dialog = this.createDialog({workflowModel: workflowModel});
        dialog.on({
            done: doneSpy,
            submit: submitSpy
        });
        this.sandbox.spy(dialog, "hide");

        expectedArguments = [{
            createGlobalTransition: false,
            statusId: "closed",
            workflowName: "Workflow"
        }];

        dialog.show().done(function () {
            var inputs;

            dialog.$(":submit").click();
            inputs = dialog.$(":input:visible");

            equal(inputs.filter(":disabled").length, inputs.length, "All inputs are disabled");
            equal(instance.addStatusStub.callCount, 1, "WorkflowStatusesAJAXManager.addStatus was called");
            deepEqual(instance.addStatusStub.args[0], expectedArguments, "It was passed the correct options");
            equal(submitSpy.callCount, 1, "A submit event was triggered");

            instance.addStatusDeferred.resolve(layoutData);
            equal(dialog.hide.callCount, 1, "The dialog was hidden");
            equal(doneSpy.callCount, 1, "A done event was triggered");
            equal(workflowModel.reset.callCount, 1, "The WorkflowModel was reset");
            ok(workflowModel.reset.args[0][0] === layoutData, "It was passed the correct layout data");
            equal(resetWorkflowStatusesAJAXManagerSpy.callCount, 1, "WorkflowStatusesAJAXManager was reset");

            start();
        });
    });

    asyncTest("The \"create global transition\" checkbox is respected", function () {
        var dialog,
            expectedArguments,
            instance = this,
            workflowModel = new WorkflowModel({name: "Workflow"});

        dialog = this.createDialog({
            workflowModel: workflowModel
        });

        expectedArguments = [{
            createGlobalTransition: true,
            statusId: "closed",
            workflowName: "Workflow"
        }];

        dialog.show().done(function () {
            dialog.ui.createGlobalTransition.attr("checked", "");
            dialog.ui.submit.click();

            equal(instance.addStatusStub.callCount, 1, "WorkflowStatusesAJAXManager.addStatus was called");
            deepEqual(instance.addStatusStub.args[0], expectedArguments, "It was passed the correct options");

            // Resolve the deferred so the dialog can close.
            instance.addStatusDeferred.resolve({});

            start();
        });
    });

    asyncTest("The \"create global transition\" checkbox state is passed to the create status dialog", function () {
        var createStatusDialogViewSpy = this.sandbox.spy(require("workflow-designer/dialogs/create-status-dialog-view"));
        this.context.mock("workflow-designer/dialogs/create-status-dialog-view", createStatusDialogViewSpy);
        this.mockStatusView();
        var dialog = this.createDialog(),
            instance = this;

        dialog.show().done(function () {
            instance.selectStatus(dialog, "", "New Status");
            dialog.ui.createGlobalTransition.attr("checked", "");
            dialog.ui.submit.click();

            equal(createStatusDialogViewSpy.callCount, 1, "A CreateStatusDialogView was created");
            ok(createStatusDialogViewSpy.args[0][0].createGlobalTransition,
                "It was passed the state of the create global transition checkbox");

            start();
        });
    });

    asyncTest("The \"create global transition\" checkbox state is persisted between shows", function () {
        var dialog = this.createDialog(),
            instance = this;

        dialog.show().done(function () {
            instance.checkCreateGlobalTransition(dialog);
            dialog.hide();

            dialog.show().done(function () {
                ok(dialog.ui.createGlobalTransition.is(":checked"), "The state was persisted");

                start();
            });
        });
    });

    asyncTest("The dialog appears after its content fails to load", function () {
        var dialog = this.createDialog(),
            getStatusesStub = this.getStatusesStub;

        getStatusesStub.returns(jQuery.Deferred().reject());

        dialog.show().done(function () {
            ok(!dialog.$el.is(":visible"), "The dialog isn't visible");

            // Its content loads the second time, so it should show.
            getStatusesStub.returns(jQuery.Deferred().resolve([]));

            dialog.show().done(function () {
                ok(dialog.$el.is(":visible"), "The dialog is visible");

                start();
            });
        });
    });

    asyncTest("The dialog is reset after submission", function () {
        var dialog = this.createDialog(),
            instance = this;

        dialog.show().done(function () {
            instance.checkCreateGlobalTransition(dialog);
            instance.selectStatus(dialog, "open", "Open");
            dialog.ui.submit.click();
            instance.addStatusDeferred.resolve({});

            dialog.show().done(function () {
                equal(dialog.$(":selected").val(), "closed", "The selected status was cleared");
                ok(!dialog.ui.createGlobalTransition.is(":checked"), "The create global transition checkbox was unchecked");

                start();
            });
        });
    });

    asyncTest("The selected new status is persisted across shows", function () {
        var dialog = this.createDialog(),
            instance = this;

        dialog.show().done(function () {
            instance.selectStatus(dialog, "", "New Status");
            dialog.hide();

            dialog.show().done(function () {
                instance.assertOption(dialog.$(":selected"), "New Status", "");

                start();
            });
        });
    });

    asyncTest("The selected status is persisted across shows", function () {
        var dialog = this.createDialog(),
            instance = this;

        dialog.show().done(function () {
            instance.selectStatus(dialog, "open", "Open");
            dialog.hide();
            dialog.show().done(function () {
                equal(dialog.$(":selected").val(), "open", "The selected status was persisted");

                start();
            });
        });
    });

    asyncTest("The submit button is initially disabled if no existing statuses are available", function () {
        var dialog,
            instance = this;

        this.getStatusesStub.returns(jQuery.Deferred().resolve([]));
        dialog = this.createDialog();

        dialog.show().done(function () {
            equal(dialog.$(":disabled:submit").length, 1, "The submit button is disabled initially");

            instance.selectStatus(dialog, "", "New Status");
            equal(dialog.$(":disabled:submit").length, 0, "The submit button is enabled after selecting a new status");

            dialog.ui.statusName.trigger("unselect");
            equal(dialog.$(":disabled:submit").length, 1, "The submit button is disabled after clearing the selection");

            start();
        });
    });

    asyncTest("Triggers analytics events after successfully adding an existing status", function () {
        var dialog = this.createDialog(),
            triggerAddStepSpy = this.sandbox.spy(Analytics, "triggerAddStep"),
            triggerFirstAddStatusSpy = this.sandbox.spy(Analytics, "triggerFirstAddStatus");
        var workflowModel = this.workflowModel;

        this.addStatusDeferred.resolve({});

        dialog.show().done(function () {
            dialog.$(":submit").click();
            equal(triggerAddStepSpy.callCount, 1, "An add step event was triggered");
            equal(triggerFirstAddStatusSpy.callCount, 1, "A first add status event was triggered");
            sinon.assert.calledWith(triggerAddStepSpy, workflowModel.get('permissions'));
            equal(triggerFirstAddStatusSpy.getCall(0).args[1], workflowModel.get('permissions'));

            start();
        });
    });

    asyncTest("Valid statuses are listed in the dialog", function () {
        var dialog,
            getStatusesDeferred = jQuery.Deferred(),
            instance = this,
            options,
            showDeferred,
            workflowModel;

        this.getStatusesStub.returns(getStatusesDeferred);
        workflowModel = new WorkflowModel();
        workflowModel.addStatus({statusId: "in-progress"});
        workflowModel.set({name: 'workflow name'});
        dialog = this.createDialog({workflowModel: workflowModel});
        showDeferred = dialog.show();
        equal(this.getStatusesStub.callCount, 1, "WorkflowStatusesAJAXManager.getStatuses was called");
        ok(this.getStatusesStub.calledWith({workflowName: 'workflow name'}), "WorkflowStatusesAJAXManager.getStatuses was called with workflowName: 'workflow name'");

        getStatusesDeferred.resolve([{
            id: "in-progress",
            name: "In Progress"
        }, {
            id: "open",
            name: "Open"
        }, {
            id: "closed",
            name: "Closed"
        }]);

        showDeferred.done(function () {
            options = dialog.$("#status-name option");
            equal(options.length, 2, "The correct number of options are present");
            instance.assertOption(options[0], "Closed", "closed");
            instance.assertOption(options[1], "Open", "open");

            start();
        });
    });

    asyncTest("Valid info message is presented to user that can create new status", function () {
        var dialog = this.createDialog();

        this.addStatusDeferred.reject();

        dialog.show().done(function () {
            ok(dialog.$el.html().indexOf("workflow.designer.add.status.description")>=0, "Dialog should contain workflow.designer.add.status.description");

            start();
        });
    });

    asyncTest("Valid info message is presented to user that can NOT create new status", function () {
        var workflowModel = this.createWorkflowModel(false);
        var dialog = this.createDialog({workflowModel: workflowModel});

        this.addStatusDeferred.reject();

        dialog.show().done(function () {
            ok(dialog.$el.html().indexOf("workflow.designer.add.existing.status.description")>=0, "Dialog should contain workflow.designer.add.existing.status.description");

            start();
        });
    });
});
