define("workflow-designer/dialogs/inline-dialog-view", [
    "aui/inline-dialog",
    "workflow-designer/backbone",
    "workflow-designer/underscore",
    "jquery"
], function(
    InlineDialog,
    Backbone,
    _,
    jQuery
) {

    return Backbone.Marionette.ItemView.extend(
    /** @lends JIRA.WorkflowDesigner.Dialogs.InlineDialogView# */
    {
        /**
         * Initialise the view.
         *
         * @class
         * @classdesc A Backbone wrapper around `AJS.InlineDialog`.
         *
         * Subclasses must implement `_getContent()`.
         * @constructs
         * @extends Backbone.Marionette.ItemView
         * @param {object} options
         * @param {element} options.trigger The element under which the dialog is to appear.
         */
        initialize: function (options) {
            _.bindAll(this, "_onDocumentClick", "_onKeyDown", "_shouldHide");

            this._trigger = options.trigger;
            this._reset();
        },

        /**
         * Determine if the inline dialog should hide in response to a click.
         *
         * Can be overridden by subclasses.
         *
         * @param {object} e A click event.
         * @private
         * @returns {boolean} Whether the inline dialog should hide.
         */
        _clickShouldHide: function (e) {
            return !jQuery.contains(this._inlineDialog[0], e.target);
        },

        /**
         * Create the view's inline dialog.
         *
         * @param {element} trigger The element under which the dialog is to appear.
         * @private
         * @returns {AJS.InlineDialog} The view's inline dialog.
         */
        _createInlineDialog: function (trigger) {
            var inlineDialog,
                instance = this;

            function getContent(element, trigger, showDialog) {
                instance._getContent(element, showDialog);
            }

            function onBeforeShow() {
                instance.bindUIElements();
                instance._onBeforeShow();

                _.delay(function () {
                    // Cover ourselves in mud to hide from JIRA's LayerManager.js.
                    InlineDialog.current = null;
                    instance._onShow();
                    jQuery(document).on({
                        click: instance._onDocumentClick,
                        keydown: instance._onKeyDown
                    });
                }, instance._inlineDialog.getOptions().fadeTime);
            }

            function onHide() {
                jQuery(document).off({
                    click: instance._onDocumentClick,
                    keydown: instance._onKeyDown
                });
            }

            inlineDialog = new InlineDialog(trigger, this.id, getContent, {
                hideCallback: onHide,
                initCallback: onBeforeShow,
                noBind: true,
                persistent: true,
                preHideCallback: this._shouldHide
            });

            return inlineDialog;
        },

        /**
         * Hide the inline dialog.
         */
        hide: function () {
            this._inlineDialog.hide();
        },

        /**
         * Hide the inline dialog in response to a click.
         *
         * Called when the user clicks anywhere on the document.
         *
         * @private
         */
        _onDocumentClick: function (e) {
            this._clickShouldHide(e) && this.hide();
        },

        /**
         * Called before the inline dialog is shown.
         *
         * To be implemented by subclasses.
         */
        _onBeforeShow: jQuery.noop,

        /**
         * Hide the inline dialog in response to pressing escape.
         *
         * @param {object} e The key down event.
         * @private
         */
        _onKeyDown: function (e) {
            var isEscape = e.which === jQuery.ui.keyCode.ESCAPE,
                isInput = jQuery(e.target).is(":input");

            if (isEscape && !isInput) {
                this.hide();
            }
        },

        /**
         * Called after the inline dialog is shown.
         *
         * To be implemented by subclasses.
         */
        _onShow: jQuery.noop,

        /**
         * Remove the view.
         */
        remove: function () {
            Backbone.Marionette.ItemView.prototype.remove.apply(this, arguments);
            this.hide();
            this._inlineDialog.remove();
        },

        /**
         * Recreate the view's inline dialog to force content re-loading.
         *
         * This is used to work around a bug in `AJS.InlineDialog` where it
         * won't show if loading its content fails. Recreate it to reset state.
         *
         * @protected
         */
        _reset: function () {
            this._inlineDialog = this._createInlineDialog(this._trigger);
            this.setElement(this._inlineDialog);
        },

        /**
         * To be overridden by subclasses.
         *
         * @returns {boolean} Whether the inline dialog should hide.
         * @private
         */
        _shouldHide: function () {
            return true;
        },

        /**
         * Show the inline dialog.
         */
        show: function () {
            var deferred = jQuery.Deferred();

            this._inlineDialog.show();
            _.delay(deferred.resolve, InlineDialog.opts.fadeTime);

            return deferred.promise();
        }
    });
});

AJS.namespace("JIRA.WorkflowDesigner.Dialogs.InlineDialogView", null, require("workflow-designer/dialogs/inline-dialog-view"));