AJS.test.require([
    "com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer",
    "com.atlassian.jira.plugins.jira-workflow-designer:test-resources"
], function () {

    var TransitionView = require("workflow-designer/transition-view");
    var CanvasModel = require("workflow-designer/canvas-model");
    var StatusView = require("workflow-designer/status-view");
    var WorkflowModel = require("workflow-designer/workflow-model");
    var WorkflowTransitionsAJAXManager = require("workflow-designer/io/ajax/workflow-transitions-ajax-manager");
    var EditTransitionTargetDialogView = require("workflow-designer/dialogs/edit-transition-target-dialog-view");
    var JIRADialog = require("jira/dialog/dialog");
    var TestUtilities = require("workflow-designer/test-utilities");
    var jQuery = require("jquery");

    module("EditTransitionTargetDialogView", {
        setup: function () {
            var canvas = TestUtilities.testDraw2DCanvas(),
                sourceView, targetView;

            this.sandbox = sinon.sandbox.create();

            this.updateTransitionTargetStub = this.sandbox.stub(WorkflowTransitionsAJAXManager, "updateTransitionTarget");
            this.workflowModel = new WorkflowModel({name: "Workflow Name"});
            this.workflowModel.updateTransitionTargets = this.sandbox.spy();

            sourceView = new StatusView({
                canvas: canvas,
                model: this.workflowModel.addStatus({stepId: 1, name: "Source"}),
                workflowModel: this.workflowModel
            }).render();

            targetView = new StatusView({
                canvas: canvas,
                model: this.workflowModel.addStatus({stepId: 2, name: "Target"}),
                workflowModel: this.workflowModel
            }).render();

            this.transitionModel = this.workflowModel.addTransition({
                actionId: 1,
                source: sourceView.model,
                target: targetView.model,
                targetAngle: 90
            });

            this.transitionView = new TransitionView({
                canvas: canvas,
                canvasModel: new CanvasModel({}, {
                    workflowModel: this.workflowModel
                }),
                model: this.transitionModel,
                sourceView: sourceView,
                targetView: targetView,
                workflowModel: this.workflowModel
            }).render();

            this.transitionView.resetConnection = this.sandbox.spy();

            this.newTargetView = new StatusView({
                canvas: canvas,
                model: this.workflowModel.addStatus({stepId: 3, name: "New target"}),
                workflowModel: this.workflowModel
            }).render();

            this.targetPort = this.newTargetView.getPorts()[0];
        },

        /**
         * Create and show a <tt>EditTransitionTargetDialogView</tt>.
         *
         * @return {object} The dialog's element and deferred result.
         */
        showDialog: function () {
            new EditTransitionTargetDialogView({
                targetPort: this.targetPort,
                targetView: this.newTargetView,
                transitionView: this.transitionView,
                workflowModel: this.workflowModel
            }).show();

            return JIRADialog.current && JIRADialog.current.get$popup();
        },

        teardown: function () {
            TestUtilities.removeDialogs();
            this.sandbox.restore();
        }
    });

    test("Editing transition target", function () {
        var deferred = jQuery.Deferred(),
            element = this.showDialog(),
            targetAngle = this.newTargetView.getAngleToPort(this.targetPort);

        ok(TestUtilities.dialogIsVisible(), "The dialog is visible");
        ok(element.find(":submit").length, "The dialog has a submit button");

        this.updateTransitionTargetStub.returns(deferred.promise());

        TestUtilities.submit(element);

        equal(element.find(":input:disabled").length, element.find(":input").length, "All inputs are disabled while submitting");
        equal(this.updateTransitionTargetStub.callCount, 1, "WorkflowTransitionsAJAXManager.updateTransitionTarget() was called");

        deferred.resolve();

        equal(this.workflowModel.updateTransitionTargets.callCount, 1, "WorkflowModel.updateTransitionTargets was called");
        deepEqual(this.workflowModel.updateTransitionTargets.args[0][0], {
            transition: this.transitionModel,
            targetAngle: targetAngle,
            targetStepId: 3
        }, "It was passed the correct arguments");

        ok(!TestUtilities.dialogIsVisible(), "The dialog closed");
    });

    test("An error message is shown if updating the transition target fails", function () {
        var element = this.showDialog(),
            errorElement,
            errorMessage = "Oh noes!";

        this.updateTransitionTargetStub.returns(jQuery.Deferred().reject(errorMessage).promise());
        TestUtilities.submit(element);

        errorElement = element.find(".aui-message.error");
        equal(errorElement.length, 1, "An error message is shown in the dialog");
        ok(errorElement.text().indexOf(errorMessage) > -1, "It contains the correct text");
        equal(element.find(":input:disabled").length, 0, "No elements in the dialog are disabled");

        this.updateTransitionTargetStub.returns(jQuery.Deferred().promise());
        TestUtilities.submit(element);
        equal(element.find(".error").length, 0, "Errors are removed on submit");
    });

    test("Closing the dialog resets transition target", function () {
        var element = this.showDialog();

        ok(TestUtilities.dialogIsVisible(), "The dialog is visible");

        element.find(".cancel").click();

        equal(this.transitionView.resetConnection.callCount, 1, "Transition target was reset");
        ok(!TestUtilities.dialogIsVisible(), "The dialog closed");
    });

});
