define("workflow-designer/dialogs/edit-transition-dialog-view", [
    "workflow-designer/dialogs/transition-dialog-view",
    "workflow-designer/io/ajax/workflow-transitions-ajax-manager",
    "workflow-designer/analytics",
    "jira/ajs/select/single-select",
    "workflow-designer/templates",
    "workflow-designer/backbone",
    "workflow-designer/underscore"
], function(
    TransitionDialogView,
    WorkflowTransitionsAJAXManager,
    Analytics,
    SingleSelect,
    Templates,
    Backbone,
    _
) {
    /**
     * @class JIRA.WorkflowDesigner.Dialogs.EditTransitionDialogView
     * @classdesc The workflow designer's "Edit Transition" dialog.
     * @extends JIRA.WorkflowDesigner.Dialogs.TransitionDialogView
     */
    return TransitionDialogView.extend(
    /** @lends JIRA.WorkflowDesigner.Dialogs.EditTransitionDialogView# */
    {
        id: "edit-transition-dialog",

        template: Templates.editTransitionDialog,

        ui: {
            description: "#transition-description",
            name: "#transition-name",
            screenId: "#transition-screen-id"
        },

        /**
         * @returns {function} A function that retrieves the dialog's content.
         * @private
         */
        _getContent: function () {
            var instance = this;

            return function (callback) {
                var data = instance._transitionModel.toJSON(),
                    request = instance._getScreenOptions();

                request.done(function (screenOptions) {
                    _.extend(data, {
                        screens: screenOptions,
                        screensPermission: instance._hasScreenPermissions()
                    });
                    callback(instance.template(data));
                });

                request.fail(instance._onLoadError);
            };
        },

        /**
         * @returns {Number} The ID of the selected screen.
         * @private
         */
        _getSelectedScreenId: function () {
            return Number(this.ui.screenId.find(":selected").val() || 0);
        },

        /**
         * @returns {string} The name of the selected screen.
         * @private
         */
        _getSelectedScreenName: function () {
            return this.ui.screenId.find(":selected").text();
        },

        /**
         * Make a request to update the transition.
         *
         * @return {jQuery.Deferred} A deferred that is resolved on success.
         * @private
         */
        _makeRequest: function (options) {
            return WorkflowTransitionsAJAXManager.updateTransition(options);
        },

        /**
         * @private
         */
        _onContentRefresh: function () {
            new SingleSelect({
                element: this.ui.screenId,
                revertOnInvalid: true
            });
        },

        /**
         * @param {jQuery.Event} e The submit event.
         * @param {function} callback A function to call when submission is complete.
         * @private
         */
        _onSubmit: function (e, callback) {
            var options, request;

            options = {
                description: this.ui.description.val(),
                name: this.ui.name.val(),
                screenId: this._getSelectedScreenId(),
                sourceStepId: this._transitionModel.getSourceStepId(),
                transitionId: this._transitionModel.getActionId(),
                workflowName: this._workflowModel.get("name")
            };

            TransitionDialogView.prototype._onSubmit.apply(this, arguments);
            request = this._makeRequest(options);
            request.done(this._onSuccess).fail(callback).fail(this._onError);
        },

        /**
         * @private
         */
        _onSuccess: function() {
            this.hide();
            this._updateTransitionModels({
                description : this.ui.description.val(),
                name : this.ui.name.val(),
                screenId : this._getSelectedScreenId(),
                screenName : this._getSelectedScreenName()
            });
            this._triggerAnalyticsEvent();
        },

        /**
         * Trigger an analytics event indicating that the update was successful.
         *
         * @protected
         */
        _triggerAnalyticsEvent: function () {
            Analytics.triggerUpdateTransition(this._workflowModel.get("permissions"));
        },

        /**
         * Update the transitions models affected after the save of the new transition data.
         * Note, this gets all transitions matching the actionId of the transition being updated, this
         * is to ensure that common transitions in the view are updated also.
         *
         * @param {object} attributes The attributes to update the transition model with
         * @private
         */
        _updateTransitionModels: function(attributes) {
            var actionId = this._transitionModel.get("actionId"),
                transitionModels = this._workflowModel.get("transitions");

            function hasActionId(transitionModel) {
                return transitionModel.get("actionId") === actionId;
            }

            transitionModels.chain().filter(hasActionId).invoke("set", attributes);
        }
    });
});

AJS.namespace("JIRA.WorkflowDesigner.Dialogs.EditTransitionDialogView", null, require("workflow-designer/dialogs/edit-transition-dialog-view"));