AJS.test.require([
    "com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer",
    "com.atlassian.jira.plugins.jira-workflow-designer:test-resources"
], function () {

    var jQuery = require("jquery");
    var ScreensAJAXManager = require("workflow-designer/io/ajax/screens-ajax-manager");
    var WorkflowTransitionsAJAXManager = require("workflow-designer/io/ajax/workflow-transitions-ajax-manager");
    var WorkflowModel = require("workflow-designer/workflow-model");
    var EditTransitionDialogView = require("workflow-designer/dialogs/edit-transition-dialog-view");
    var Analytics = require("workflow-designer/analytics");
    var JIRADialog = require("jira/dialog/dialog");
    var TestUtilities = require("workflow-designer/test-utilities");
    var PermissionModel = require("workflow-designer/permissions-model");

    module("EditTransitionDialogView", {
        setup: function () {
            this.sandbox = sinon.sandbox.create();

            this.getScreensStub = this.sandbox.stub(ScreensAJAXManager, "getScreens");
            this.getScreensStub.returns(jQuery.Deferred().resolve([{
                id: 1,
                name: "Screen"
            }]).promise());

            this.updateTransitionStub = this.sandbox.stub(WorkflowTransitionsAJAXManager, "updateTransition");
            this.workflowModel = new WorkflowModel({
                name: "Workflow Name",
                permissions: new PermissionModel({
                    editPropertyOptions: true,
                    selectScreenOnTransition: true
                })
            });
        },

        /**
         * Create and show an <tt>EditTransitionDialogView</tt>.
         *
         * @return {object} The dialog's element and deferred result.
         */
        showDialog: function () {
            this.transitionModel = this.workflowModel.addTransition({
                actionId: "1",
                description: "This is a description.",
                name: "Transition",
                source: this.workflowModel.addStatus({id: 100, stepId: 1}),
                target: this.workflowModel.addStatus({id: 200, stepId: 2})
            });

            new EditTransitionDialogView({
                transitionModel: this.transitionModel,
                workflowModel: this.workflowModel
            }).show();

            return JIRADialog.current && JIRADialog.current.get$popup();
        },

        teardown: function () {
            TestUtilities.removeDialogs();
            this.sandbox.restore();
        }
    });

    test("An error message is shown if editing the transition fails", function () {
        var element = this.showDialog(),
            errorElement,
            errorMessage = "Oh noes!";

        this.updateTransitionStub.returns(jQuery.Deferred().reject(errorMessage).promise());
        TestUtilities.submit(element);

        errorElement = element.find(".aui-message.error");
        equal(errorElement.length, 1, "An error message is shown in the dialog");
        ok(errorElement.text().indexOf(errorMessage) > -1, "It contains the correct text");
        equal(element.find(":input:disabled").length, 0, "No elements in the dialog are disabled");

        this.updateTransitionStub.returns(jQuery.Deferred().promise());
        TestUtilities.submit(element);
        equal(element.find(".error").length, 0, "Errors are removed on submit");
    });

    test("Doesn't contain source or target fields", function () {
        var element = this.showDialog();

        equal(element.find("#transition-source-step-id").length, 0, "The dialog doesn't contain a source field");
        equal(element.find("#transition-target-step-id").length, 0, "The dialog doesn't contain a target field");
    });

    test("Editing a single transition", function () {
        var attributes,
            deferred = jQuery.Deferred(),
            element = this.showDialog(),
            expectedArgumentsToAjaxManager,
            expectedAttributes;

        this.updateTransitionStub.returns(deferred.promise());
        this.sandbox.stub(Analytics, "triggerUpdateTransition");

        TestUtilities.submit(element, {
            "#transition-description": "Awesome new description!",
            "#transition-name": "New Name"
        });

        equal(element.find(":input:disabled").length, element.find(":input").length, "All inputs are disabled while submitting");
        equal(this.updateTransitionStub.callCount, 1, "TransitionsAJAXManager.updateTransition() was called");

        deferred.resolve(100);

        attributes = this.workflowModel.get("transitions").at(0).attributes;
        expectedAttributes = {
            description: "Awesome new description!",
            name: "New Name"
        };
        deepEqual(_.pick(attributes, _.keys(expectedAttributes)), expectedAttributes,
            "The TransitionModel's attributes were updated");

        expectedArgumentsToAjaxManager = {
            description: "Awesome new description!",
            name: "New Name",
            screenId: 1,
            sourceStepId: 1,
            transitionId: "1",
            workflowName: "Workflow Name"
        };

        deepEqual(this.updateTransitionStub.args[0][0], expectedArgumentsToAjaxManager,
            "The correct arguments were passed to the AJAX Manager");

        ok(Analytics.triggerUpdateTransition.calledOnce, "Analytics event for update/transition has been triggered");
        sinon.assert.calledWith(Analytics.triggerUpdateTransition, this.workflowModel.get('permissions'));
    });

    test("Editing common transitions", function () {
        var attributes,
            deferred = jQuery.Deferred(),
            element = this.showDialog(),
            expectedArgumentsToAjaxManager,
            expectedAttributes;

        // Add a common transition.
        this.workflowModel.addTransition({
            actionId: "1",
            description: "This is a description.",
            name: "Transition",
            source: this.workflowModel.addStatus({id: 300, stepId: 3}),
            target: this.workflowModel.addStatus({id: 200, stepId: 2})
        });

        this.updateTransitionStub.returns(deferred.promise());
        this.sandbox.stub(Analytics, "triggerUpdateTransition");

        TestUtilities.submit(element, {
            "#transition-description": "Awesome new description!",
            "#transition-name": "New Name"
        });

        equal(element.find(":input:disabled").length, element.find(":input").length, "All inputs are disabled while submitting");
        equal(this.updateTransitionStub.callCount, 1, "TransitionsAJAXManager.updateTransition() was called");

        deferred.resolve(100);

        // Check the first transition.
        attributes = this.workflowModel.get("transitions").at(0).attributes;
        expectedAttributes = {
            description: "Awesome new description!",
            name: "New Name"
        };
        deepEqual(_.pick(attributes, _.keys(expectedAttributes)), expectedAttributes,
            "The TransitionModel's attributes were updated");

        // Check the second (common) transition.
        attributes = this.workflowModel.get("transitions").at(1).attributes;
        expectedAttributes = {
            description: "Awesome new description!",
            name: "New Name"
        };
        deepEqual(_.pick(attributes, _.keys(expectedAttributes)), expectedAttributes,
            "The common TransitionModel's attributes were updated");

        expectedArgumentsToAjaxManager = {
            description: "Awesome new description!",
            name: "New Name",
            screenId: 1,
            sourceStepId: 1,
            transitionId: "1",
            workflowName: "Workflow Name"
        };

        deepEqual(this.updateTransitionStub.args[0][0], expectedArgumentsToAjaxManager,
            "The correct arguments were passed to the AJAX Manager");

        ok(Analytics.triggerUpdateTransition.calledOnce, "Analytics event for update/transition has been triggered");
    });

    test("Fields are populated with the transition's attributes", function () {
        var element = this.showDialog();

        equal(element.find("#transition-description").val(), this.transitionModel.get("description"), "The transition's description was used");
        equal(element.find("#transition-name").val(), this.transitionModel.get("name"), "The transition's name was used");
    });

    test("When no screen permission no rest call is made to screen rest endpoint", function () {
        this.workflowModel.get("permissions").set({selectScreenOnTransition: false});
        this.showDialog();

        sinon.assert.notCalled(this.getScreensStub, "No rest call is made for screens");
    });

    test("When there is screen permission rest call is made to screen rest endpoint", function () {
        this.showDialog();

        sinon.assert.calledOnce(this.getScreensStub, "Rest call is made for screens");
    });

    test("When no screen permission no screen select should be present", function () {
        this.workflowModel.get("permissions").set({selectScreenOnTransition: false});

        var $dialog = this.showDialog();

        equal($dialog.find('#transition-screen-id').length, 0, "No screen permission select should be present");
    });

    test("When there is screen permission screen select should be present", function () {
        var $dialog = this.showDialog();

        equal($dialog.find('#transition-screen-id').length, 1, "Screen permission select should be present");
    });

});
