define("workflow-designer/dialogs/delete-transition-dialog-view", [
    "workflow-designer/io/ajax/workflow-transitions-ajax-manager",
    "workflow-designer/io/ajax/workflow-global-transitions-ajax-manager",
    "workflow-designer/io/ajax/workflow-looped-transitions-ajax-manager",
    "workflow-designer/dialogs/form-dialog-view",
    "workflow-designer/analytics",
    "workflow-designer/templates",
    "workflow-designer/underscore",
    "jquery"
], function(
    WorkflowTransitionsAJAXManager,
    WorkflowGlobalTransitionsAJAXManager,
    WorkflowLoopedTransitionsAJAXManager,
    FormDialogView,
    Analytics,
    Templates,
    _,
    jQuery
) {
    return FormDialogView.extend(
    /** @lends JIRA.WorkflowDesigner.Dialogs.DeleteTransitionDialogView# */
    {
        id: "delete-transition-dialog",

        /**
         * Initialise the view.
         *
         * @constructs
         * @classdesc The workflow designer's "Delete transition" dialog.
         * @extends JIRA.WorkflowDesigner.Dialogs.FormDialogView
         * @param {object} options
         * @param {JIRA.WorkflowDesigner.TransitionModel} options.transitionModel The transition model to destroy.
         * @param {JIRA.WorkflowDesigner.WorkflowModel} options.workflowModel The application's workflow model.
         */
        initialize: function (options) {
            _.bindAll(this, "_onError", "_onSuccess");

            this.transitionModel = options.transitionModel;
            this.workflowModel = options.workflowModel;
        },

        /**
         * @returns {jQuery} Content of the dialog
         * @private
         */
        _getContent: function () {
            var template = this._chooseContentTemplate();
            return jQuery(template({
                transitionName: this.transitionModel.get("name")
            }));
        },

        /**
         * @returns {function} Template used for the content
         * @private
         */
        _chooseContentTemplate: function() {
            if (this.transitionModel.isInitial()) {
                return Templates.cannotDeleteInitialTransitionDialog;
            }

            return Templates.confirmDeleteTransitionDialog;
        },

        /**
         * @param {jQuery.Event} e Submit event
         * @param {function} callback Function to call when this event has been handled
         * @private
         */
        _onSubmit: function (e, callback) {
            var options, request;

            e.preventDefault();
            this.disable();
            this.removeErrorMessages();

            options = {
                transitionId: this.transitionModel.get("actionId"),
                sourceStepId: this.transitionModel.getSourceStepId(),
                workflowName: this.workflowModel.get("name")
            };

            if (this.transitionModel.isLoopedTransition()) {
                request = WorkflowLoopedTransitionsAJAXManager.deleteLoopedTransition(options);
            } else if (this.transitionModel.isGlobalTransition()) {
                request = WorkflowGlobalTransitionsAJAXManager.deleteGlobalTransition(options);
            } else {
                request = WorkflowTransitionsAJAXManager.deleteTransition(options);
            }

            request.done(this._onSuccess).fail(callback).fail(this._onError);
        },

        /**
         * @private
         */
        _onSuccess: function () {
            this.transitionModel.isGlobalTransition() ?
                Analytics.triggerRemoveGlobalTransition(this.workflowModel.get('permissions')) :
                Analytics.triggerRemoveTransition(this.workflowModel.get('permissions'));

            this.hide();
            this.transitionModel.destroy();
            this.workflowModel.removeTransition(this.transitionModel);
        }
    });
});

AJS.namespace("JIRA.WorkflowDesigner.Dialogs.DeleteTransitionDialogView", null, require("workflow-designer/dialogs/delete-transition-dialog-view"));