AJS.test.require([
    "com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer",
    "com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer-loader",
    "com.atlassian.jira.plugins.jira-workflow-designer:test-resources"
], function () {

    var jQuery = require("jquery");
    var InlineDialog = require("aui/inline-dialog");
    var draw2d = require("workflow-designer/draw-2d");
    var TransitionModel = require("workflow-designer/transition-model");
    var TransitionView = require("workflow-designer/transition-view");
    var Draw2DCanvas = require("workflow-designer/draw-2d-canvas");
    var LayerRootFigure = require("workflow-designer/draw-2d-canvas/layer-root-figure");
    var StatusModel = require("workflow-designer/status-model");
    var InitialStatusView = require("workflow-designer/initial-status-view");
    var StatusView = require("workflow-designer/status-view");
    var WorkflowModel = require("workflow-designer/workflow-model");
    var CanvasModel = require("workflow-designer/canvas-model");
    var CanvasView = require("workflow-designer/canvas-view");
    var TestUtilities = require("workflow-designer/test-utilities");
    var TestMocks = require("workflow-designer/test-mocks");

    module("CanvasView", {
        setup: function () {
            this.sandbox = sinon.sandbox.create();

            this.workflowModel = new WorkflowModel();
            this.canvasModel = new CanvasModel({}, {
                workflowModel: this.workflowModel
            });

            this.canvasView = TestUtilities.testCanvasView({
                canvasModel: this.canvasModel,
                workflowModel: this.workflowModel
            });
            this.canvas = this.canvasView.canvas;
        },

        teardown: function () {
            TestUtilities.removeDialogs();
            this.canvasView.remove();
            this.sandbox.restore();
        },

        triggerKeyDown: function (keyCode, properties) {
            properties = _.extend({which: keyCode}, properties);
            jQuery(document).trigger(jQuery.Event("keydown", properties));
        }
    });

    test("Adding a connection displays an AddTransitionDialogView", function () {
        var AddTransitionDialogView = require("workflow-designer/dialogs/add-transition-dialog-view");
        var connection = new draw2d.Connection();

        var originalTransitionModelInitialize = TransitionModel.prototype.initialize;
        var createdTransitionModel;
        this.sandbox.stub(TransitionModel.prototype, "initialize", function() {
            createdTransitionModel = this;
            originalTransitionModelInitialize.apply(this, arguments);
        });

        this.sandbox.spy(AddTransitionDialogView.prototype, "initialize");
        this.sandbox.spy(AddTransitionDialogView.prototype, "show");

        var status1 = this.workflowModel.addStatus();
        var status2 = this.workflowModel.addStatus();

        connection.setSource(this.canvasView.statusViews.at(0).getPorts()[0]);
        connection.setTarget(this.canvasView.statusViews.at(1).getPorts()[0]);

        sinon.spy(this.canvas, "removeFigure");
        this.canvasView.canvas.onNewConnection(connection);

        equal(AddTransitionDialogView.prototype.initialize.callCount, 1, "An AddTransitionDialogView was created");
        ok(AddTransitionDialogView.prototype.initialize.args[0][0].canvasModel === this.canvasModel, "It was passed the correct canvas model");
        ok(AddTransitionDialogView.prototype.initialize.args[0][0].transitionModel === createdTransitionModel, "It was passed the correct transition model");
        ok(AddTransitionDialogView.prototype.initialize.args[0][0].transitionModel.get("source").model === status1.model, "It was passed the correct transition model");
        ok(AddTransitionDialogView.prototype.initialize.args[0][0].transitionModel.get("target").model === status2.model, "It was passed the correct transition model");
        ok(AddTransitionDialogView.prototype.initialize.args[0][0].workflowModel === this.workflowModel, "It was passed the correct workflow model");
        equal(AddTransitionDialogView.prototype.show.callCount, 1, "It was shown");
        ok(this.canvas.removeFigure.calledWith(connection), "The connection was removed from the canvas");
        equal(this.workflowModel.get("transitions").length, 0, "No TransitionModel was added to the WorkflowModel");
    });

    test("addStatus()", function () {
        var statusModel = new StatusModel();

        this.sandbox.spy(InitialStatusView.prototype, "initialize");
        this.sandbox.spy(StatusView.prototype, "initialize");

        this.canvasView.addStatus(statusModel);
        equal(InitialStatusView.prototype.initialize.callCount, 0, "No InitialStatusView was created for a normal status");
        equal(StatusView.prototype.initialize.callCount, 1, "A StatusView was created for a normal status");
        equal(StatusView.prototype.initialize.args[0][0].model, statusModel, "It is associated with the given model");

        statusModel.set("initial", true);
        this.canvasView.addStatus(statusModel);
        equal(InitialStatusView.prototype.initialize.callCount, 1, "An InitialStatusView was created for an initial status");
        equal(InitialStatusView.prototype.initialize.args[0][0].model, statusModel, "It is associated with the given model");
        equal(StatusView.prototype.initialize.callCount, 1, "No StatusView was created for an initial status");
    });

    test("addStatus() selects the new status if necessary", function () {
        var selectSpy = this.sandbox.spy(StatusView.prototype, "select"),
            status;

        status = new StatusModel();
        this.canvasView.addStatus(status);
        equal(selectSpy.callCount, 0, "The status wasn't selected");

        status = new StatusModel();
        this.canvasModel.set("selectedModel", status);
        this.canvasView.addStatus(status);
        equal(selectSpy.callCount, 1, "The status was selected");
    });

    test("addTransition()", function () {
        this.sandbox.spy(TransitionView.prototype, "initialize");
        var statusModel = this.workflowModel.addStatus({});
        var transitionModel;

        transitionModel = this.workflowModel.addTransition({
            name: "Transition",
            source: statusModel,
            target: statusModel
        });

        this.canvasView.addTransition(transitionModel);
        equal(TransitionView.prototype.initialize.callCount, 1, "A new TransitionView was created");
        equal(TransitionView.prototype.initialize.args[0][0].model, transitionModel, "It is associated with the given model");

        this.canvasView.addTransition(transitionModel);
        equal(TransitionView.prototype.initialize.callCount, 1, "Attempting to add duplicate transitions does nothing");
    });

    test("addTransition() selects the new transition if necessary", function () {
        var selectSpy = this.sandbox.spy(TransitionView.prototype, "select");
        var transition;

        transition = new TransitionModel({
            source: this.workflowModel.addStatus({}),
            target: this.workflowModel.addStatus({})
        });

        this.canvasView.addTransition(transition);
        equal(selectSpy.callCount, 0, "The transition wasn't selected");

        transition = new TransitionModel({
            source: this.workflowModel.addStatus({}),
            target: this.workflowModel.addStatus({})
        });

        this.canvasModel.set("selectedModel", transition);
        this.canvasView.addTransition(transition);
        equal(selectSpy.callCount, 1, "The transition was selected");
    });

    test("getCanvasBoundingBox() doesn't include LayerRootFigures", function () {
        var expected;

        this.canvasView.canvas.getFigures = function () {
            return new draw2d.util.ArrayList([
                new LayerRootFigure(),
                TestMocks.figure([10, 10, 10, 10])
            ]);
        };

        expected = new draw2d.geo.Rectangle(10, 10, 10, 10);
        ok(expected.equals(this.canvasView.getCanvasBoundingBox()),
            "The bounding box was calculated correctly");
    });

    test("getCanvasBoundingBox() includes relevant figures and lines", function () {
        var expected;

        this.canvasView.canvas.getFigures = function () {
            return new draw2d.util.ArrayList(_.map([
                [-10, -10, 5, 5],
                [0, 0, 5, 5],
                [10, 10, 5, 5]
            ], TestMocks.figure));
        };

        this.canvasView.canvas.getLines = function () {
            return new draw2d.util.ArrayList(_.map([
                [-8, -15, 13, 18],
                [-3, 3, 13, 18],
                [-5, -8, 23, 13]
            ], TestMocks.figure));
        };

        expected = new draw2d.geo.Rectangle(-10, -15, 28, 36);
        ok(expected.equals(this.canvasView.getCanvasBoundingBox()), "The bounding box was calculated correctly");
    });

    test("WorkflowModel StatusCollection observation", function () {
        var removeStatusSpy = this.sandbox.spy(this.canvasView, "removeStatus");
        var statusModel;

        this.sandbox.spy(StatusView.prototype, "initialize");

        statusModel = this.workflowModel.addStatus({});
        equal(StatusView.prototype.initialize.callCount, 1, "Adding a StatusModel causes a StatusView to be created");
        equal(StatusView.prototype.initialize.args[0][0].model, statusModel, "It is associated with the new StatusModel");

        this.workflowModel.reset({
            statuses: [],
            transitions: []
        });

        ok(removeStatusSpy.callCount === 1 && removeStatusSpy.args[0][0] === statusModel,
            "Removing a StatusModel causes its StatusView to be removed");
    });

    test("WorkflowModel TransitionCollection observation", function () {
        this.sandbox.spy(TransitionView.prototype, "initialize");
        var removeTransitionSpy = this.sandbox.spy(this.canvasView, "removeTransition");
        var statusModel;
        var transitionModel;

        statusModel = this.workflowModel.addStatus({});
        transitionModel = this.workflowModel.addTransition({
            name: "Transition",
            source: statusModel,
            target: statusModel
        });

        equal(TransitionView.prototype.initialize.callCount, 1, "Adding a TransitionModel causes a TransitionView to be created");
        equal(TransitionView.prototype.initialize.args[0][0].model, transitionModel, "It is associated with the new TransitionModel");

        this.workflowModel.reset({
            statuses: [],
            transitions: []
        });

        ok(removeTransitionSpy.callCount === 1 && removeTransitionSpy.args[0][0] === transitionModel,
            "Removing a TransitionModel causes its TransitionView to be removed");
    });

    test("Transition reconnection updates the TransitionModel if target status was not changed", function () {
        var closed = this.workflowModel.addStatus({x: 0, y: 100}),
            closedView = this.canvasView.addStatus(closed),
            expectedAttributes,
            open = this.workflowModel.addStatus(),
            openView = this.canvasView.addStatus(open),
            transition,
            transitionView;

        transition = this.workflowModel.addTransition({
            name: "Transition",
            source: open,
            sourceAngle: 90,
            target: closed,
            targetAngle: -90
        });

        transitionView = this.canvasView.addTransition(transition);
        transitionView._connection.setSource(openView.getPortForAngle(0));
        transitionView._connection.setTarget(closedView.getPortForAngle(0));
        transitionView.trigger("reconnect", transitionView);

        expectedAttributes = {
            source: open,
            sourceAngle: openView.getAngleToPort(transitionView._connection.getSource()),
            target: closed,
            targetAngle: closedView.getAngleToPort(transitionView._connection.getTarget())
        };

        deepEqual(_.pick(transition.attributes, "source", "sourceAngle", "target", "targetAngle"), expectedAttributes,
            "The TransitionModel's sourceAngle and targetAngle attributes were updated");
    });

    test("Transition reconnection shows EditTransitionTargetDialogView if target status was changed", function () {
        var EditTransitionTargetDialogView = require("workflow-designer/dialogs/edit-transition-target-dialog-view");
        this.sandbox.spy(EditTransitionTargetDialogView.prototype, "initialize");
        this.sandbox.stub(EditTransitionTargetDialogView.prototype, "show");

        var closed = this.workflowModel.addStatus({x: 0, y: 100});
        var expectedAttributes;
        var open = this.workflowModel.addStatus();
        var openView = this.canvasView.addStatus(open);
        var reopened = this.workflowModel.addStatus();
        var reopenedView = this.canvasView.addStatus(reopened);
        var transition;
        var transitionView;

        transition = this.workflowModel.addTransition({
            name: "Transition",
            source: open,
            sourceAngle: 90,
            target: closed,
            targetAngle: -90
        });

        transitionView = this.canvasView.addTransition(transition);
        transitionView._connection.setSource(openView.getPortForAngle(0));
        transitionView._connection.setTarget(reopenedView.getPortForAngle(0));
        transitionView.trigger("reconnect", transitionView);

        expectedAttributes = {
            targetPort: reopenedView.getPortForAngle(0),
            targetView: reopenedView,
            transitionView: transitionView,
            workflowModel: this.workflowModel
        };

        equal(EditTransitionTargetDialogView.prototype.initialize.callCount, 1, "An EditTransitionTargetDialogView was created");
        ok(_.isEqual(EditTransitionTargetDialogView.prototype.initialize.args[0][0], expectedAttributes), "It was passed the correct arguments");
    });

    test("Transition reconnection shows EditTransitionSourceDialogView if source status was changed", function () {
        var EditTransitionSourceDialogViewStub = require("workflow-designer/dialogs/edit-transition-source-dialog-view");
        this.sandbox.spy(EditTransitionSourceDialogViewStub.prototype, "initialize");
        this.sandbox.stub(EditTransitionSourceDialogViewStub.prototype, "show");

        var closed = this.workflowModel.addStatus({x: 0, y: 100});
        var expectedAttributes;
        var open = this.workflowModel.addStatus();
        var reopened = this.workflowModel.addStatus();
        var reopenedView = this.canvasView.addStatus(reopened);
        var transition;
        var transitionView;

        transition = this.workflowModel.addTransition({
            name: "Transition",
            source: open,
            sourceAngle: 90,
            target: closed,
            targetAngle: -90
        });

        transitionView = this.canvasView.addTransition(transition);
        transitionView._connection.setSource(reopenedView.getPortForAngle(0));
        transitionView.trigger("reconnect", transitionView);

        expectedAttributes = {
            newSourcePort: reopenedView.getPortForAngle(0),
            newSourceView: reopenedView,
            originalSourceStatus: open,
            transitionView: transitionView,
            workflowModel: this.workflowModel
        };

        equal(EditTransitionSourceDialogViewStub.prototype.initialize.callCount, 1, "An EditTransitionSourceDialogView was created");
        ok(_.isEqual(EditTransitionSourceDialogViewStub.prototype.initialize.args[0][0], expectedAttributes), "It was passed the correct arguments");
    });

    test("Statuses without coordinates are positioned automatically", function () {
        var canvasView = this.canvasView,
            layoutChangedSpy = sinon.spy(),
            resetConnectionSpy = sinon.spy(),
            statusModelOne = new StatusModel({id: "S<1>", x: 10, y: 10}),
            statusModelTwo = new StatusModel({id: "S<2>", x: 200, y: 200}),
            statusModelThree = new StatusModel({id: "S<3>"}),
            statusModelFour = new StatusModel({id: "S<4>"}),
            transitionModel = new TransitionModel({
                name: "Transition",
                source: statusModelThree,
                target: statusModelFour
            });

        canvasView.addTransition = function () {
            var transitionView = CanvasView.prototype.addTransition.apply(canvasView, arguments);
            transitionView.resetConnection = resetConnectionSpy;
            return transitionView;
        };

        this.workflowModel.on("layoutChanged", layoutChangedSpy);
        this.workflowModel.reset({
            statuses: [statusModelOne, statusModelTwo, statusModelThree, statusModelFour],
            transitions: [transitionModel]
        });

        ok(_.isNumber(statusModelThree.get("x")), "The first model has an x coordinate");
        ok(_.isNumber(statusModelThree.get("y")), "The first model has a y coordinate");
        ok(_.isNumber(statusModelFour.get("x")), "The second model has an x coordinate");
        ok(_.isNumber(statusModelFour.get("y")), "The second model has a y coordinate");
        ok(layoutChangedSpy.called, "Triggered a layoutChanged event");
        ok(resetConnectionSpy.called, "Transition between positioned statuses was reset");
    });

    test("Autocalculated transition ports must trigger autosaving", function () {
        var layoutChangedSpy = sinon.spy(),
            source = new StatusModel({x: 10, y: 10}),
            target = new StatusModel({x: 200, y: 200}),
            transitionModel;

        transitionModel = new TransitionModel({
            name: "Transition",
            source: source,
            target: target,
            ports: null
        });

        this.workflowModel.on("layoutChanged", layoutChangedSpy);
        this.workflowModel.reset({
            statuses: [source, target],
            transitions: [transitionModel]
        });

        ok(layoutChangedSpy.called, "Triggered a layoutChanged event");
    });

    test("Zooming with the mouse wheel hides the current inline dialog", function () {
        this.sandbox.stub(AJS, "InlineDialog");
        InlineDialog.current = {hide: sinon.spy()};

        sinon.stub(this.canvasView, "getCanvasBoundingBox").returns(new draw2d.geo.Rectangle(0, 0, 0, 0));
        this.canvasView._zoomHandler.trigger('zoom', {
            clientX: 10,
            clientY: 10,
            factor: 1.05
        });

        equal(InlineDialog.current.hide.callCount, 1, "The current inline dialog was hidden");
    });

    test("The canvas is resized after its container is resized", function () {
        TestUtilities.fakeTimer(function (clock) {
            var canvasView = TestUtilities.testCanvasView({model: this.workflowModel});

            var originalContainerWidth = canvasView.$el.width();
            var originalContainerHeight = canvasView.$el.height();

            clock.tick(150);
            equal(originalContainerWidth, canvasView.canvas.svgElement.width());
            equal(originalContainerHeight, canvasView.canvas.svgElement.height());

            var assertDimensions = function () {
                equal(canvasView.$el.width(), canvasView.canvas.svgElement.width());
                equal(canvasView.$el.height(), canvasView.canvas.svgElement.height());
            };

            canvasView.$el.width(originalContainerWidth * 0.5);
            canvasView.$el.height(originalContainerHeight * 0.5);
            clock.tick(150);
            assertDimensions();

            canvasView.$el.width(originalContainerWidth);
            canvasView.$el.height(originalContainerHeight);
            clock.tick(150);
            assertDimensions();

            canvasView.remove();
        }, this);
    });

    test("Deleting the selected transition", function () {
        var destroyStub,
            keyCodes = [jQuery.ui.keyCode.BACKSPACE, jQuery.ui.keyCode.DELETE],
            transitionView;

        this.workflowModel.addTransition({
            name: "Transition",
            source: this.workflowModel.addStatus({}),
            target: this.workflowModel.addStatus({})
        });

        transitionView = this.canvasView.transitionViews.at(0);
        this.canvas.selectFigure(transitionView._connection);
        destroyStub = this.sandbox.stub(transitionView, "destroy");

        _.each(keyCodes, function (keyCode) {
            this.triggerKeyDown(keyCode);
            equal(destroyStub.callCount, 1, "TransitionView.destroy() was called for key code " + keyCode);
            destroyStub.reset();
        }, this);
    });

    test("Deleting the selected status", function () {
        var destroyStub,
            keyCodes = [jQuery.ui.keyCode.BACKSPACE, jQuery.ui.keyCode.DELETE],
            statusView;

        this.workflowModel.addStatus({x: 0, y: 100});
        statusView = this.canvasView.statusViews.at(0);
        this.canvas.selectFigure(statusView._figure);
        destroyStub = this.sandbox.stub(statusView, "destroy");

        _.each(keyCodes, function (keyCode) {
            this.triggerKeyDown(keyCode);
            equal(destroyStub.callCount, 1, "StatusView.destroy() was called for key code " + keyCode);
            destroyStub.reset();
        }, this);
    });

    test("Deletion is disabled when a dialog is visible", function () {
        var destroyStub,
            keyCodes = [jQuery.ui.keyCode.BACKSPACE, jQuery.ui.keyCode.DELETE],
            statusModel,
            statusView,
            transitionModel,
            transitionView;

        // "Show" a dialog
        jQuery("#qunit-fixture").append("<div class=\"aui-blanket\">");

        transitionModel = this.workflowModel.addTransition({
            name: "Transition",
            source: this.workflowModel.addStatus({}),
            target: this.workflowModel.addStatus({})
        });

        transitionView = this.canvasView.addTransition(transitionModel).render();
        this.canvas.selectFigure(transitionView._connection);
        destroyStub = this.sandbox.stub(transitionView, "destroy");

        _.each(keyCodes, function (keyCode) {
            this.triggerKeyDown(keyCode);
            equal(destroyStub.callCount, 0, "TransitionView.destroy() was not called");
        }, this);

        statusModel = this.workflowModel.addStatus({x: 0, y: 100});
        statusView = this.canvasView.addStatus(statusModel).render();
        statusView.select();
        destroyStub = this.sandbox.stub(statusView, "destroy");

        _.each(keyCodes, function (keyCode) {
            this.triggerKeyDown(keyCode);
            equal(destroyStub.callCount, 0, "StatusView.destroy() was not called");
        }, this);
    });

    test("Deletion is disabled when the view is locked", function () {
        var statusView;

        this.canvasView.setLocked(true);
        this.workflowModel.addStatus({x: 0, y: 0});
        statusView = this.canvasView.statusViews.at(0);
        statusView.select();
        this.sandbox.stub(statusView, "destroy");

        this.triggerKeyDown(jQuery.ui.keyCode.BACKSPACE);
        equal(statusView.destroy.callCount, 0, "StatusView#destroy wasn't called");
    });

    test("Pressing delete key when target is an input does not display a DeleteTransitionDialogView", function () {
        var destroyStub,
            keyCodes = [jQuery.ui.keyCode.BACKSPACE, jQuery.ui.keyCode.DELETE],
            transitionModel,
            transitionView;

        transitionModel = this.workflowModel.addTransition({
            name: "Transition",
            source: this.workflowModel.addStatus({}),
            target: this.workflowModel.addStatus({})
        });

        transitionView = this.canvasView.addTransition(transitionModel).render();
        this.canvas.selectFigure(transitionView._connection);
        destroyStub = this.sandbox.stub(transitionView, "destroy");

        _.each(keyCodes, function (keyCode) {
            this.triggerKeyDown(keyCode, {target: jQuery("<input>")});
            equal(destroyStub.callCount, 0, "TransitionView.destroy() was not called");
        }, this);
    });

    test("Auto-fitting fits the SVG element to its container", function () {
        sinon.spy(this.canvasView.canvas, "fitToContainer");

        this.canvasView.autoFit();
        equal(this.canvasView.canvas.fitToContainer.callCount, 1,
            "Canvas#fitToContainer() was called");
    });

    test("Auto-fitting centers the diagram if it fits into default bounds", function () {
        var setViewBoxSpy = sinon.spy(this.canvasView.canvas, "setViewBox");

        this.canvasView.$el.width(500);
        this.canvasView.$el.height(200);

        this.canvasView.canvas.getFigures = function () {
            return new draw2d.util.ArrayList([TestMocks.figure([-10, -10, 25, 25])]);
        };

        this.canvasView.autoFit();

        ok(setViewBoxSpy.args[0][0].equals(new draw2d.geo.Rectangle(-247.5, -97.5, 500, 200)), "View box dimensions were not changed and the diagram was centered");
    });

    test("Auto-fitting scales and centers the diagram if it doesn't fit into default bounds", function () {
        var setViewBoxSpy = sinon.spy(this.canvasView.canvas, "setViewBox");

        this.canvasView.$el.width(500);
        this.canvasView.$el.height(200);

        sinon.stub(this.canvasView.canvas, "getViewBox").returns(new draw2d.geo.Rectangle(-20, -20, 620, 350));

        this.canvasView.canvas.getFigures = function () {
            return new draw2d.util.ArrayList([TestMocks.figure([-10, -10, 600, 300])]);
        };

        this.canvasView.autoFit();

        ok(setViewBoxSpy.args[1][0].equals(new draw2d.geo.Rectangle(-20, -35, 620, 350)), "View box was scaled and the diagram was centered");
    });

    test("Auto-fitting sets the CanvasModel's zoomLevel property", function () {
        // Add a status so the canvas has a bounding box.
        this.canvasView.addStatus(new StatusModel());

        ok(!this.canvasModel.has("zoomLevel"), "The CanvasModel's zoomLevel property isn't set");

        this.canvasView.autoFit();
        ok(this.canvasModel.has("zoomLevel"), "The CanvasModel's zoomLevel property is set");
        equal(this.canvasModel.get("zoomLevel"), this.canvasView.canvas.getZoom(), "It is set to the canvas zoom level");
    });

    test("setResizeInterval()", function () {
        var setResizeIntervalSpy = this.sandbox.spy(Draw2DCanvas.prototype, "setResizeInterval");

        this.canvasView.setResizeInterval(42);
        equal(setResizeIntervalSpy.callCount, 1, "Draw2DCanvas#setResizeInterval() was called");
        deepEqual(setResizeIntervalSpy.args[0], [42], "It was passed the correct arguments");
    });

    test("Selects views in response to the CanvasModel's selectedModel property changing", function () {
        var onSelectSpy = this.sandbox.spy(StatusView.prototype, "_onSelect"),
            statusModel = this.workflowModel.addStatus({});

        this.canvasModel.set("selectedModel", statusModel);
        equal(onSelectSpy.callCount, 1, "The view was selected");
    });

    test("Updates the CanvasModel in response to view (de)selection", function () {
        var statusModel = this.workflowModel.addStatus({}),
            statusView = this.canvasView.statusViews.at(0);

        statusView.select();
        ok(this.canvasModel.get("selectedModel") === statusModel, "The CanvasModel's selectedModel property was updated");
        ok(this.canvasModel.get("selectedView") === statusView, "The CanvasModel's selectedView property was updated");

        statusView.deselect();
        ok(!this.canvasModel.has("selectedModel"), "The CanvasModel's selectedModel property was cleared");
        ok(!this.canvasModel.has("selectedView"), "The CanvasModel's selectedView property was cleared");
    });

    test("Source view is updated when model source is updated", function () {
        var newSourceStatusModel, transitionModel, transitionView;

        transitionModel = this.workflowModel.addTransition({
            name: "Transition",
            source: this.workflowModel.addStatus({}),
            target: this.workflowModel.addStatus({})
        });

        transitionView = this.canvasView._getTransitionViewWithModel(transitionModel);
        this.sandbox.spy(transitionView, "resetConnection");

        newSourceStatusModel = this.workflowModel.addStatus({});
        transitionModel.set("source", newSourceStatusModel);

        ok(transitionView._getSourceView() === this.canvasView._getStatusViewWithModel(newSourceStatusModel), "Source view was updated");
        equal(transitionView.resetConnection.callCount, 1, "Connection was reset");
    });

    test("Target view is updated when model target is updated", function () {
        var newTargetStatusModel, transitionModel, transitionView;

        transitionModel = this.workflowModel.addTransition({
            name: "Transition",
            source: this.workflowModel.addStatus({}),
            target: this.workflowModel.addStatus({})
        });

        transitionView = this.canvasView._getTransitionViewWithModel(transitionModel);
        this.sandbox.spy(transitionView, "resetConnection");

        newTargetStatusModel = this.workflowModel.addStatus({});
        transitionModel.set("target", newTargetStatusModel);

        ok(transitionView._getTargetView() === this.canvasView._getStatusViewWithModel(newTargetStatusModel), "Target view was updated");
        equal(transitionView.resetConnection.callCount, 1, "Connection was reset");
    });

    test("Zooming sets the CanvasModel's zoomLevel property", function () {
        // Add a status so the canvas has a bounding box.
        this.canvasView.addStatus(new StatusModel());

        ok(!this.canvasModel.has("zoomLevel"), "The CanvasModel's zoomLevel property isn't set");

        this.canvasView.zoom(1.0);
        ok(this.canvasModel.has("zoomLevel"), "The CanvasModel's zoomLevel property is set");
        equal(this.canvasModel.get("zoomLevel"), this.canvasView.canvas.getZoom(), "It is set to the canvas zoom level");
    });

    test("When a transition is (de)selected, its siblings are (de)selected", function () {
        var closed = this.workflowModel.addStatus({id: 1}),
            inProgress = this.workflowModel.addStatus({id: 2}),
            open = this.workflowModel.addStatus({id: 3}),
            otherTransition,
            otherTransitionView,
            siblingTransition,
            siblingTransitionView,
            transition,
            transitionView;

        otherTransition = this.workflowModel.addTransition({
            actionId: "2",
            source: open,
            target: inProgress
        });

        siblingTransition = this.workflowModel.addTransition({
            actionId: "1",
            source: open,
            target: closed
        });

        transition = this.workflowModel.addTransition({
            actionId: "1",
            source: inProgress,
            target: closed
        });

        otherTransitionView = this.canvasView._getTransitionViewWithModel(otherTransition);
        siblingTransitionView = this.canvasView._getTransitionViewWithModel(siblingTransition);
        transitionView = this.canvasView._getTransitionViewWithModel(transition);

        this.sandbox.spy(otherTransitionView, "appearSelected");
        this.sandbox.spy(siblingTransitionView, "appearSelected");
        transitionView.trigger("selected", transitionView);
        equal(otherTransitionView.appearSelected.callCount, 0, "Non-sibling transitions aren't selected");
        equal(siblingTransitionView.appearSelected.callCount, 1, "Sibling transitions are selected");

        this.sandbox.spy(otherTransitionView, "unhighlight");
        this.sandbox.spy(siblingTransitionView, "unhighlight");
        transitionView.trigger("deselected", transitionView);
        equal(otherTransitionView.unhighlight.callCount, 0, "Non-sibling transitions aren't deselected");
        equal(siblingTransitionView.unhighlight.callCount, 1, "Sibling transitions are deselected");
    });

    test("validation is triggered when status or transition is added or removed", function () {
        var source = this.workflowModel.addStatus({});
        var transition = this.workflowModel.addTransition({
            actionId: "1",
            source: source,
            target: this.workflowModel.addStatus({})
        });
        equal(this.canvasView._validate.callCount, 3, "validation was triggered three times after two statuses and one transition were added");

        this.canvasView._validate.reset();
        this.workflowModel.get("transitions").remove(transition);
        equal(this.canvasView._validate.callCount, 1, "validation was triggered after transition was removed");

        this.canvasView._validate.reset();
        this.workflowModel.get("statuses").remove(source);
        equal(this.canvasView._validate.callCount, 1, "validation was triggered after status was removed");
    });

});
