define("workflow-designer/actions-view", [
    "workflow-designer/dialogs/add-transition-dialog-view",
    "workflow-designer/dialogs/add-status-inline-dialog-view",
    "workflow-designer/status-model",
    "workflow-designer/transition-model",
    "jira/util/users/logged-in-user",
    "workflow-designer/templates",
    "workflow-designer/backbone",
    "workflow-designer/underscore",
    "jquery"
], function(
    AddTransitionDialogView,
    AddStatusInlineDialogView,
    StatusModel,
    TransitionModel,
    LoggedInUser,
    Templates,
    Backbone,
    _,
    jQuery
) {
    return Backbone.Marionette.ItemView.extend(
    /** @lends JIRA.WorkflowDesigner.ActionsView# */
    {
        className: "actions",

        events: {
            "change .show-transition-labels": "_setShowTransitionLabels",
            "click .add-status": "_addStatus",
            "click .add-transition": "_addTransition",
            "click .enter-full-screen": "_enterFullScreen",
            "click .exit-full-screen": "_exitFullScreen"
        },

        template: Templates.actions,

        ui: {
            addStatusButton: ".add-status",
            fullScreenTriggers: ".full-screen-trigger",
            lastEditedMessage: ".last-edited-message"
        },

        /**
         * Initialise the view.
         *
         * @classdesc The workflow designer's actions menu ("Add Status", etc.).
         * @constructs
         * @extends Backbone.View
         * @param {object} options
         * @param {JIRA.WorkflowDesigner.CanvasModel} options.canvasModel The application's `CanvasModel`.
         * @param {boolean} options.fullScreenButton Whether full screen button is enabled.
         * @param {boolean} options.immutable Whether the workflow is immutable.
         * @param {JIRA.WorkflowDesigner.WorkflowModel} options.workflowModel The application's `WorkflowModel`.
         */
        initialize: function (options) {
            this._canvasModel = options.canvasModel;
            this._fullScreenButton = options.fullScreenButton;
            this._immutable = options.immutable;
            this._workflowModel = options.workflowModel;

            this.listenTo(this._workflowModel, "change:updateAuthor change:updatedDate", function () {
                if (this._workflowModel.get("updateAuthor") && this._workflowModel.get("updatedDate")) {
                    this.stopListening(this._workflowModel);
                    this.render();
                }
            });
        },

        /**
         * Open the "Add Status" dialog.
         *
         * Called when the "Add status" button is clicked.
         *
         * @private
         */
        _addStatus: function () {
            var isActive = this.ui.addStatusButton.hasClass("active");

            this._addStatusDialog || (this._addStatusDialog = this._createAddStatusDialog());

            if (isActive) {
                this._addStatusDialog.hide();
            } else {
                this._addStatusDialog.show();
            }
        },

        /**
         * Open the "Add Transition" dialog.
         *
         * Called when the "Add transition" button is clicked. If a status is
         * currently selected, it is selected in the source and target fields.
         *
         * @private
         */
        _addTransition: function () {
            var selectedModel = this._canvasModel.get("selectedModel"),
                statusSelected = selectedModel && selectedModel instanceof StatusModel,
                transition = new TransitionModel();

            if (statusSelected) {
                transition.set({
                    source: selectedModel,
                    target: selectedModel
                });
            }

            new AddTransitionDialogView({
                canvasModel: this._canvasModel,
                transitionModel: transition,
                workflowModel: this._workflowModel
            }).show();
        },

        /**
         * @private
         * @returns {JIRA.WorkflowDesigner.Dialogs.AddStatusInlineDialogView} An add status dialog.
         */
        _createAddStatusDialog: function () {
            var dialog;

            dialog = new AddStatusInlineDialogView({
                trigger: this.ui.addStatusButton,
                workflowModel: this._workflowModel
            });

            this.listenTo(dialog, {
                done: _.bind(this.trigger, this, "addStatus:done"),
                submit: _.bind(this.trigger, this, "addStatus:submit")
            });

            return dialog;
        },

        /**
         * @private
         */
        _enterFullScreen: function () {
            this._toggleFullScreenTriggers();
            this.trigger("fullScreen:enter");
        },

        /**
         * @private
         */
        _exitFullScreen: function () {
            this._toggleFullScreenTriggers();
            this.trigger("fullScreen:exit");
        },

        /**
         * Prepare the full screen buttons after rendering.
         */
        onRender: function () {
            if (this._addStatusDialog) {
                // button is re-rendered so need to recreate the dialog.
                this._addStatusDialog.remove();
                delete this._addStatusDialog;
            }

            this.ui.fullScreenTriggers.tooltip();
            this.ui.fullScreenTriggers.filter(".hidden").tipsy("disable");
            this.ui.lastEditedMessage.find("time").livestamp();
        },

        /**
         * Remove the view.
         */
        remove: function () {
            Backbone.Marionette.ItemView.prototype.remove.apply(this, arguments);
            this._addStatusDialog && this._addStatusDialog.remove();
        },

        /**
         * @returns {object} Data for the view's template.
         */
        serializeData: function () {
            return {
                currentUserName: LoggedInUser.username(),
                updatedDate: this._workflowModel.get("updatedDate") && this._workflowModel.get("updatedDate").getTime(),
                updateAuthor: this._workflowModel.get("updateAuthor"),
                fullScreenButton: this._fullScreenButton,
                immutable: this._immutable,
                showTransitionLabels: this._canvasModel.get("showTransitionLabels")
            };
        },

        /**
         * Set the `CanvasModel`'s `showTransitionLabels` property.
         *
         * @param {jQuery.Event} e The change event that triggered this method.
         * @private
         */
        _setShowTransitionLabels: function (e) {
            var showTransitionLabels = jQuery(e.target).is(":checked");
            this._canvasModel.set("showTransitionLabels", showTransitionLabels);
        },

        /**
         * Toggle the visibility of the enter/exit full screen buttons.
         *
         * @private
         */
        _toggleFullScreenTriggers: function () {
            this.ui.fullScreenTriggers.each(function (i, trigger) {
                jQuery(trigger).tipsy("toggleEnabled");
            }).toggleClass("hidden");
        }
    });
});

AJS.namespace("JIRA.WorkflowDesigner.ActionsView", null, require("workflow-designer/actions-view"));