AJS.test.require([
    "com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer",
    "com.atlassian.jira.plugins.jira-workflow-designer:test-resources"
], function () {

    var ActionsView; // required later
    var WorkflowModel = require("workflow-designer/workflow-model");
    var CanvasModel = require("workflow-designer/canvas-model");
    var TestUtilities = require("workflow-designer/test-utilities");
    var Backbone = require("workflow-designer/backbone");

    module("ActionsView", {
        createView: function (options) {
            var actionsView;

            ActionsView = this.context.require("workflow-designer/actions-view");

            actionsView = new ActionsView(_.extend({
                canvasModel: this.canvasModel,
                fullScreenButton: true,
                workflowModel: this.workflowModel
            }, options));

            this.container.empty().append(actionsView.render().el);
            return actionsView;
        },

        mockUser: function() {
            this.context.mock("jira/util/users/logged-in-user",{username: this.sandbox.stub().returns("bob")});
        },

        setup: function () {
            this.context = AJS.test.mockableModuleContext();
            this.context.mock("workflow-designer/status-model", require("workflow-designer/status-model"));

            this.container = jQuery("#qunit-fixture");
            this.sandbox = sinon.sandbox.create();
            this.workflowModel = new WorkflowModel();
            this.canvasModel = new CanvasModel({}, {
                workflowModel: this.workflowModel
            });
        },

        teardown: function () {
            TestUtilities.removeDialogs();
            this.sandbox.restore();
        }
    });

    test("Add Status", function () {
        var addStatusButton,
            dialog = _.extend({hide: sinon.stub(), show: sinon.stub()}, Backbone.Events),
            dialogStub = this.sandbox.stub().returns(dialog);

        this.context.mock("workflow-designer/dialogs/add-status-inline-dialog-view", dialogStub);
        this.createView();

        addStatusButton = this.container.find(".add-status");

        addStatusButton.click();
        equal(dialogStub.callCount, 1, "An AddStatusInlineDialogView was created");
        ok(dialogStub.args[0][0].trigger[0] === this.container.find(".add-status")[0], "Its constructor was passed the correct arguments");
        equal(dialog.show.callCount, 1, "The dialog was shown");

        addStatusButton.addClass("active").click();
        equal(dialog.hide.callCount, 1, "The dialog was hidden");
    });

    test("Add transition", function () {
        var dialog = {show: sinon.spy()},
            dialogStub = this.sandbox.stub().returns(dialog),
            transition;

        this.context.mock("workflow-designer/dialogs/add-transition-dialog-view", dialogStub);
        var actionsView = this.createView();

        actionsView.$(".add-transition").click();
        equal(dialogStub.callCount, 1, "An AddTransitionDialogView was created");
        ok(dialogStub.args[0][0].workflowModel === this.workflowModel, "Its constructor was passed the correct arguments");
        equal(dialog.show.callCount, 1, "The dialog was shown");

        transition = dialogStub.args[0][0].transitionModel;
        ok(!transition.has("source"), "The transition doesn't have a source set");
        ok(!transition.has("target"), "The transition doesn't have a target set");
    });

    test("Add transition with a status selected", function () {
        var dialogSpy = this.sandbox.spy(require("workflow-designer/dialogs/add-transition-dialog-view"));
        this.context.mock("workflow-designer/dialogs/add-transition-dialog-view", dialogSpy);
        var status = this.workflowModel.addStatus();
        var actionsView = this.createView();
        var transition;

        this.canvasModel.set("selectedModel", status);
        actionsView.$(".add-transition").click();

        transition = dialogSpy.args[0][0].transitionModel;
        ok(transition.get("source") === status, "The transition's source is the selected status");
        ok(transition.get("target") === status, "The transition's target is the selected status");
    });

    test("Contains editing buttons in mutable mode", function () {
        this.createView();
        equal(this.container.find(".aui-toolbar2-primary").length, 1, "Editing buttons are present");
    });

    test("Doesn't contain editing buttons in immutable mode", function () {
        this.createView({immutable: true});
        equal(this.container.find(".aui-toolbar2-primary").length, 0, "Editing buttons are not present");
    });

    test("Entering and exiting full screen mode", function () {
        var actionsView = this.createView(),
            enterFullScreenButton = this.container.find(".enter-full-screen"),
            enterFullScreenButtonTipsy = enterFullScreenButton.tipsy(true),
            exitFullScreenButton = this.container.find(".exit-full-screen"),
            exitFullScreenButtonTipsy = exitFullScreenButton.tipsy(true),
            enterFullScreenSpy = sinon.spy(),
            exitFullScreenSpy = sinon.spy();

        actionsView.on("fullScreen:enter", enterFullScreenSpy);
        actionsView.on("fullScreen:exit", exitFullScreenSpy);

        ok(!enterFullScreenButton.hasClass("hidden"), "Enter full screen button is visible initially");
        ok(enterFullScreenButtonTipsy.enabled, "Enter full screen button tooltip is enabled");
        ok(exitFullScreenButton.hasClass("hidden"), "Exit full screen button is hidden initially");
        ok(!exitFullScreenButtonTipsy.enabled, "Exit full screen button tooltip is disabled");

        enterFullScreenButton.click();

        ok(enterFullScreenButton.hasClass("hidden"), "Enter full screen button is hidden after entering full screen");
        ok(!enterFullScreenButtonTipsy.enabled, "Enter full screen button tooltip is disabled");
        ok(!exitFullScreenButton.hasClass("hidden"), "Exit full screen button is visible after entering full screen");
        ok(exitFullScreenButtonTipsy.enabled, "Exit full screen button tooltip is enabled");
        ok(enterFullScreenSpy.calledOnce, "fullScreen:enter event triggered upon entering full screen");

        exitFullScreenButton.click();

        ok(!enterFullScreenButton.hasClass("hidden"), "Enter full screen button is visible after exiting full screen");
        ok(enterFullScreenButtonTipsy.enabled, "Enter full screen button tooltip is enabled");
        ok(exitFullScreenButton.hasClass("hidden"), "Exit full screen button is hidden after exiting full screen");
        ok(!exitFullScreenButtonTipsy.enabled, "Exit full screen button tooltip is disabled");
        ok(exitFullScreenSpy.calledOnce, "fullScreen:exit event triggered upon exiting full screen");
    });

    test("Events are forwarded from the AddStatusInlineDialogView", function () {
        var dialog = _.extend({show: jQuery.noop}, Backbone.Events),
            doneSpy = sinon.spy(),
            submitSpy = sinon.spy();

        this.context.mock("workflow-designer/dialogs/add-status-inline-dialog-view", this.sandbox.stub().returns(dialog));
        this.createView().on({
            "addStatus:done": doneSpy,
            "addStatus:submit": submitSpy
        });

        this.container.find(".add-status").click();

        dialog.trigger("submit");
        equal(submitSpy.callCount, 1, "An addStatus:submit event was triggered");

        dialog.trigger("done");
        equal(doneSpy.callCount, 1, "An addStatus:done event was triggered");
    });

    test("Full screen mode can be disabled", function () {
        this.createView({fullScreenButton: false});
        equal(this.container.find(".full-screen-trigger").length, 0, "No full screen triggers are present");
    });

    test("Toggling the \"Show Transition Labels\" checkbox updates CanvasModel", function () {
        var checkbox;

        this.canvasModel.set("showTransitionLabels", false);
        this.createView();
        checkbox = this.container.find('#show-transition-labels');
        ok(!checkbox.is(":checked"), "Checkbox is rendered unchecked when the model attribute is false");

        this.canvasModel.set("showTransitionLabels", true);
        this.createView();
        checkbox = this.container.find('#show-transition-labels');
        ok(checkbox.is(":checked"), "Checkbox is rendered checked when the model attribute is true");

        checkbox.attr("checked", true).trigger("change");
        ok(this.canvasModel.get("showTransitionLabels"), "The model property is true when checkbox is checked");

        checkbox.attr("checked", false).trigger("change");
        ok(!this.canvasModel.get("showTransitionLabels"), "The model property is false when checkbox is not checked");
    });

    test("Last edited message not displayed when immutable", function () {
        var lastEditedMessage;
        this.createView({
            immutable: true
        });

        this.workflowModel.set("updateAuthor", {userName: "bob", displayName: "Bob Dude"});
        this.workflowModel.set("updatedDate", new Date());

        lastEditedMessage = this.container.find(".last-edited-message");

        ok(!lastEditedMessage.length, "Last edited message should not be displayed.");
    });

    test("Last edited message displayed when both updateAuthor and updatedDate defined", function () {
        var lastEditedMessage;
        this.mockUser();
        this.createView();

        this.workflowModel.set("updateAuthor", {userName: "bob", displayName: "Bob Dude"});
        this.workflowModel.set("updatedDate", new Date());

        lastEditedMessage = this.container.find(".last-edited-message");

        equal(lastEditedMessage.text(), "workflow.designer.last.edited.by.you", "Last edited message should be displayed.");
    });

    test("Last edited message not displayed when lastUpdater defined and updatedDate not defined", function () {
        var lastEditedMessage;
        this.mockUser();
        this.createView();

        this.workflowModel.set("updateAuthor", {userName: "bob", displayName: "Bob Dude"});

        lastEditedMessage = this.container.find(".last-edited-message");

        ok(!lastEditedMessage.length, "Last edited message should not be displayed.");
    });

    test("Last edited message not displayed when updateAuthor not defined and updatedDate defined", function () {
        var lastEditedMessage;
        this.mockUser();
        this.createView();

        this.workflowModel.set("updatedDate", new Date());

        lastEditedMessage = this.container.find(".last-edited-message");

        ok(!lastEditedMessage.length, "Last edited message should not be displayed.");
    });

    test("Last edited message not displayed when both updateAuthor and updatedDate not defined", function () {
        var lastEditedMessage;
        this.mockUser();
        this.createView();

        lastEditedMessage = this.container.find(".last-edited-message");

        ok(!lastEditedMessage.length, "Last edited message should not be displayed.");
    });

    test("'You' last edited message should be displayed if current user is same as last updater", function () {
        var lastEditedMessage;
        this.mockUser();
        this.createView();

        this.workflowModel.set("updateAuthor", {userName: "bob", displayName: "Bob Dude"});
        this.workflowModel.set("updatedDate", new Date());

        lastEditedMessage = this.container.find(".last-edited-message");

        equal(lastEditedMessage.text(), "workflow.designer.last.edited.by.you", "Last edited message should be displayed.");
    });

    test("'User' last edited message should be displayed if current user is not same as last updater", function () {
        this.sandbox.stub(AJS, "format");
        this.mockUser();
        this.createView();

        this.workflowModel.set("updateAuthor", {userName: "jim", displayName: "Jim Dude"});
        this.workflowModel.set("updatedDate", new Date());

        sinon.assert.calledWith(AJS.format, "workflow.designer.last.edited.by.other.user", "Jim Dude");
    });

    test("Last edited message not updated when model updated", function () {
        this.mockUser();
        this.createView();

        this.workflowModel.set("updateAuthor", {userName: "bob", displayName: "Bob Dude"});
        this.workflowModel.set("updatedDate", new Date(new Date().getTime()));

        equal(this.container.find(".last-edited-message").text(), "workflow.designer.last.edited.by.you", "Last edited message should be displayed.");

        this.workflowModel.set("updateAuthor", {userName: "jim", displayName: "Jim Man"});

        equal(this.container.find(".last-edited-message").text(), "workflow.designer.last.edited.by.you", "Last edited message should still be the same as before.");

    });

    test("The add status inline dialog is recreated on render", function () {
        var dialog = _.extend({remove: sinon.stub(), show: sinon.stub()}, Backbone.Events),
            dialogStub = this.sandbox.stub().returns(dialog);

        this.context.mock("workflow-designer/dialogs/add-status-inline-dialog-view", dialogStub);
        var actionsView = this.createView();

        actionsView.$(".add-status").click();
        actionsView.render();
        equal(dialog.remove.callCount, 1, "The AddStatusInlineDialogView was removed");

        actionsView.$(".add-status").click();
        equal(dialogStub.callCount, 2, "A new AddStatusInlineDialogView was created");
    });

});
