(function () {
    var ApplicationLoader = require("workflow-designer/application-loader");
    var Dialog = require('workflow-designer/dialog');
    var jQuery = require("jquery");
    var skate = require("jira/skate");

    var WorkflowDataReader;
    var WorkflowAJAXManager;
    var Messages;

    function requireDependencies() {
        WorkflowDataReader = require("workflow-designer/io/workflow-data-reader");
        WorkflowAJAXManager = require("workflow-designer/io/ajax/workflow-ajax-manager");
        Messages = require("workflow-designer/messages");
    }



    /**
     * Current step id
     *
     * @inner
     * @type {number}
     */
    var currentStepId;

    /**
     * Cache for layout data
     *
     * @inner
     * @type {object}
     */
    var layoutData;

    /**
     * Dialog
     *
     * @inner
     * @type {JIRA.WorkflowDesigner.Dialog}
     */
    var workflowDialog;

    /**
     * Name of the workflow
     *
     * @inner
     * @type {string}
     */
    var workflowName;

    /**
     * Extract a parameter from a workflow image URL.
     *
     * @inner
     * @param {string} imageURL The workflow image URL.
     * @param {string} key The key of the parameter to extract.
     * @return {string} The value corresponding to <tt>key</tt> in <tt>imageURL</tt>'s query string.
     */
    function getImageURLParameter(imageURL, key) {
        var value = parseUri(imageURL).queryKey[key] || "";
        return decodeURIComponent(value.replace(/\+/g, "%20"));
    }

    /**
     * Load the layout data of the current issue's workflow.
     *
     * @inner
     * @return {jQuery.Deferred} A deferred that is resolved when the layout data has loaded.
     */
    function loadLayoutData() {
        var request;

        if (layoutData) {
            return jQuery.Deferred().resolve(layoutData).promise();
        } else {
            request = WorkflowAJAXManager.load(workflowName);
            request.done(function (models, response) {
                layoutData = response;
            });

            return request;
        }
    }

    workflowDialog = new Dialog({
        content: function (callback) {
            var dialog = this;

            ApplicationLoader.load()
                .pipe(requireDependencies)
                .pipe(loadLayoutData)
                .done(function () {
                    dialog.createDialogContent(callback);
                })
                .fail(function (errorMessage) {
                    errorMessage = errorMessage || AJS.I18n.getText("common.forms.ajax.commserror");
                    Messages.showErrorMessage(errorMessage);
                    workflowDialog.hide(true);
                });
        },
        id: "view-workflow-dialog",
        workflowId: function() {
            return workflowName;
        },
        currentStepId: function() {
            return currentStepId;
        },
        layoutData: function() {
            return WorkflowDataReader.read(layoutData);
        }
    });


    var CLASS_TYPE = (skate.type) ? skate.type.CLASSNAME : skate.types.CLASS;

    /**
     * @skate jira-workflow-designer-link
     */
    skate('jira-workflow-designer-link', {
        type: CLASS_TYPE,
        created: function(element) {
            var el = (this instanceof Element) ? this : element;
            el.classList.add('new-workflow-designer');
        },
        attached: function(element) {
            var el = (this instanceof Element) ? this : element;
            var workflowImageURL = el.getAttribute("href");
            workflowName = getImageURLParameter(workflowImageURL, "workflowName");
            currentStepId = parseInt(getImageURLParameter(workflowImageURL, "stepId"), 10);
            layoutData = null;
        },
        events: {
            "click": function(element, e) {
                workflowDialog && workflowDialog.show();
                e.preventDefault();
            }
        }
    });

}());


/**
 * Code related to showing a read-only version of the new workflow designer on the view issue page.
 *
 * @function JIRA.WorkflowDesigner.ViewIssue.replaceViewWorkflowLink
 * @deprecated Since JIRA 7.0. This module is unnecessary now that the {@link skate:jira-workflow-designer-link} component exists.
 */
AJS.namespace("JIRA.WorkflowDesigner.ViewIssue.replaceViewWorkflowLink", null, jQuery.noop);

/**
 * @function JIRA.WorkflowDesigner.Templates.ViewIssue.viewWorkflowLink
 * @deprecated Since JIRA 7.0. This module is unnecessary now that the {@link skate:jira-workflow-designer-link} component exists.
 */
AJS.namespace("JIRA.WorkflowDesigner.Templates.ViewIssue.viewWorkflowLink", null, jQuery.noop);
