define("workflow-designer/viewbox-transformer", [
    "workflow-designer/draw-2d"
], function(
    draw2d
) {
    // Algorithm description for limiting the panning/zooming operations
    //
    // 1. Generate the diagram's `visibleArea`. At least one pixel of this area must be visible at all times.
    //
    //     +--------------------+
    //     |....................|<-----+ Diagram bounding box
    //     |...+------------+...|
    //     |...|            |...|
    //     |...|            |<---------+ visibleArea
    //     |...|            |...|
    //     |...+------------+...|
    //     |....................|
    //     +--------------------+
    //
    // 2. Compute the rectangle (`viewBoxBoundaries`) that contains all the possible positions of the viewBox that meets
    // the previous requirement.
    //
    //
    //     X···············             X···············
    //     ·              ·             ·              ·
    //     ·              ·             ·              ·
    //     ·              ·             ·              ·
    //     ·          +---·-------------·---+          ·
    //     ·          |   ·             ·   |          ·
    //     ················-------------················
    //                |   |             |   |<-----------+ Diagram bounding box
    //                |   |             |<---------------+ visibleArea
    //                |   |             |   |
    //     X···············-------------X···············
    //     ·          |   ·             ·   |          ·
    //     ·          +---·-------------·---+          ·
    //     ·              ·             ·              ·
    //     ·              ·             ·              ·
    //     ·              ·             ·              ·
    //     ················             ················
    //
    //   The dotted rectangles are some of the possible positions for the viewBox. `viewBoxBoundaries` is the rectangle
    //   defined by the four X's.
    //
    // 3. Calculate the viewBox for panning/zooming, but restrict the origin to be located inside `viewBoxBoundaries`

    /**
     * Minimum portion of the workflow that would remain visible on panning/zooming, from 0 to 1.
     * 0 = Diagram will be completely hidden
     * 1 = The whole workflow will remain visible
     *
     * @inner
     * @constant
     * @type {number}
     */
    var MINIMUM_VISIBLE_AREA = 0.25;

    /**
     * Maximum zoom level allowed in the Workflow Designer.
     *
     * @inner
     * @type {number}
     * @constant
     */
    var MAXIMUM_ZOOM_LEVEL = 2;

    /**
     * Minimum zoom level allowed in the Workflow Designer.
     *
     * @inner
     * @type {number}
     * @constant
     */
    var MINIMUM_ZOOM_LEVEL = 0.5;

    /**
     * Padding which is added to the diagram in case view box is smaller than the visible margin.
     *
     * @inner
     * @type {number}
     * @constant
     */
    var PADDING = 10;

    /**
     * Create a rectangle that encloses the part of the diagram that must be visible. At least one pixel of this
     * area must be always visible.
     *
     * @inner
     * @param diagramBoundingBox {draw2d.geo.Rectangle} Minimum bounding box containing all items on the canvas
     * @param {number} viewBoxWidth Width of the viewBox
     * @param {number} viewBoxHeight Height of the viewBox
     * @returns {draw2d.geo.Rectangle} Area that must be visible
     */
    function computeMinimumVisibleArea(diagramBoundingBox, viewBoxWidth, viewBoxHeight) {
        var visibleArea = diagramBoundingBox.clone().scaleByFactor(MINIMUM_VISIBLE_AREA * 2),
            visibleMarginWidth = visibleArea.getLeft() - diagramBoundingBox.getLeft(),
            visibleMarginHeight = visibleArea.getTop() - diagramBoundingBox.getTop();

        // If the view box is smaller than the default visible margin, then expand the visible
        // area to make sure the whole diagram fits into the view box boundaries.
        viewBoxWidth < visibleMarginWidth && visibleArea.scaleWidth(2 * (visibleMarginWidth - viewBoxWidth + PADDING));
        viewBoxHeight < visibleMarginHeight && visibleArea.scaleHeight(2 * (visibleMarginHeight - viewBoxHeight + PADDING));

        return visibleArea;
    }

    /**
     * Compute the viewBox boundaries. This rectangle encloses all the possible positions of the viewBox that allows
     * the a portion of the visibleArea to remain visible.
     *
     * @inner
     * @param {draw2d.geo.Rectangle} diagramBoundingBox Minimum bounding box containing all items on the canvas
     * @param {number} viewBoxWidth Width of the viewBox
     * @param {number} viewBoxHeight Height of the viewBox
     * @returns {draw2d.geo.Rectangle} Rectangle that contains all the possible origins for the viewBox
     */
    function computeViewBoxBoundaries(diagramBoundingBox, viewBoxWidth, viewBoxHeight) {
        return computeMinimumVisibleArea(diagramBoundingBox, viewBoxWidth, viewBoxHeight)
            .translate(-viewBoxWidth, -viewBoxHeight)
            .resize(viewBoxWidth, viewBoxHeight);
    }

    /**
     * Computes the max/min values for a new zoom factor based on the current zoom level.
     *
     * @inner
     * @param {number} newFactor Factor that needs to be corrected.
     * @param {number} currentZoom Current zoom level (a value < 1.0 means zoomed out, > 1.0 is zoomed in).
     * @returns {number} Factor level allowed for this zoom level.
     */
    function computeClampedFactor(newFactor, currentZoom) {
        // Compute min/max factor values allowed at this zoom level
        var maxFactor = MAXIMUM_ZOOM_LEVEL * currentZoom,
            minFactor = MINIMUM_ZOOM_LEVEL * currentZoom,
            clampedFactor = Math.max(Math.min(newFactor, maxFactor), minFactor);

        // Check the factor change is significant enough before do something (i.e. avoid round-off errors)
        if (Math.abs(clampedFactor - 1) > 0.0001) {
            return clampedFactor;
        } else {
            return 1;
        }
    }

    /**
     * @namespace
     */
    return {
        /**
         * Gets the viewBox to apply in a zoom operation.
         *
         * @param options
         * @param {number} options.factor Desired zoom factor to apply (e.g. 0.5 zooms in 2x, 2.0 zooms out 2x).
         * @param {draw2d.geo.Rectangle} options.diagramBoundingBox Minimum bounding box containing all items on the canvas.
         * @param {draw2d.geo.Point} options.target The target point in canvas space. Zoom will be centered in this point.
         * @param {draw2d.geo.Rectangle} options.viewBox Current viewBox.
         * @param {number} options.zoom Current canvas zoom level (value < 1.0 means zoomed out, > 1.0 is zoomed in).
         *
         * @returns {draw2d.geo.Rectangle} ViewBox that matches the requested transformation within the zoom/pan limits.
         */
        getZoomViewBox: function (options) {
            var currentViewBox = options.viewBox,
                diagramBoundingBox = options.diagramBoundingBox,
                factor = computeClampedFactor(options.factor, options.zoom),
                newHeight,
                newWidth,
                percentagePosition,
                target = options.target,
                viewBoxBoundaries,
                viewBoxPosition;

            newWidth = currentViewBox.getWidth() * factor;
            newHeight = currentViewBox.getHeight() * factor;

            viewBoxBoundaries = computeViewBoxBoundaries(diagramBoundingBox, newWidth, newHeight);

            percentagePosition = currentViewBox.getRelativePositionAsPercentage(target);
            viewBoxPosition = new draw2d.geo.Point(
                target.getX() - percentagePosition.x * newWidth,
                target.getY() - percentagePosition.y * newHeight
            );
            viewBoxPosition.setBoundary(
                viewBoxBoundaries.getLeft(),
                viewBoxBoundaries.getTop(),
                viewBoxBoundaries.getRight(),
                viewBoxBoundaries.getBottom()
            );

            return new draw2d.geo.Rectangle(
                viewBoxPosition.getX(),
                viewBoxPosition.getY(),
                newWidth,
                newHeight
            );
        },

        /**
         * Gets the viewBox to apply in a pan operation.
         *
         * @param options
         * @param {draw2d.geo.Rectangle} options.diagramBoundingBox Minimum bounding box containing all items on the canvas.
         * @param {JIRA.WorkflowDesigner.Vector2D} options.displacement Amount of space to displace the viewBox
         * @param {draw2d.geo.Rectangle} options.viewBox Current viewBox.
         *
         * @returns {draw2d.geo.Rectangle} ViewBox that matches the requested transformation within the pan limits.
         */
        getPanViewBox: function (options) {
            var currentViewBox = options.viewBox,
                diagramBoundingBox = options.diagramBoundingBox,
                displacement = options.displacement,
                viewBoxBoundaries,
                viewBoxPosition;

            viewBoxBoundaries = computeViewBoxBoundaries(
                diagramBoundingBox,
                currentViewBox.getWidth(),
                currentViewBox.getHeight()
            );

            viewBoxPosition = currentViewBox.getTopLeft().translate(displacement.x, displacement.y).setBoundary(
                viewBoxBoundaries.getLeft(),
                viewBoxBoundaries.getTop(),
                viewBoxBoundaries.getRight(),
                viewBoxBoundaries.getBottom()
            );

            return new draw2d.geo.Rectangle(
                viewBoxPosition.getX(),
                viewBoxPosition.getY(),
                currentViewBox.getWidth(),
                currentViewBox.getHeight()
            );
        }
    };
});

AJS.namespace("JIRA.WorkflowDesigner.ViewBoxTransformer", null, require("workflow-designer/viewbox-transformer"));