define("workflow-designer/test-utilities", [
    "workflow-designer/workflow-model",
    "workflow-designer/canvas-model",
    "workflow-designer/canvas-view",
    "workflow-designer/canvas",
    "jira/dialog/dialog",
    "jira/util/navigator",
    "aui/inline-dialog",
    "workflow-designer/underscore",
    "jquery",
    "wrm/context-path"
], function (
    WorkflowModel,
    CanvasModel,
    CanvasView,
    Canvas,
    JIRADialog,
    Navigator,
    InlineDialog,
    _,
    jQuery,
    contextPath
) {
    /**
     * @return A container element for a Draw2D canvas.
     */
    function createCanvasContainer() {
        return jQuery("<div/>").appendTo("#qunit-fixture").css("height", 500);
    }


    /**
     * Test utility methods.
     *
     * @namespace
     */
    return {
        /**
         * Call a function in a context where it "appears" to be running on IE.
         *
         *     TestUtilities.asIE(8, function () {
         *         jQuery.browser.msie; // true
         *         jQuery.browser.version // 8
         *     });
         *
         * @param {number} version The version to impersonate.
         * @param {function} callback The function to execute.
         * @param {object} [context=window] The context to execute in.
         */
        asIE: function (version, callback, context) {
            var originalMsie = Navigator.isIE,
                originalVersion = Navigator.majorVersion;

            Navigator.isIE = function() { return true; };
            Navigator.majorVersion = function() { return version; };

            try {
                callback.apply(context || window, []);
            } finally {
                Navigator.isIE = originalMsie;
                Navigator.majorVersion = originalVersion;
            }
        },

        /**
         * Checks if a JIRA.Dialog is currently visible.
         *
         * @return {boolean} Whether a <tt>JIRA.Dialog</tt> is currently visible.
         */
        dialogIsVisible: function () {
            return !!jQuery(".jira-dialog").filter(":visible").length;
        },

        /**
         * Run a test with a fake timer.
         *
         * @param {function} test The test to execute.
         * @param {object} [context=window] The context to execute in.
         */
        fakeTimer: function (test, context) {
            var clock = sinon.useFakeTimers();

            try {
                test.call(context, clock);
            } finally {
                clock.restore();
            }
        },

        /**
         * Set the values of an element's inputs.
         *
         * @param {element} element An element containing inputs.
         * @param {object} values The input values to set; keys are CSS selectors, values are values.
         */
        fill: function (element, values) {
            element = jQuery(element);

            _.each(values, function (value, selector) {
                element.find(selector).val(value);
            });
        },

        /**
         * Convert a request body in the format 'param=value&param2=value2' into an object.
         * Note, this assumes there is a properly formed request body.
         * @param requestBody
         * @returns {object}
         */
        getRequestBodyAsObject: function(requestBody) {
            var queryObject = {};

            if (requestBody) {
                requestBody.replace(/\+/g, ' ').replace(/([^&;=]+)=?([^&;]*)/g,
                    function (m, k, v) {
                        queryObject[decodeURIComponent(k)] = decodeURIComponent(v);
                    });
            }

            return queryObject;
        },

        /**
         * Hide the currently visible <tt>AJS.InlineDialog</tt> and <tt>JIRA.Dialog</tt>.
         */
        removeDialogs: function () {
            InlineDialog.current && InlineDialog.current.hide();
            delete InlineDialog.current;

            JIRADialog.current && JIRADialog.current.hide();
            delete JIRADialog.current;

            jQuery(".aui-blanket, .aui-inline-dialog, .jira-dialog").remove();
        },

        /**
         * Make the sandbox server respond with an error for the specified path.
         * @param deferred
         * @param path
         */
        respondWithErrorAndAssertRejected: function(deferred, path, server) {
            server.respondWith(contextPath() + path, [400, {}, "No deal."]);
            server.respond();

            equal(deferred.state(), "rejected", "The returned deferred was rejected");
            deferred.fail(function (errorMessage) {
                equal(errorMessage, "No deal.", "It was rejected with the error message.");
            });
        },

        /**
         * Set the values of a form's inputs and submit it.
         *
         * @param {element} element An element containing inputs and a submit button.
         * @param {object} values The input values to set; keys are CSS selectors, values are values.
         */
        submit: function (element, values) {
            element = jQuery(element);
            this.fill(element, values);

            // Don't click the submit button as Firefox seems to have a bug
            // where it will only trigger one "submit" event per call stack.
            element.find("form").submit();
        },

        /**
         * Returns {@link JIRA.WorkflowDesigner.Canvas} to make testing mocked
         * AMD modules easier as cashew reewaluates all of them on each require
         * returning different instances.
         */
        getCanvasClass: function () {
            return Canvas;
        },

        /**
         * Create a {@link JIRA.WorkflowDesigner.Canvas} in the QUnit fixture element.
         *
         * @param [options] Options to pass to the constructor.
         * @return {JIRA.WorkflowDesigner.Canvas} A test canvas.
         */
        testCanvas: function (options) {
            options = _.defaults({}, options, {
                el: createCanvasContainer(),
                workflowModel: new WorkflowModel()
            });

            return new Canvas(options).render();
        },

        /**
         * Create a canvas view in the QUnit fixture element.
         *
         * @param [options] Options to pass to the constructor.
         * @return {JIRA.WorkflowDesigner.CanvasView} A test view.
         */
        testCanvasView: function (options) {
            var workflowModel = new WorkflowModel();

            options = _.defaults({}, options, {
                canvasModel: new CanvasModel({}, {
                    workflowModel: workflowModel
                }),
                workflowModel: workflowModel
            });

            var canvasView = new CanvasView(options).render(createCanvasContainer());
            canvasView._validate = sinon.spy();
            return  canvasView;
        },

        /**
         * Create a Draw2D canvas in the QUnit fixture element.
         *
         * @return {JIRA.WorkflowDesigner.Draw2DCanvas} A test Draw2D canvas.
         */
        testDraw2DCanvas: function () {
            return this.testCanvasView().canvas;
        }
    };
});