define("workflow-designer/collection", [
    "workflow-designer/backbone",
    "workflow-designer/underscore"
], function(
    Backbone,
    _
) {
    /**
     * Underscore methods to mix in the Collection object
     *
     * @inner
     * @type {string[]}
     */
    var underscoreMethods = ["chain", "each", "filter", "find", "invoke", "without"];

    /**
     * A collection of <tt>Backbone.Events</tt> objects.
     *
     * Useful for observing events on a collection of objects.
     *
     * @param {Backbone.Events|Backbone.Events[]} [objects] The object(s) in the collection.
     * @extends Backbone.Events
     * @constructor
     */
    var Collection = function (objects) {
        this.length = 0;
        this._objects = [];
        this.reset(objects || []);
    };

    _.extend(Collection.prototype, Backbone.Events,
    /** @lends JIRA.WorkflowDesigner.Collection# */
    {
        /**
         * Add one or more objects to the collection.
         *
         * @param {Backbone.Events|Backbone.Events[]} objects The object(s) to add.
         * @return {JIRA.WorkflowDesigner.Collection} <tt>this</tt>
         */
        add: function (objects) {
            objects = [].concat(objects);

            _.each(objects, function (object) {
                object.on("all", this.trigger, this);
            }, this);

            this._objects = this._objects.concat(objects);
            this.length = this._objects.length;
            return this;
        },

        /**
         * Returns the item located at specified position.
         *
         * @param {number} index The index of the object to retrieve.
         * @return {Backbone.Events} The object at index <tt>index</tt>.
         */
        at: function (index) {
            return this._objects[index];
        },

        /**
         * Remove one or more objects from the collection.
         *
         * @param {Backbone.Events|Backbone.Events[]} objects The object(s) to remove.
         * @return {JIRA.WorkflowDesigner.Collection} <tt>this</tt>
         */
        remove: function (objects) {
            objects = _.isArray(objects) ? objects : [objects];

            _.each(objects, function (object) {
                object.off("all", this.trigger, this);
            }, this);

            this._objects = _.difference(this._objects, objects);
            this.length = this._objects.length;
            return this;
        },

        /**
         * Replace the collection's objects with one or more objects.
         *
         * @param {Backbone.Events|Backbone.Events[]} objects The collection's new object(s).
         * @return {JIRA.WorkflowDesigner.Collection} <tt>this</tt>
         */
        reset: function (objects) {
            this.remove(this._objects);
            return this.add(objects);
        }
    });

    // Mix in Underscore methods, applying them to the objects array.
    _.each(underscoreMethods, function (methodName) {
        Collection.prototype[methodName] = function () {
            return _[methodName].apply(_, [this._objects].concat(_.toArray(arguments)));
        };
    });

    return Collection;
});

AJS.namespace("JIRA.WorkflowDesigner.Collection", null, require("workflow-designer/collection"));