define("workflow-designer/policy/canvas/snap-to-geometry/snap-edit-policy", [
    "workflow-designer/policy/canvas/snapper",
    "workflow-designer/policy/canvas/snap-to-geometry/snap-points",
    "workflow-designer/draw-2d",
    "workflow-designer/underscore"
], function(
    Snapper,
    SnapPoints,
    draw2d,
    _
) {

    return draw2d.policy.canvas.SnapToEditPolicy.extend(
    /** @lends JIRA.WorkflowDesigner.Policy.Canvas.SnapToGeometry.SnapEditPolicy# */
    {
        /**
         * The maximum distance figures will snap.
         *
         * @constant
         * @type {number}
         * @default
         */
        SNAP_THRESHOLD: 3,

        /**
         * Initialise the instance.
         *
         * Store references to dependencies.
         *
         * @classdesc
         * A Draw2D edit policy that snaps figures to align with other figures.
         *
         * Figures must opt in by setting their <tt>snapToGeometry</tt> property to <tt>true</tt>.
         * @constructs
         * @extends draw2d.policy.canvas.SnapToEditPolicy
         */
        init: function () {
            this._super.apply(this, arguments);
        },

        /**
         * Create a <tt>Snapper</tt> for each axis.
         *
         * @param {draw2d.Canvas} canvas A Draw2D canvas.
         * @private
         */
        _createSnappers: function (canvas) {
            var figures;

            if (this.snappers) {
                return;
            }

            figures = this._getFigures(canvas);
            this.snappers = {
                x: new Snapper(this._getXSnapPoints(figures)),
                y: new Snapper(this._getYSnapPoints(figures))
            };
        },

        /**
         * @param {draw2d.Canvas} canvas A Draw2D canvas.
         * @return {draw2d.Figure[]} All figures on <tt>canvas</tt> participating in snap to geometry.
         * @private
         */
        _getFigures: function (canvas) {
            return _.filter(canvas.getFigures().asArray(), function (figure) {
                return !figure.isInDragDrop && figure.snapToGeometry;
            });
        },

        /**
         * @param {draw2d.Figure|draw2d.Figure[]} figures One or more Draw2D figures.
         * @return {number[]} The values of the figures' snap points on the x axis.
         * @private
         */
        _getXSnapPoints: function (figures) {
            return _.pluck(SnapPoints.getXSnapPoints(figures), "value");
        },

        /**
         * @param {draw2d.Figure|draw2d.Figure[]} figures One or more Draw2D figures.
         * @return {number[]} The values of the figures' snap points on the y axis.
         * @private
         */
        _getYSnapPoints: function (figures) {
            return _.pluck(SnapPoints.getYSnapPoints(figures), "value");
        },

        /**
         * Handler for mouse up
         */
        onMouseUp: function () {
            delete this.snappers;
        },

        /**
         * Snap a given figure to other figures.
         *
         * @param {draw2d.Canvas} canvas The canvas.
         * @param {draw2d.Figure} figure The figure that is to be snapped.
         * @param {draw2d.geo.Point} position <tt>figure</tt>'s original position.
         * @return {draw2d.geo.Point} <tt>figure</tt>'s position after snapping.
         */
        snap: function (canvas, figure, position) {
            // Figures must opt in to this policy.
            if (!figure.snapToGeometry) {
                return position;
            }

            this._createSnappers(canvas);
            return this._snapFigure(figure).getTopLeft();
        },

        /**
         * Snaps a figure
         *
         * @param {draw2d.Figure} figure The figure that is to be snapped.
         * @returns {draw2d.geo.Rectangle} New position of the figure.
         * @private
         */
        _snapFigure: function (figure) {
            return figure.getBoundingBox().translate(
                this._snapPoints(this._getXSnapPoints(figure), this.snappers.x),
                this._snapPoints(this._getYSnapPoints(figure), this.snappers.y)
            );
        },

        /**
         * Snaps points
         *
         * @param {number[]} points
         * @param {JIRA.WorkflowDesigner.Policy.Canvas.SnapToGeometry.Snapper} snapper
         * @returns {number}
         * @private
         */
        _snapPoints: function (points, snapper) {
            var minimumOffset = _.chain(points).map(snapper.snap).min(Math.abs).value();
            return Math.abs(minimumOffset) <= this.SNAP_THRESHOLD ? minimumOffset : 0;
        }
    });
});

AJS.namespace("JIRA.WorkflowDesigner.Policy.Canvas.SnapToGeometry.SnapEditPolicy", null, require("workflow-designer/policy/canvas/snap-to-geometry/snap-edit-policy"));