define("workflow-designer/workflow-model", [
    "workflow-designer/status-model",
    "workflow-designer/status-collection",
    "workflow-designer/transition-model",
    "workflow-designer/transition-collection",
    "workflow-designer/looped-transition-container-model",
    "workflow-designer/backbone",
    "workflow-designer/underscore",
    "workflow-designer/permissions-model"
], function(
    StatusModel,
    StatusCollection,
    TransitionModel,
    TransitionCollection,
    LoopedTransitionContainerModel,
    Backbone,
    _,
    PermissionsModel
) {
    return Backbone.Model.extend(
    /** @lends JIRA.WorkflowDesigner.WorkflowModel# */
    {
        defaults: function () {
            return {
                currentStepId: null,
                draft: false,
                name: null,
                liveStepIds: null,
                loadedAt: null,
                loopedTransitionContainer: null,
                statuses: new StatusCollection(),
                transitions: new TransitionCollection(),
                updatedDate: null,
                updateAuthor: null,
                permissions: new PermissionsModel()
            };
        },

        /**
         * @classdesc A JIRA workflow, containing statuses and transitions.
         * @constructs
         * @extends Backbone.Model
         */
        initialize: function () {
            this._triggerLayoutChanged = _.bind(this.trigger, this, "layoutChanged");

            this.listenTo(this.get("statuses"), {
                "change:x change:y": this._triggerLayoutChanged,
                reset: this._detectNewStatus
            });

            this.listenTo(this.get("transitions"), {
                "change:actionId change:sourceAngle change:target change:targetAngle remove": this._triggerLayoutChanged,
                reset: this._detectNewTransition
            });

            this.on("validation:statuses", this._updateValidation("statuses", "stepId"));
            this.on("validation:transitions", this._updateValidation("transitions", "id"));
        },

        _updateValidation: function (collectionName, idProperty) {
            var collection = this.get(collectionName);
            return function (validationResults) {
                collection.forEach(function (status) {
                    status.set("validations", validationResults[status.get(idProperty)]);
                });
            };
        },

        /**
         * Add a status to the workflow.
         *
         * @param {JIRA.WorkflowDesigner.StatusModel|object} status The status or its attributes.
         * @return {JIRA.WorkflowDesigner.StatusModel} The new status.
         */
        addStatus: function (status) {
            var isStatus = status instanceof StatusModel;
            isStatus || (status = new StatusModel(status));
            this.get("statuses").add(status);
            return status;
        },

        /**
         * Add a transition to the workflow.
         *
         * @param {JIRA.WorkflowDesigner.TransitionModel|object} transition The transition or its attributes.
         * @return {JIRA.WorkflowDesigner.TransitionModel} The new transition.
         */
        addTransition: function (transition) {
            var isTransition = transition instanceof TransitionModel, loopedTransitionContainer;
            isTransition || (transition = new TransitionModel(transition));
            this.get("transitions").add(transition);

            if (transition.isLoopedTransition()) {
                loopedTransitionContainer = this.get("loopedTransitionContainer");
                if (!loopedTransitionContainer) {
                    loopedTransitionContainer = new LoopedTransitionContainerModel();
                    this.set("loopedTransitionContainer", loopedTransitionContainer);
                    this.listenTo(loopedTransitionContainer, {
                        "change:x change:y": this._triggerLayoutChanged
                    });
                }
                this.trigger("loopedTransitionAdded");
            }

            this._triggerLayoutChanged();
            return transition;
        },

        /**
         * Trigger a "new:status" event if a single new status is detected.
         *
         * Called when the status collection is reset.
         *
         * @param {JIRA.WorkflowDesigner.StatusCollection} statuses The status collection.
         * @param {object} options
         * @param {JIRA.WorkflowDesigner.StatusModel[]} options.previousModels The previous contents of the collection.
         * @private
         */
        _detectNewStatus: function (statuses, options) {
            var newStatus = this._getNewModel(statuses, options.previousModels);
            newStatus && this.trigger("new:status", newStatus);
        },

        /**
         * Trigger a "new:transition" event if a single new transition is detected.
         *
         * Called when the transition collection is reset.
         *
         * @param {JIRA.WorkflowDesigner.StatusCollection} transitions The transition collection.
         * @param {object} options
         * @param {JIRA.WorkflowDesigner.StatusModel[]} options.previousModels The previous contents of the collection.
         * @private
         */
        _detectNewTransition: function (transitions, options) {
            var newTransition = this._getNewModel(transitions, options.previousModels);
            newTransition && this.trigger("new:transition", newTransition);
        },

        /**
         * @param {JIRA.WorkflowDesigner.StatusModel} status A status.
         * @returns {JIRA.WorkflowDesigner.TransitionModel|undefined} The global
         *     transition associated with `status` or `undefined`.
         */
        getGlobalTransitionForStatus: function (status) {
            return this.get("transitions").findWhere({
                globalTransition: true,
                target: status
            });
        },

        /**
         * @returns All initial transitions.
         * @private
         */
        _getInitialTransitions: function () {
            return this.get("transitions").filter(function (transition) {
                return transition.isInitial();
            });
        },

        /**
         * @returns Statuses that are direct destinations of initial transitions.
         */
        getInitialTransitionDestinations: function () {
            return _.map(this._getInitialTransitions(), function (transition) {
                return transition.get("target");
            });
        },

        /**
         * Get the one new model in an array of models.
         *
         * @param {Backbone.Collection} newModels The new models.
         * @param {Backbone.Model[]} oldModels The old models.
         * @return {Backbone.Model|undefined} The one new model or undefined.
         * @private
         */
        _getNewModel: function (newModels, oldModels) {
            var isInitialLoad = oldModels.length === 0;

            newModels = newModels.filter(function (newModel) {
                return _.every(oldModels, function (oldModel) {
                    return newModel.id !== oldModel.id;
                });
            });

            if (newModels.length === 1 && !isInitialLoad) {
                return newModels[0];
            }
        },

        /**
         * @param {string} id The ID of the status to retrieve.
         * @return {JIRA.WorkflowDesigner.StatusModel|undefined} The status with
         *     the given ID or `undefined`.
         */
        getStatus: function (id) {
            return this.get("statuses").get(id);
        },

        removeLoopedTransitionContainer: function () {
            this.unset("loopedTransitionContainer");
            this._triggerLayoutChanged();
        },

        /**
         * Removes a transition model from this workflow model.
         *
         * @param {JIRA.WorkflowDesigner.TransitionModel} transition Transition model to remove.
         */
        removeTransition: function (transition) {
            this.get("transitions").remove(transition);
            if (transition.isLoopedTransition()) {
                this.trigger("loopedTransitionRemoved");
            }
        },

        /**
         * Reset the model to a given state.
         *
         * Triggers "reset:before" and "reset:after" events.
         *
         * @param {object} attributes
         * @param {boolean} [attributes.draft] Whether the workflow is a draft.
         * @param {string} [attributes.name] The workflow's name.
         * @param {JIRA.WorkflowDesigner.StatusModel[]} attributes.statuses The workflow's statuses.
         * @param {JIRA.WorkflowDesigner.TransitionModel[]} attributes.transitions The workflow's transitions.
         */
        reset: function (attributes) {
            this.trigger("reset:before");
            this.get("permissions").set(attributes.permissions);
            this.get("statuses").reset(attributes.statuses);
            this.get("transitions").reset(attributes.transitions);
            this.set(_.omit(attributes, "statuses", "transitions", "permissions"));
            this.get("loopedTransitionContainer") && this.listenTo(this.get("loopedTransitionContainer"), {
                "change:x change:y": this._triggerLayoutChanged
            });
            this.trigger("reset:after");
        },

        /**
         * @param {JIRA.WorkflowDesigner.StatusModel} status A status.
         * @returns {boolean} Whether `status` is the target of a global transition.
         */
        statusHasGlobalTransition: function (status) {
            return !!this.getGlobalTransitionForStatus(status);
        },

        /**
         * A status can't be deleted if it's a draft workflow and the same status exists on the live workflow.
         *
         * @param {JIRA.WorkflowDesigner.StatusModel} status Status model.
         * @returns {boolean} Whether the given status can be deleted.
         */
        statusIsDeletable: function (status) {
            var isDraft = this.get("draft"),
                isLiveStatus = _.contains(this.get("liveStepIds"), status.get("stepId"));

            return !(isDraft && isLiveStatus);
        },

        /**
         * Updates the source step of the specified transition.
         *
         * @param {object} options
         * @param {number} options.sourceAngle Source angle.
         * @param {number} options.sourceStepId The ID of the transition's source step.
         * @param {JIRA.WorkflowDesigner.TransitionModel} options.transition The transition.
         */
        updateTransitionSource: function (options) {
            var sourceStatus = this.get("statuses").findWhere({
                initial: false,
                stepId: options.sourceStepId
            });

            options.transition.setSource(sourceStatus, options.sourceAngle);
        },

        /**
         * Updates the target step of the specified transition. Will update multiple transition models in case the transition is common.
         *
         * @param {object} options
         * @param {number} options.targetAngle Target angle.
         * @param {number} options.targetStepId The ID of the transition's target step.
         * @param {JIRA.WorkflowDesigner.TransitionModel} options.transition The transition.
         */
        updateTransitionTargets: function (options) {
            var actionId = options.transition.get("actionId"),
                targetStatus;

            targetStatus = this.get("statuses").findWhere({
                initial: false,
                stepId: options.targetStepId
            });

            options.transition.setTarget(targetStatus, options.targetAngle);

            // Other connections of the same common transition.
            this.get("transitions").chain()
                    .filter(function (transition) { return transition !== options.transition && transition.get("actionId") === actionId; })
                    .invoke("setTarget", targetStatus);
        }
    });
});

AJS.namespace("JIRA.WorkflowDesigner.WorkflowModel", null, require("workflow-designer/workflow-model"));