define("workflow-designer/transition-model", [
    "workflow-designer/transition-option-type-collection",
    "workflow-designer/backbone",
    "workflow-designer/underscore",
    "jquery"
], function(
    TransitionOptionTypeCollection,
    Backbone,
    _,
    jQuery
) {
    return Backbone.Model.extend(
    /** @lends JIRA.WorkflowDesigner.TransitionModel# */
    {
        defaults: function () {
            return {
                id: null,
                actionId: null,
                description: null,
                globalTransition: false,
                loopedTransition: false,
                name: null,
                screenId: null,
                screenName: null,
                source: null,
                sourceAngle: null,
                target: null,
                targetAngle: null,
                transitionOptions: new TransitionOptionTypeCollection(),
                validations: null
            };
        },

        /**
         * @classdesc
         * A transition to a target status.
         *
         * Global transitions don't have a source status, normal transitions do (and they may be the same).
         * @constructs
         * @extends Backbone.Model
         */
        initialize: function() {
            this._createId();

            this.on("change:actionId change:source change:target", _.bind(this._createId, this));
        },

        /**
         * Sets the unique ID of the transition.
         *
         * @private
         */
        _createId: function() {
            var model = this;
            function getId(entityName) {
                return model.get(entityName) && model.get(entityName).get("id") || null;
            }

            var sourceId, targetId;
            if (this.isLoopedTransition()) {
                sourceId = targetId = "S<-1>";
            } else if (this.isGlobalTransition()) {
                sourceId = targetId = getId("target");
            } else {
                sourceId = getId("source");
                targetId = getId("target");
            }

            if (this.getActionId() && sourceId && targetId) {
                var prefix = this.get("source") && this.get("source").isInitial() ? "IA" : "A";
                this.set("id", prefix + "<" + this.getActionId() + ":" + sourceId + ":" + targetId + ">");
            }
        },

        /**
         * @returns {boolean} Whether the transition is global
         */
        isGlobalTransition: function () {
            return this.get("globalTransition");
        },

        /**
         * @returns {boolean}  Whether the transition is a looped transition.
         */
        isLoopedTransition: function () {
            return this.get("loopedTransition");
        },

        /**
         * @returns {boolean} Whether the transition is the initial transition
         */
        isInitial: function () {
            return !this.isGlobalTransition() && !this.isLoopedTransition() && this.get("source").isInitial();
        },

        /**
         * @returns {string} The `actionId` for this transition
         */
        getActionId: function () {
            return this.get("actionId");
        },

        /**
         * @returns {number} The `stepId` of the transition's source
         */
        getSourceStepId: function () {
            var source = this.get("source");
            return source && source.get("stepId");
        },

        /**
         * @returns {number} The `stepId` of the transition's target
         */
        getTargetStepId: function () {
            var target = this.get("target");
            return target && target.get("stepId");
        },

        /**
         * Sets the source status and angle.
         * @param {JIRA.WorkflowDesigner.StatusModel} source New source status.
         * @param {number} targetAngle New source angle.
         */
        setSource: function(source, sourceAngle) {
            this.set({
                source: source,
                sourceAngle: sourceAngle
            });
        },

        /**
         * Sets the target status and angle.
         * @param {JIRA.WorkflowDesigner.StatusModel} target New target status.
         * @param {number} targetAngle New target angle.
         */
        setTarget: function(target, targetAngle) {
            this.set({
                target: target,
                targetAngle: targetAngle
            });
        },

        // We don't want to use Backbone's synchronisation logic.
        sync: jQuery.noop
    });
});

AJS.namespace("JIRA.WorkflowDesigner.TransitionModel", null, require("workflow-designer/transition-model"));