define("workflow-designer/layout/connection/gapped-connection", [
    "workflow-designer/direction",
    "workflow-designer/draw-2d"
], function(
    Direction,
    draw2d
) {
    return draw2d.Connection.extend(
    /** @lends JIRA.WorkflowDesigner.Layout.Connection.GappedConnection.prototype */
    {
        /**
         * Initialises the GappedConnection.
         *
         * @classdesc A Connection with gaps in both ends.
         * @extends draw2d.Connection
         * @constructs
         * @param {number} sourceGap Gap distance in pixels for the source (i.e. the start point of the connection).
         * @param {number} targetGap Gap distance in pixels for the target (i.e. the end point of the connection).
         */
        init: function (sourceGap, targetGap) {
            this._super();
            this._sourceGap = sourceGap;
            this._targetGap = targetGap;
        },

        /**
         * Gets the transformation based on a direction.
         *
         * @private
         * @param {number} direction Direction of the connection (valid values: JIRA.WorkflowDesigner.Direction.*).
         * @param {number} gap Gap to apply.
         * @returns {{x: number, y: number}} Displacement to apply.
         */
        _getTransformation: function (direction, gap) {
            var dx = 0,
                dy = 0;

            switch (direction) {
            case Direction.UP:
                dy = -gap;
                break;
            case Direction.RIGHT:
                dx = gap;
                break;
            case Direction.DOWN:
                dy = gap;
                break;
            case Direction.LEFT:
                dx = -gap;
                break;
            }

            return { x: dx, y: dy };
        },

        /**
         * Computes the gap for the endpoint, based on the connection direction.
         *
         * @returns {{x: number, y: number}} Displacement applied to the endpoint.
         */
        getEndPointGapTransformation: function () {
            return this._getTransformation(this.getTarget().getConnectionDirection(), this._targetGap);
        },

        /**
         * Computes the gap for the source based on the connection direction.
         *
         * @returns {{x: number, y: number}} Displacement applied to the startpoint.
         */
        getStartPointGapTransformation: function () {
            return this._getTransformation(this.getSource().getConnectionDirection(), this._sourceGap);
        },

        /**
         * Sets the endpoint of the connection.
         *
         * This method will translate the position to leave a gap between the desired position and the actual position.
         *
         * @param {number} x Target x-coordinate.
         * @param {number} y Target y-coordinate.
         */
        setEndPoint: function (x, y) {
            var gap;
            if (this.isMoving) {
                this._super(x, y);
            } else {
                gap = this.getEndPointGapTransformation();
                this._super(x + gap.x, y + gap.y);
            }
        },

        /**
         * Sets the startpoint of the connection.
         *
         * This method will translate the position to leave a gap between the desired position and the actual position.
         *
         * @param {number} x Target x-coordinate.
         * @param {number} y Target y-coordinate.
         */
        setStartPoint: function (x, y) {
            var gap;
            if (this.isMoving) {
                this._super(x, y);
            } else {
                gap = this.getStartPointGapTransformation();
                this._super(x + gap.x, y + gap.y);
            }
        },

        /**
         * Gets the endpoint of the connection.
         *
         * This method will correct the computed endpoint to include the gap.
         *
         * @returns {draw2d.Point} Corrected point.
         */
        getEndPoint: function () {
            var gap,
                originalEndPoint = this._super();

            // Don't add the gap if the connection is being moved to another port.
            if (this.isMoving) {
                return originalEndPoint;
            }

            gap = this.getEndPointGapTransformation();
            return originalEndPoint.clone().translate(gap.x, gap.y);
        },

        /**
         * Gets the startpoint of the connection.
         *
         * This method will correct the computed endpoint to include the gap.
         *
         * @returns {draw2d.Point} Corrected point.
         */
        getStartPoint: function () {
            var gap = this.getStartPointGapTransformation(),
                originalStartPoint = this._super();

            // Don't add the gap if the connection is being moved to another port.
            if (this.isMoving) {
                return originalStartPoint;
            }

            return originalStartPoint.clone().translate(gap.x, gap.y);
        }
    });

});

AJS.namespace("JIRA.WorkflowDesigner.Layout.Connection.GappedConnection", null, require("workflow-designer/layout/connection/gapped-connection"));