define("workflow-designer/io/ajax/workflow-transitions-ajax-manager", [
    "workflow-designer/io/ajax/ajax-manager",
    "workflow-designer/io/error-handler",
    "workflow-designer/underscore",
    "jquery",
    "wrm/context-path"
], function(
    AJAXManager,
    ErrorHandler,
    _,
    jQuery,
    contextPath
) {
    var URL = contextPath() + "/rest/workflowDesigner/latest/workflows/transitions";

    /**
     * Make a request to the transitions REST resource.
     *
     * @param {object} options Options to pass to `jQuery.ajax`.
     * @returns {jQuery.Deferred} A deferred that is resolved on success.
     */
    function makeRequest(options) {
        var deferred = jQuery.Deferred(),
            request;

        request = this._makeRequest(_.defaults({}, options, {
            dataType: "json",
            url: URL
        }));

        request.done(deferred.resolve);
        request.fail(function (response, status) {
            deferred.reject(ErrorHandler.getErrorMessage(response, status));
        });

        return deferred.promise();
    }

    /**
     * Handles manipulating a workflow's transitions.
     *
     * @namespace
     * @extends JIRA.WorkflowDesigner.IO.AJAX.AJAXManager
     */
    return _.extend({}, AJAXManager,
    /** @lends JIRA.WorkflowDesigner.IO.AJAX.WorkflowTransitionsAJAXManager */
    {
        /**
         * Add a common transition to a workflow.
         *
         * @param {object} options
         * @param {string} options.sourceStatusId The ID of the transition's source status.
         * @param {number} options.transitionId The ID of the transition.
         * @param {string} options.workflowName The workflow's name.
         * @returns {jQuery.Deferred} A promise that is resolved on completion.
         */
        addCommonTransition: function (options) {
            return makeRequest.call(this, {
                data: _.pick(options, "sourceStatusId", "transitionId", "workflowName"),
                type: "post",
                url: URL + "/common"
            });
        },

        /**
         * Add a transition to a workflow.
         *
         * @param {object} options
         * @param {string} options.description The transition's description.
         * @param {string} options.name The transition's name.
         * @param {number} options.screenId The ID of the transition's screen.
         * @param {number} options.sourceStepId The ID of the transition's source step.
         * @param {number} options.targetStepId The ID of the transition's target step.
         * @param {string} options.workflowName The workflow's name.
         * @return {jQuery.Deferred} A deferred that is resolved on success.
         */
        addTransition: function (options) {
            return makeRequest.call(this, {
                data: options,
                type: "post"
            });
        },

        /**
         * Deletes transition from a workflow.
         *
         * @param {object} options
         * @param {number} options.sourceStepId The ID of the transition's source step.
         * @param {string} options.transitionId The transition's ID.
         * @param {string} options.workflowName The workflow's name.
         * @return {jQuery.Deferred} A deferred that is resolved on success.
         */
        deleteTransition: function (options) {
            return makeRequest.call(this, {
                data: options,
                type: "delete"
            });
        },

        /**
         * Update an existing transition on the workflow.
         *
         * @param {object} options
         * @param {string} options.description The transition's description.
         * @param {string} options.name The transition's name.
         * @param {string} options.screenId The transition's screen ID.
         * @param {number} options.sourceStepId The ID of the transition's source step.
         * @param {string} options.transitionId The transition's ID.
         * @param {string} options.workflowName The workflow's name.
         * @return {jQuery.Deferred} A deferred that is resolved on success.
         */
        updateTransition: function (options) {
            return makeRequest.call(this, {
                data: options,
                type: "put"
            });
        },

        /**
         * Update the source step of an existing transition on the workflow.
         *
         * @param {object} options
         * @param {number} options.newSourceStatusId The ID of the transition's new source status.
         * @param {number} options.originalSourceStatusId The ID of the transition's original source status.
         * @param {string} options.transitionId The transition's ID.
         * @param {string} options.workflowName The workflow's name.
         * @return {jQuery.Deferred} A deferred that is resolved on success.
         */
        updateTransitionSource: function (options) {
            return makeRequest.call(this, {
                data: options,
                type: "post",
                url: URL + "/source"
            });
        },

        /**
         * Update the target step of an existing transition on the workflow.
         *
         * @param {object} options
         * @param {number} options.targetStatusId The ID of the transition's target status.
         * @param {string} options.transitionId The transition's ID.
         * @param {string} options.workflowName The workflow's name.
         * @return {jQuery.Deferred} A deferred that is resolved on success.
         */
        updateTransitionTarget: function (options) {
            return makeRequest.call(this, {
                data: options,
                type: "post",
                url: URL + "/target"
            });
        }
    });
});

AJS.namespace("JIRA.WorkflowDesigner.IO.AJAX.WorkflowTransitionsAJAXManager", null, require("workflow-designer/io/ajax/workflow-transitions-ajax-manager"));