define("workflow-designer/io/ajax/workflow-looped-transitions-ajax-manager", [
    "workflow-designer/io/ajax/ajax-manager",
    "workflow-designer/io/error-handler",
    "workflow-designer/underscore",
    "jquery",
    "wrm/context-path"
], function(
    AJAXManager,
    ErrorHandler,
    _,
    jQuery,
    contextPath
) {
    var URL = contextPath() + "/rest/workflowDesigner/latest/workflows/loopedTransitions";

    /**
     * Make a request to the looped transitions REST resource.
     *
     * @param {object} options Options to pass to `jQuery.ajax`.
     * @returns {jQuery.Deferred} A deferred that is resolved on success.
     */
    function makeRequest(options) {
        var deferred = jQuery.Deferred(),
            request;

        request = this._makeRequest(_.defaults({}, options, {
            dataType: "json",
            url: URL
        }));

        request.done(deferred.resolve);
        request.fail(function (response, status) {
            deferred.reject(ErrorHandler.getErrorMessage(response, status));
        });

        return deferred.promise();
    }

    /**
     * Handles manipulating a workflow's looped transitions.
     *
     * @namespace
     * @extends JIRA.WorkflowDesigner.IO.AJAX.AJAXManager
     */
    return _.extend({}, AJAXManager,
        /** @lends JIRA.WorkflowDesigner.IO.AJAX.WorkflowLoopedTransitionsAJAXManager */
        {
            /**
             * Add a looped transition to a workflow.
             *
             * @param {object} options
             * @param {string} options.description The global transition description.
             * @param {string} options.name The global transition name.
             * @param {string} options.screenId The global transition screen id.
             * @param {string} options.workflowName The workflow's name.
             * @returns {jQuery.Deferred} A promise that is resolved on completion.
             */
            addLoopedTransition: function (options) {
                return makeRequest.call(this, {
                    data: _.pick(options, "workflowName", "name", "description", "screenId"),
                    type: "post"
                });
            },

            /**
             * Deletes a looped transition from a workflow.
             *
             * @param {object} options
             * @param {string} options.transitionId The transition's ID.
             * @param {string} options.workflowName The workflow's name.
             * @return {jQuery.Deferred} A deferred that is resolved on success.
             */
            deleteLoopedTransition: function (options) {
                return makeRequest.call(this, {
                    data: options,
                    type: "delete"
                });
            },

            /**
             * Update an existing looped transition on the workflow.
             *
             * @param {object} options
             * @param {string} options.description The transition's description.
             * @param {string} options.name The transition's name.
             * @param {string} options.screenId The transition's screen ID.
             * @param {string} options.transitionId The transition's ID.
             * @param {string} options.workflowName The workflow's name.
             * @return {jQuery.Deferred} A deferred that is resolved on success.
             */
            updateLoopedTransition: function (options) {
                return makeRequest.call(this, {
                    data: options,
                    type: "put"
                });
            }
        });

});

AJS.namespace("JIRA.WorkflowDesigner.IO.AJAX.WorkflowLoopedTransitionsAJAXManager", null, require("workflow-designer/io/ajax/workflow-looped-transitions-ajax-manager"));