define("workflow-designer/io/ajax/workflow-global-transitions-ajax-manager", [
    "workflow-designer/io/ajax/ajax-manager",
    "workflow-designer/io/workflow-data-reader",
    "workflow-designer/io/error-handler",
    "workflow-designer/underscore",
    "jquery",
    "wrm/context-path"
], function(
    AJAXManager,
    WorkflowDataReader,
    ErrorHandler,
    _,
    jQuery,
    contextPath
) {
    /**
     * Handles manipulating a workflow's global transitions.
     *
     * @namespace
     * @extends JIRA.WorkflowDesigner.IO.AJAX.AJAXManager
     */
    return _.extend({}, AJAXManager,
    /** @lends JIRA.WorkflowDesigner.IO.AJAX.WorkflowGlobalTransitionsAJAXManager */
    {
        /**
         * Add a global transition to a workflow. The core difference with this method
         * and that below is that this doesn't return the full workflow.
         *
         * @param {object} options
         * @param {string} options.description The global transition description.
         * @param {string} options.name The global transition name.
         * @param {string} options.screenId The global transition screen id.
         * @param {string} options.statusId The ID of the status to add.
         * @param {string} options.workflowName The workflow's name.
         * @return {jQuery.Deferred} A deferred that is resolved on success.
         */
        addGlobalTransition: function (options) {
            var deferred = jQuery.Deferred(),
                request;

            request = this._makeRequest({
                data: _.pick(options, "statusId", "workflowName", "name", "description", "screenId"),
                dataType: "json",
                type: "post",
                url: contextPath() + "/rest/workflowDesigner/latest/workflows/globalTransitions/simple"
            });


            request.done(deferred.resolve);
            request.fail(function (response, status) {
                deferred.reject(ErrorHandler.getErrorMessage(response, status));
            });

            return deferred.promise();
        },

        /**
         * Add a global transition to a workflow.
         *
         * @param {object} options
         * @param {string} options.description The global transition description.
         * @param {string} options.name The global transition name.
         * @param {string} options.screenId The global transition screen id.
         * @param {string} options.statusId The ID of the status to add.
         * @param {string} options.workflowName The workflow's name.
         * @return {jQuery.Deferred} A deferred that is resolved on success.
         */
        createGlobalTransition: function (options) {
            var deferred = jQuery.Deferred(),
                request;

            request = this._makeRequest({
                data: _.pick(options, "statusId", "workflowName", "name", "description", "screenId"),
                dataType: "json",
                type: "post",
                url: contextPath() + "/rest/workflowDesigner/latest/workflows/globalTransitions"
            });

            request.done(function (response) {
                deferred.resolve(WorkflowDataReader.read(response));
            });

            request.fail(function (response, status) {
                deferred.reject(ErrorHandler.getErrorMessage(response, status));
            });

            return deferred.promise();
        },

        /**
         * Deletes a global transition from a workflow.
         *
         * @param {object} options The global transition details.
         * @param {string} options.transitionId The global transition's id.
         * @param {string} options.workflowName The workflow's name.
         * @return {jQuery.Deferred} A deferred that is resolved on success.
         */
        deleteGlobalTransition: function (options) {
            var deferred = jQuery.Deferred(),
                request;

            request = this._makeRequest({
                data: _.pick(options, "transitionId", "workflowName"),
                dataType: "json",
                type: "delete",
                url: contextPath() + "/rest/workflowDesigner/latest/workflows/globalTransitions"
            });

            request.done(deferred.resolve);
            request.fail(function (response, status) {
                deferred.reject(ErrorHandler.getErrorMessage(response, status));
            });

            return deferred.promise();
        },

        /**
         * Update a global transition in a workflow.
         *
         * @param {object} options The global transition details.
         * @param {number} options.transitionId The global transition's id.
         * @param {string} options.name The global transition's name.
         * @param {string} options.description The global transition's description.
         * @param {string} options.screenId The global transition name.
         * @param {string} options.workflowName The workflow's name.
         * @return {jQuery.Deferred} A deferred that is resolved on success.
         */
        updateGlobalTransition: function(options) {
            var deferred = jQuery.Deferred(),
                request;

            request = this._makeRequest({
                data: _.pick(options, "transitionId", "name", "description", "screenId", "workflowName"),
                dataType: "json",
                type: "put",
                url: contextPath() + "/rest/workflowDesigner/latest/workflows/globalTransitions"
            });

            request.done(deferred.resolve);
            request.fail(function (response, status) {
                deferred.reject(ErrorHandler.getErrorMessage(response, status));
            });

            return deferred.promise();
        }
    });
});

AJS.namespace("JIRA.WorkflowDesigner.IO.AJAX.WorkflowGlobalTransitionsAJAXManager", null, require("workflow-designer/io/ajax/workflow-global-transitions-ajax-manager"));