AJS.test.require([
    "com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer",
    "com.atlassian.jira.plugins.jira-workflow-designer:test-resources"
], function () {

    var WorkflowGlobalTransitionsAJAXManager = require('workflow-designer/io/ajax/workflow-global-transitions-ajax-manager');
    var WorkflowDataReader = require("workflow-designer/io/workflow-data-reader");
    var TestUtilities = require("workflow-designer/test-utilities");
    var contextPath = require("wrm/context-path");

    module("WorkflowGlobalTransitionsAJAXManager", {
        setup: function () {
            this.sandbox = sinon.sandbox.create({
                useFakeServer: true
            });
            this.workflowGlobalTransitionsAJAXManager = WorkflowGlobalTransitionsAJAXManager;
        },

        teardown: function () {
            this.sandbox.restore();
        }
    });

    test("create() responds successfully", function () {
        var deferred,
            expectedRequestBody,
            sentRequestBody,
            response = {},
            workflowDataReaderStub = this.sandbox.stub(WorkflowDataReader, "read");

        deferred = this.workflowGlobalTransitionsAJAXManager.createGlobalTransition({
            statusId: "Open",
            workflowName: "JIRA",
            name: "name",
            description: "description"
        });

        workflowDataReaderStub.returns(response);
        this.sandbox.server.respondWith(contextPath() + "/rest/workflowDesigner/latest/workflows/globalTransitions", [200, {}, "{}"]);
        this.sandbox.server.respond();

        expectedRequestBody = {
            statusId: "Open",
            workflowName: "JIRA",
            name: "name",
            description: "description"
        };
        sentRequestBody = TestUtilities.getRequestBodyAsObject(this.sandbox.server.requests[0].requestBody);
        deepEqual(sentRequestBody, expectedRequestBody, "The request body was correct");

        equal(workflowDataReaderStub.callCount, 1, "The AJAX response was passed to WorkflowDataReader.read()");
        deepEqual(workflowDataReaderStub.args[0][0], {}, "It was passed the correct arguments");

        equal(deferred.state(), "resolved", "The returned deferred was resolved.");
        deferred.done(function (data) {
            ok(data === response, "It was resolved with the output of WorkflowDataReader.read()");
        });
    });

    test("create() rejects the returned deferred on error", function () {
        var deferred = this.workflowGlobalTransitionsAJAXManager.createGlobalTransition({
            statusId: "1",
            workflowName: "JIRA",
            name: "name",
            description: "description"
        });

        TestUtilities.respondWithErrorAndAssertRejected(
            deferred, "/rest/workflowDesigner/latest/workflows/globalTransitions", this.sandbox.server);
    });


    test("deleteGlobalTransition() resolves deferred result on success", function () {
        var deferred, expectedRequestBody, sentRequestBody;

        deferred = this.workflowGlobalTransitionsAJAXManager.deleteGlobalTransition({
            transitionId: 100,
            unknownParameter: true,
            workflowName: "Workflow"
        });

        this.sandbox.server.respondWith(contextPath() + "/rest/workflowDesigner/latest/workflows/globalTransitions", [200, {}, ""]);
        this.sandbox.server.respond();

        expectedRequestBody = {
            transitionId: "100",
            workflowName: "Workflow"
        };
        sentRequestBody = TestUtilities.getRequestBodyAsObject(this.sandbox.server.requests[0].requestBody);
        deepEqual(sentRequestBody, expectedRequestBody, "The request body was correct");
    });

    test("deleteGlobalTransition() rejects the returned deferred on error", function () {
        var deferred = this.workflowGlobalTransitionsAJAXManager.deleteGlobalTransition({
            transitionId: 100,
            workflowName: "Workflow"
        });

        TestUtilities.respondWithErrorAndAssertRejected(
            deferred, "/rest/workflowDesigner/latest/workflows/globalTransitions", this.sandbox.server);
    });

    test("updateGlobalTransition() resolves deferred result on success", function () {
        var deferred,
            sentRequestBody,
            expectedRequestBody;

        deferred = this.workflowGlobalTransitionsAJAXManager.updateGlobalTransition({
            transitionId: "100",
            name: "name",
            description: "description",
            workflowName: "Workflow",
            screenId: "1",
            extraParam: "a new beginning"
        });
        this.sandbox.server.respondWith(contextPath() + "/rest/workflowDesigner/latest/workflows/globalTransitions", [200, {}, ""]);
        this.sandbox.server.respond();
        equal(deferred.state(), "resolved", "The returned deferred was resolved.");

        expectedRequestBody = {
            transitionId: "100",
            name: "name",
            description: "description",
            workflowName: "Workflow",
            screenId: "1"
        };
        sentRequestBody = TestUtilities.getRequestBodyAsObject(this.sandbox.server.requests[0].requestBody);
        deepEqual(sentRequestBody, expectedRequestBody, "The request body for update was correct.");
    });

    test("updateGlobalTransition() rejects the returned deferred on error", function () {
        var deferred = this.workflowGlobalTransitionsAJAXManager.updateGlobalTransition({
            transitionId: 100,
            name: "name",
            description: "description",
            workflowName: "Workflow",
            screenId: 1
        });

        TestUtilities.respondWithErrorAndAssertRejected(
            deferred, "/rest/workflowDesigner/latest/workflows/globalTransitions", this.sandbox.server);
    });

});
