define("workflow-designer/io/ajax/workflow-ajax-manager", [
    "workflow-designer/io/workflow-data-reader",
    "workflow-designer/io/workflow-data-writer",
    "workflow-designer/io/ajax/ajax-manager",
    "workflow-designer/io/error-handler",
    "workflow-designer/underscore",
    "jquery",
    "wrm/context-path"
], function(
    WorkflowDataReader,
    WorkflowDataWriter,
    AJAXManager,
    ErrorHandler,
    _,
    jQuery,
    contextPath
) {
    /**
     * Handles workflow information via AJAX.
     *
     * @namespace
     * @extends JIRA.WorkflowDesigner.IO.AJAX.AJAXManager
     */
    return _.extend({}, AJAXManager,
    /** @lends JIRA.WorkflowDesigner.IO.AJAX.WorkflowAJAXManager */
    {
        /**
         * Discard the draft of the workflow.
         * Avoid calling this function multiple times or in combination with {@link JIRA.WorkflowDesigner.IO.AJAX.WorkflowAJAXManager.publish} as this can result in conflicting behaviour.
         *
         * @param {string} workflowName The name of the workflow of which the draft should be discarded.
         * @returns {jQuery.Deferred} A deferred that is resolved on success.
         */
        discard: function (workflowName) {
            var deferred = jQuery.Deferred(),
                instance = this;

            this._publishOrDiscardRequest = this._makeRequest({
                contentType: "application/json",
                data: workflowName,
                success: deferred.resolve,
                type: "POST",
                url: contextPath() + "/rest/workflowDesigner/latest/workflows/discardDraft"
            });

            this._publishOrDiscardRequest.fail(function (response, status) {
                deferred.reject(ErrorHandler.getErrorMessage(response, status));
            });

            this._publishOrDiscardRequest.always(function () {
                delete instance._publishOrDiscardRequest;
            });

            return deferred.promise();
        },

        /**
         * Load the workflow with the given name.
         *
         * @param {string} name The name of the workflow to load.
         * @param {boolean} draft Whether to load the draft or active layout.
         * @return {jQuery.Deferred} A promise that is resolved with the parsed workflow information and the JSON response.
         */
        load: function (name, draft) {
            var deferred = jQuery.Deferred(),
                request;

            request = this._makeRequest({
                data: {
                    name: name,
                    draft: !!draft
                },
                dataType: "json",
                url: contextPath() + "/rest/workflowDesigner/latest/workflows"
            });

            request.done(function (response) {
                var data,
                    layout = response.layout;

                // The following lines for resolving the data are common between ProjectConfigAJAXManager and
                // WorkflowAJAXManager. They should be kept in sync.
                data = WorkflowDataReader.read(layout);
                data = _.extend(data, {
                    liveStepIds: response.liveStepIds,
                    permissions: response.workflowPermissions

                });

                deferred.resolve(data, response);
            });

            request.fail(function (response, status) {
                deferred.reject(ErrorHandler.getErrorMessage(response, status));
            });

            return deferred.promise();
        },

        /**
         * Publish the draft of the workflow.
         * Avoid calling this function multiple times or in combination with {@link JIRA.WorkflowDesigner.IO.AJAX.WorkflowAJAXManager.discard} as this can result in conflicting behaviour.
         *
         * @param {JIRA.WorkflowDesigner.WorkflowModel} workflowModel The workflow to publish.
         * @return {jQuery.Deferred} A promise that is resolved on success.
         */
        publish: function (workflowModel) {
            var deferred = jQuery.Deferred(),
                instance = this,
                workflowData = WorkflowDataWriter.write(workflowModel);

            this._publishOrDiscardRequest = this._makeRequest({
                contentType: "application/json",
                data: JSON.stringify(workflowData),
                success: deferred.resolve,
                type: "POST",
                url: contextPath() + "/rest/workflowDesigner/latest/workflows/publishDraft"
            });

            this._publishOrDiscardRequest.fail(function (response, status) {
                deferred.reject(ErrorHandler.getErrorMessage(response, status));
            });

            this._publishOrDiscardRequest.always(function () {
                delete instance._publishOrDiscardRequest;
            });

            return deferred.promise();
        },

        /**
         * Save a {@link JIRA.WorkflowDesigner.WorkflowModel} to the server.
         *
         * @param {JIRA.WorkflowDesigner.WorkflowModel} workflowModel The workflow to save.
         * @return {jQuery.Deferred} A promise that is resolved on success.
         */
        save: function (workflowModel) {
            var deferred = jQuery.Deferred(),
                saveRequest,
                workflowData = WorkflowDataWriter.write(workflowModel);

            if (!this._publishOrDiscardRequest) {
                saveRequest = this._makeRequest({
                    contentType: "application/json",
                    data: JSON.stringify(workflowData),
                    success: deferred.resolve,
                    type: "POST",
                    url: contextPath() + "/rest/workflowDesigner/latest/workflows"
                });

                saveRequest.fail(function (response, status) {
                    deferred.reject(ErrorHandler.getErrorMessage(response, status));
                });
            }

            return deferred.promise();
        },

        /**
         * Trigger a MAU event for the current project that this workflow belongs to.
         * This doesn't return a deferred because we don't care if this fails or not.
         *
         * @param {string} projectKey The project key to use.
         */
        triggerMauEventForProject: function (projectKey) {
            this._makeRequest({
                data: {
                    projectKey: projectKey
                },
                dataType: "json",
                type: "POST",
                url: AJS.contextPath() + "/rest/workflowDesigner/latest/workflowMau"
            });
        }
    });
});

AJS.namespace("JIRA.WorkflowDesigner.IO.AJAX.WorkflowAJAXManager", null, require("workflow-designer/io/ajax/workflow-ajax-manager"));