AJS.test.require([
    "com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer",
    "com.atlassian.jira.plugins.jira-workflow-designer:test-resources"
], function () {

    var LayoutAutoSaver = require("workflow-designer/io/layout-auto-saver");
    var WorkflowModel = require("workflow-designer/workflow-model");
    var Messages = require("workflow-designer/messages");
    var WorkflowAJAXManager = require("workflow-designer/io/ajax/workflow-ajax-manager");
    var jQuery = require("jquery");
    var TestUtilities = require("workflow-designer/test-utilities");

    module("LayoutAutoSaver", {
        createLayoutAutoSaver: function () {
            return new LayoutAutoSaver({
                window: this.window,
                workflowModel: this.workflowModel
            });
        },

        setup: function () {
            this.sandbox = sinon.sandbox.create();

            this.showErrorMessageStub = this.sandbox.stub(Messages, "showErrorMessage");
            this.saveStub = this.sandbox.stub(WorkflowAJAXManager, "save");
            this.saveStub.returns(jQuery.Deferred().resolve());
            this.window = {onbeforeunload: jQuery.noop};
            this.workflowModel = new WorkflowModel();
        },

        teardown: function () {
            this.sandbox.restore();
        },

        triggerLayoutChangedEvent: function () {
            this.workflowModel.trigger("layoutChanged");
        }
    });

    test("Calls the error handler on failure", function () {
        TestUtilities.fakeTimer(function (clock) {
            this.createLayoutAutoSaver();
            this.saveStub.returns(jQuery.Deferred().reject());
            this.triggerLayoutChangedEvent();

            clock.tick(1100);
            equal(this.showErrorMessageStub.callCount, 1, "The error handler was called");
            equal(this.saveStub.callCount, 1, "_save() was called");
        }, this);
    });

    test("Saves when WorkflowModel triggers a layoutChanged event", function () {
        TestUtilities.fakeTimer(function (clock) {
            this.createLayoutAutoSaver();
            this.triggerLayoutChangedEvent();

            clock.tick(1100);
            equal(this.showErrorMessageStub.callCount, 0, "The error handler wasn't called");
            equal(this.saveStub.callCount, 1, "_save() was called");
        }, this);
    });

    test("Shows an 'unsaved changes' message", function () {
        TestUtilities.fakeTimer(function (clock) {
            var onBeforeUnload;

            this.createLayoutAutoSaver();
            this.triggerLayoutChangedEvent();

            onBeforeUnload = this.window.onbeforeunload;
            ok(jQuery.noop !== onBeforeUnload && _.isFunction(onBeforeUnload), "window.onbeforeunload was set");

            clock.tick(1100);
            equal(jQuery.noop, this.window.onbeforeunload, "The original window.onbeforeunload was restored");
        }, this);
    });

    test("The 'unsaved changes' message is maintained through concurrent saves", function () {
        TestUtilities.fakeTimer(function (clock) {
            var deferred = jQuery.Deferred();

            this.createLayoutAutoSaver();
            this.saveStub.returns(deferred);
            this.triggerLayoutChangedEvent();

            clock.tick(1100);
            equal(this.saveStub.callCount, 1, "_save() was called");

            // Trigger another layout changed event before the first save completes.
            this.saveStub.returns(jQuery.Deferred().resolve());
            this.triggerLayoutChangedEvent();
            deferred.resolve();

            // There are unsaved changes (the second event), so the onbeforeunload shouldn't be restored.
            ok(jQuery.noop !== this.window.onbeforeunload, "The original window.onbeforeunload was not restored");

            clock.tick(1100);
            equal(this.saveStub.callCount, 2, "_save() was called");
            equal(jQuery.noop, this.window.onbeforeunload, "The original window.onbeforeunload was restored");
        }, this);
    });

    test("The 'unsaved changes' message is not cleared if saving failed", function () {
        TestUtilities.fakeTimer(function (clock) {
            var deferred = jQuery.Deferred();
            this.saveStub.returns(deferred);

            this.createLayoutAutoSaver();
            this.triggerLayoutChangedEvent();

            clock.tick(1100);
            deferred.reject();

            ok(jQuery.noop !== this.window.onbeforeunload, "The original window.onbeforeunload was not restored");
        }, this);
    });

    test("Doesn't show an 'unsaved changes' message if it is disabled", function () {
        TestUtilities.fakeTimer(function () {
            LayoutAutoSaver.disableMessage();

            this.createLayoutAutoSaver();
            this.triggerLayoutChangedEvent();

            equal(jQuery.noop, this.window.onbeforeunload, "window.onbeforeunload was not set");
        }, this);
    });

    test("destroy() removes the onbeforeunload handler", function () {
        var layoutAutoSaver;

        TestUtilities.fakeTimer(function () {
            layoutAutoSaver = this.createLayoutAutoSaver();
            this.saveStub.returns(jQuery.Deferred());
            this.triggerLayoutChangedEvent();

            layoutAutoSaver.destroy();
            equal(jQuery.noop, this.window.onbeforeunload, "The original window.onbeforeunload was restored");

            // No original onbeforeunload handler.
            layoutAutoSaver = this.createLayoutAutoSaver();
            this.window.onbeforeunload = undefined;
            this.triggerLayoutChangedEvent();

            layoutAutoSaver.destroy();
            equal(undefined, this.window.onbeforeunload, "The original window.onbeforeunload was restored");
        }, this);
    });

});
