AJS.test.require([
    "com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer",
    "com.atlassian.jira.plugins.jira-workflow-designer:test-resources"
], function () {

    var Application;
    var Canvas;
    var WorkflowModel;
    var WorkflowAJAXManager;
    var Messages;
    var LayoutAutoSaver;
    var BrowserSupport;
    var Templates;
    var TestUtilities = require("workflow-designer/test-utilities");
    var jQuery = require("jquery");
    
    module("Application", {
        setup: function () {
            this.sandbox = sinon.sandbox.create();
            this.context = AJS.test.mockableModuleContext();

            this.canvas = TestUtilities.testCanvas();
            var workflowAJAXManager = require("workflow-designer/io/ajax/workflow-ajax-manager");
            this.loadWorkflowStub = this.sandbox.stub(workflowAJAXManager, "load");
            this.publishStub = this.sandbox.stub(workflowAJAXManager, "publish");
            this.discardStub = this.sandbox.stub(workflowAJAXManager, "discard");
            this.mauStub = this.sandbox.stub(workflowAJAXManager, "triggerMauEventForProject");
            this.context.mock("workflow-designer/io/ajax/workflow-ajax-manager", workflowAJAXManager);
            WorkflowAJAXManager = this.context.require("workflow-designer/io/ajax/workflow-ajax-manager");

            this.context.mock("workflow-designer/browser-support", require("workflow-designer/browser-support"));
            BrowserSupport = this.context.require("workflow-designer/browser-support");

            this.context.mock("workflow-designer/io/layout-auto-saver", require("workflow-designer/io/layout-auto-saver"));
            LayoutAutoSaver = this.context.require("workflow-designer/io/layout-auto-saver");

            var WorkflowModelClass = require("workflow-designer/workflow-model");
            this.workflowModelResetSpy = this.sandbox.spy(WorkflowModelClass.prototype, "reset");
            var workflowModel = new WorkflowModelClass();
            this.context.mock("workflow-designer/workflow-model", sinon.stub().returns(workflowModel));
            this.context.mock("workflow-designer/canvas", sinon.stub().returns(this.canvas));

            this.context.mock("workflow-designer/templates", require("workflow-designer/templates"));
            Templates = this.context.require("workflow-designer/templates");

            this.context.mock("workflow-designer/messages", require("workflow-designer/messages"));
            Messages = this.context.require("workflow-designer/messages");

            WorkflowModel = this.context.require("workflow-designer/workflow-model");
            Canvas = this.context.require("workflow-designer/canvas");
        },
        prepareApplication: function() {
            Application = this.context.require("workflow-designer/application");
        },
        teardown: function () {
            this.sandbox.restore();
        }
    });

    test("A global message is shown on error", function () {
        var errorMessage = "No!",
            showErrorMessageSpy = this.sandbox.stub(Messages, "showErrorMessage");
        this.prepareApplication();

        this.loadWorkflowStub.returns(jQuery.Deferred().reject(errorMessage));
        new Application({workflowId: "Workflow"});

        equal(showErrorMessageSpy.callCount, 1, "Messages.showErrorMessage was called");
        deepEqual(showErrorMessageSpy.args[0], [errorMessage], "It was passed the correct arguments");
    });

    test("A warning is shown in unsupported browsers", function () {
        var element = jQuery("#qunit-fixture"),
            supportedStub = this.sandbox.stub(BrowserSupport, "browserIsSupported").returns(false),
            templateSpy = this.sandbox.spy(Templates, "browserNotSupportedWarning");

        this.prepareApplication();
        new Application({element: element});
        equal(Canvas.callCount, 0, "No Canvas was created");
        equal(supportedStub.callCount, 1, "BrowserSupport.browserIsSupported was called");
        equal(templateSpy.callCount, 1, "JIRA.WorkflowDesigner.Templates.browserNotSupportWarning was called");
        equal(element.find(".aui-message.warning").length, 1, "The warning was appended to the application's element");
    });

    test("destroy() calls LayoutAutoSaver#destroy", function () {
        this.prepareApplication();
        var destroySpy = this.sandbox.spy(LayoutAutoSaver.prototype, "destroy");

        new Application().destroy();
        equal(destroySpy.callCount, 1, "LayoutAutoSaver#destroy was called");
    });

    test("destroy() doesn't throw an exception in unsupported browsers", function () {
        this.prepareApplication();
        var application;

        expect(0);
        this.sandbox.stub(BrowserSupport, "browserIsSupported").returns(false);

        application = new Application({element: jQuery("#qunit-fixture")});
        application.destroy();
    });

    test("Passing layout data to the constructor", function () {
        this.prepareApplication();
        new Application({layoutData: {}});
        equal(this.workflowModelResetSpy.callCount, 1, "The Application's WorkflowModel was reset");
    });

    test("Supports loading a workflow by ID", function () {
        var deferred = jQuery.Deferred();
        var workflowModel = WorkflowModel();

        sinon.spy(this.canvas, "autoFit");
        sinon.spy(this.canvas, "hideProgressIndicator");
        sinon.spy(this.canvas, "showProgressIndicator");
        this.loadWorkflowStub.returns(deferred.promise());

        this.prepareApplication();
        new Application({workflowId: "Workflow"});

        equal(this.loadWorkflowStub.callCount, 1, "WorkflowAJAXManager#load() was called");
        deepEqual(this.loadWorkflowStub.args[0], ["Workflow", false], "It was passed the correct arguments");
        equal(this.canvas.showProgressIndicator.callCount, 1, "Canvas#showProgressIndicator() was called");

        deferred.resolve({});
        equal(this.canvas.autoFit.callCount, 1, "Canvas#autoFit() was called");
        equal(this.canvas.hideProgressIndicator.callCount, 1, "Canvas#hideProgressIndicator() was called");
        equal(this.mauStub.callCount, 1, "WorkflowAJAXManager#triggerMauEventForProject() was called");
        equal(this.workflowModelResetSpy.callCount, 1, "WorkflowModel#reset() was called");
        deepEqual(this.workflowModelResetSpy.args[0], [{}], "It was passed the correct arguments");
        ok(workflowModel.has("loadedAt"), "WorkflowModel's loadedAt property was set");
        ok(workflowModel.get("loadedAt") instanceof Date, "It is a Date object");
        ok(new Date() - workflowModel.get("loadedAt") < 500, "It is the current time");
    });

    test("The current step ID, if given, is passed to WorkflowModel", function () {
        this.prepareApplication();
        this.workflowModelSpy = WorkflowModel;
        new Application({currentStepId: 42});

        equal(this.workflowModelSpy.args[0][0].currentStepId, 42, "The current step ID was passed to the WorkflowModel constructor");
    });

    test("Triggers a \"loaded\" event after successfully loading a workflow via AJAX", function () {
        this.prepareApplication();
        var application,
            deferred = jQuery.Deferred(),
            loadedSpy = sinon.spy();

        this.loadWorkflowStub.returns(deferred.promise());
        application = new Application({workflowId: "Workflow"});
        application.on("loaded", loadedSpy);

        deferred.resolve({});
        equal(loadedSpy.callCount, 1, "A \"loaded\" event was triggered");
    });

    test("Triggers a \"loaded\" event after successfully loading the given workflow data", function () {
        this.prepareApplication();
        var triggerSpy = this.sandbox.spy(Application.prototype, "trigger");

        new Application({layoutData: {}});
        ok(triggerSpy.calledWithExactly("loaded"), "A \"loaded\" event was triggered");
    });

    test("publishDraft() successfully publishes draft", function () {
        this.prepareApplication();
        var application = new Application({workflowId: "Workflow"}),
            resultDeferred,
            deferred = jQuery.Deferred(),
            loadingDeferred = jQuery.Deferred().resolve(),
            showErrorMessageSpy = this.sandbox.stub(Messages, "showErrorMessage"),
            loadWorkflowStub = this.sandbox.stub(Application.prototype, "_loadWorkflow");

        sinon.spy(this.canvas, "hideProgressIndicator");
        sinon.spy(this.canvas, "showProgressIndicator");

        this.publishStub.returns(deferred.promise());
        loadWorkflowStub.returns(loadingDeferred);

        resultDeferred = application.publishDraft();
        equal(this.canvas.showProgressIndicator.callCount, 1, "Canvas#showProgressIndicator() was called");
        equal(resultDeferred.state(), "pending", "The returned deferred is pending");
        equal(this.publishStub.callCount, 1, "WorkflowAJAXManager#publish() was called");

        deferred.resolve();
        equal(resultDeferred.state(), "resolved", "The returned deferred is resolved on success");
        equal(this.canvas.hideProgressIndicator.callCount, 0, "Canvas#hideProgressIndicator() was not called");
        equal(showErrorMessageSpy.callCount, 0, "Messages.showErrorMessage was not called");
        equal(loadWorkflowStub.callCount, 1, "Workflow designer was reloaded");
    });

    test("publishDraft() without reloadDesigner flag successfully publishes draft", function () {
        this.prepareApplication();
        var application = new Application({workflowId: "Workflow"}),
            resultDeferred,
            deferred = jQuery.Deferred(),
            showErrorMessageSpy = this.sandbox.stub(Messages, "showErrorMessage"),
            loadWorkflowStub = this.sandbox.stub(Application.prototype, "_loadWorkflow");

        sinon.spy(this.canvas, "hideProgressIndicator");
        sinon.spy(this.canvas, "showProgressIndicator");

        this.publishStub.returns(deferred.promise());

        resultDeferred = application.publishDraft({reloadDesigner: false});
        equal(this.canvas.showProgressIndicator.callCount, 1, "Canvas#showProgressIndicator() was called");
        equal(resultDeferred.state(), "pending", "The returned deferred is pending");
        equal(this.publishStub.callCount, 1, "WorkflowAJAXManager#publish() was called");

        deferred.resolve();
        equal(resultDeferred.state(), "resolved", "The returned deferred is resolved on success");
        equal(this.canvas.hideProgressIndicator.callCount, 0, "Canvas#hideProgressIndicator() was not called");
        equal(showErrorMessageSpy.callCount, 0, "Messages.showErrorMessage was not called");
        equal(loadWorkflowStub.callCount, 0, "Workflow designer was not reloaded");
    });

    test("publishDraft() fails publishing draft", function () {
        this.prepareApplication();
        var application = new Application({workflowId: "Workflow"}),
            resultDeferred,
            deferred = jQuery.Deferred(),
            showErrorMessageSpy = this.sandbox.stub(Messages, "showErrorMessage");

        sinon.spy(this.canvas, "hideProgressIndicator");
        sinon.spy(this.canvas, "showProgressIndicator");

        this.publishStub.returns(deferred.promise());

        resultDeferred = application.publishDraft();
        equal(resultDeferred.state(), "pending", "The returned deferred is pending");
        equal(this.publishStub.callCount, 1, "WorkflowAJAXManager#publish() was called");
        equal(this.canvas.showProgressIndicator.callCount, 1, "Canvas#showProgressIndicator() was called");

        deferred.reject("error message");
        equal(resultDeferred.state(), "rejected", "The returned deferred is rejected on failure");
        equal(this.canvas.hideProgressIndicator.callCount, 1, "Canvas#hideProgressIndicator() was called");
        equal(showErrorMessageSpy.callCount, 1, "Messages.showErrorMessage was called");
    });

    test("discardDraft() successfully discards draft", function () {
        this.prepareApplication();
        var application = new Application({workflowId: "Workflow"}),
            resultDeferred,
            deferred = jQuery.Deferred(),
            showErrorMessageSpy = this.sandbox.stub(Messages, "showErrorMessage"),
            loadWorkflowStub = this.sandbox.stub(Application.prototype, "_loadWorkflow");

        sinon.spy(this.canvas, "hideProgressIndicator");
        sinon.spy(this.canvas, "showProgressIndicator");

        this.discardStub.returns(deferred.promise());
        loadWorkflowStub.returns(jQuery.Deferred().resolve());

        resultDeferred = application.discardDraft();
        equal(resultDeferred.state(), "pending", "The returned deferred is pending");
        equal(this.canvas.showProgressIndicator.callCount, 1, "Canvas#showProgressIndicator() was called");
        equal(this.discardStub.callCount, 1, "WorkflowAJAXManager#discard() was called");

        deferred.resolve();
        equal(resultDeferred.state(), "resolved", "The returned deferred is resolved on success");
        equal(this.canvas.hideProgressIndicator.callCount, 0, "Canvas#hideProgressIndicator() was not called");
        equal(showErrorMessageSpy.callCount, 0, "Messages.showErrorMessage was not called");
        equal(this.discardStub.callCount, 1, "WorkflowAJAXManager#discard() was called");
        equal(loadWorkflowStub.callCount, 1, "Workflow designer was reloaded");
    });

    test("discardDraft() without reloadDesigner flag successfully discards draft", function () {
        this.prepareApplication();
        var application = new Application({workflowId: "Workflow"}),
            resultDeferred,
            deferred = jQuery.Deferred(),
            showErrorMessageSpy = this.sandbox.stub(Messages, "showErrorMessage"),
            loadWorkflowStub = this.sandbox.stub(Application.prototype, "_loadWorkflow");

        sinon.spy(this.canvas, "hideProgressIndicator");
        sinon.spy(this.canvas, "showProgressIndicator");

        this.discardStub.returns(deferred.promise());
        loadWorkflowStub.returns(jQuery.Deferred().resolve());

        resultDeferred = application.discardDraft({reloadDesigner: false});
        equal(resultDeferred.state(), "pending", "The returned deferred is pending");
        equal(this.canvas.showProgressIndicator.callCount, 1, "Canvas#showProgressIndicator() was called");
        equal(this.discardStub.callCount, 1, "WorkflowAJAXManager#discard() was called");

        deferred.resolve();
        equal(resultDeferred.state(), "resolved", "The returned deferred is resolved on success");
        equal(this.canvas.hideProgressIndicator.callCount, 0, "Canvas#hideProgressIndicator() was not called");
        equal(showErrorMessageSpy.callCount, 0, "Messages.showErrorMessage was not called");
        equal(loadWorkflowStub.callCount, 0, "Workflow designer was not reloaded");
    });

    test("discardDraft() fails discarding draft", function () {
        this.prepareApplication();
        var application = new Application({workflowId: "Workflow"}),
            resultDeferred,
            deferred = jQuery.Deferred(),
        showErrorMessageSpy = this.sandbox.stub(Messages, "showErrorMessage");

        sinon.spy(this.canvas, "hideProgressIndicator");
        sinon.spy(this.canvas, "showProgressIndicator");

        this.discardStub.returns(deferred.promise());

        resultDeferred = application.discardDraft();
        equal(resultDeferred.state(), "pending", "The returned deferred is pending");
        equal(this.canvas.showProgressIndicator.callCount, 1, "Canvas#showProgressIndicator() was called");

        deferred.reject("error message");
        equal(resultDeferred.state(), "rejected", "The returned deferred is rejected on failure");
        equal(this.canvas.hideProgressIndicator.callCount, 1, "Canvas#hideProgressIndicator() was called");
        equal(showErrorMessageSpy.callCount, 1, "Messages.showErrorMessage was called");
    });

});
