require([
    "jquery",
    "aui/flag",
    "shc/ajs",
    "shc/location"
], function($, flag, AJS, location) {

    if (location.isSupportTools()) {
        // Exit if we are on the STP screen, we don't want to display healthcheck notification if user browsing on STP
        return;
    }

    AJS.toInit(function() {

        var restBase = AJS.contextPath() + '/rest/supportHealthCheck/1.0';
        var user = $('meta[name=ajs-remote-user]').attr('content');

        // Initialise the healthcheck notification. Make a GET call to /check/{username} endpoint to get the list of failed check by user
        var initHealthcheckNotification = function() {
            $.getJSON(restBase + '/check/' + user).done(function(statusReport) {
                statusReport.statuses.forEach(function(status) {
                    showNotification(status);
                });

                // Listen to click event to dismiss or snooze notification
                $(document).find('.dismiss-notification').on('click', function() {
                    var $notification = $(this).parents().eq(3);
                    dismissNotification($notification);
                });

                $(document).find('.healthcheck-detail').on('click', function() {
                    var $notification = $(this).parents().eq(3);
                    var application = $notification.attr('application');
                    var severity = $notification.attr('severity');
                    var completeKey = $notification.attr('completeKey');

                    analyticsModule.run("Viewed", completeKey ,application,severity);
                    snoozeNotification($notification);
                });
            });
        };

        // Renders the flat notification here, and set the necessary attribute to the flag
        var showNotification = function(status) {

            var healthCheckName = status.name;
            var severity = status.severity;
            var notificationId = status.id;
            var application = status.application;
            var completeKey = status.completeKey;

            var flagType = "error";
            if (severity == "warning") {
                flagType = "warning";
            }

            var healthcheckNotification = flag ({
                type : flagType,
                body : "<p>The <strong>" + healthCheckName + "</strong> health check has failed in your system. </p>" +
                       "<ul class='aui-nav-actions-list'>" +
                       "    <li><a href='" + AJS.contextPath() + "/plugins/servlet/stp/view/?source=notification' class='healthcheck-detail' target='_self'>Find out more</a></li>" +
                       "    <li><a href='#' class='dismiss-notification'>Don’t remind me again</a></li>" +
                       "</ul>",
                close: "manual"
            });

            healthcheckNotification.setAttribute("id", "healthcheck-notification");
            healthcheckNotification.setAttribute("healthcheck-name", healthCheckName);
            healthcheckNotification.setAttribute("notification-id", notificationId);
            healthcheckNotification.setAttribute("application", application);
            healthcheckNotification.setAttribute("severity", severity);
            healthcheckNotification.setAttribute("completeKey", completeKey);

            healthcheckNotification.addEventListener("aui-flag-close", function() {
                var $notification = $(this);
                snoozeNotification($notification);
            });


            analyticsModule.run("Display", completeKey, application, severity);
        };

        // Called when the "Don’t remind me again" link is clicked. Notifications that are dismissed / marked as read won't reappear to the user anymore
        var dismissNotification = function($notification) {
            $notification.attr("aria-hidden", "true");

            var id = $notification.attr('notification-id');
            var name = $notification.attr('healthcheck-name');
            var application = $notification.attr('application');
            var severity = $notification.attr('severity');
            var completeKey = $notification.attr('completeKey');

            var data = {};
            data["username"] = user;
            data["notificationId"] = id;
            data["snooze"] = false;

            var jsonData = JSON.stringify(data);

            AJS.$.ajax({
                async      : false,
                type       : 'POST',
                url        : restBase + '/dismissNotification',
                data       : jsonData,
                contentType: "application/json",
                dataType   : "json",
                success    : function() {

                    analyticsModule.run("Dismissed", completeKey, application, severity);
                }
            })
        };

        // Called when the notification is closed via the cross button or if user open the link to the healthcheck
        // A notification is dismissed / marked as read automatically if it snoozes 3 times
        var snoozeNotification = function($notification) {
            var id = $notification.attr('notification-id');
            var name = $notification.attr('healthcheck-name');
            var application = $notification.attr('application');
            var severity = $notification.attr('severity');
            var completeKey = $notification.attr('completeKey');

            var data = {};
            data["username"] = user;
            data["notificationId"] = id;
            data["snooze"] = true;

            var jsonData = JSON.stringify(data);

            AJS.$.ajax({
                async      : false,
                type       : 'POST',
                url        : restBase + '/dismissNotification',
                data       : jsonData,
                contentType: "application/json",
                dataType   : "json",
                success    : function(data, textStatus, jqXHR) {

                    isAutoDismissed(id, completeKey, name, application, severity);
                    analyticsModule.run("Snoozed", completeKey, application, severity);

                }
            })
        };
        
        var isAutoDismissed = function(notificationId, completeKey, name, application, severity) {
            var url = restBase + '/dismissNotification/' + user + '/' + notificationId;

            AJS.$.ajax({
                async      : false,
                type       : 'GET',
                url        : url,
                contentType: "text/xml",
                success    : function(data, textStatus, jqXHR) {
                    if (jqXHR.status == 200) {
                        notificationAutoDismissed(name);
                        analyticsModule.run("Autodismissed", completeKey, application, severity);
                    }


                }
            });
        };

        // This module allows for us to capture the correct Analytics events per display as per SHC-344.
        var analyticsModule = (function () {
            var lastEvent;
            var event = {};
            
            event.run = function (action, completeKey, application, severity) {
                if (lastEvent == "Viewed") {
                    // when user views info, we snooze the notification. We only want to record the Viewed AA event.
                    lastEvent = action;
                } else {
                    lastEvent = action;
                    if (action == "Display") {
                        AJS.trigger('analyticsEvent', {
                            name: "healthcheck.notification.display",
                            data: {action: action, name: completeKey, application: application, severity: severity}
                        });
                    } else {
                        AJS.trigger('analyticsEvent', {
                            name: "healthcheck.notification.action",
                            data: {action: action, name: completeKey, application: application, severity: severity}
                        });
                    }
                }
            };
            return event;
        }());

        // This function is called when a notification is automatically marked as read (normally is called when a notification is snoozed 3 times)
        var notificationAutoDismissed = function(healthCheckName) {
            // We set a quick timeout here so the animation will be displayed properly
            setTimeout(function() {
                flag({
                    type : "info",
                    body : "<p>The <strong>" + healthCheckName + "</strong> notification is now dismissed. </p>" +
                    "<p>Please visit <a href='" + AJS.contextPath() + "/plugins/servlet/stp/view/?source=notification' class='healthcheck-detail' target='_self'>this link</a> for more on the check.</p>",
                    close: "auto"
                });
            }, 1500);
        };

        initHealthcheckNotification();
    });
});