define("stp/detect-issues/main", [
    "stp/detect-issues/config",
    "stp/detect-issues/helper"
], function (Config, Helper) {
    // Responsible for:
    // - displaying the spinner while the check is still running on the background
    // - get the current completed health check run from /check/process/{processId}/results endpoint
    // - call the displayCheckResults function to dynamically render the checks that has been completed
    var loadCheckResults = function (processReport, $container) {
        "use strict";

        // Display the health check status container using jQuery's fadeIn() function
        $container.fadeIn("slow");

        Helper.addSpinner($container);

        // Get the process ID from response process report
        var processId = processReport.processId;
        // URL for retrieving the healthcheck results from background process
        var url = Config.instanceHealthProcessUrl + processId + "/results";

        var getResults = AJS.$.ajax({
            url: url,
            dataType: "json"
        });

        // Woot! Handle the request response here
        getResults.done(function (healthCheckResults) {
            displayCheckResults(processReport, healthCheckResults, $container);
        })
    };

    // What this function does:
    // - dynamically render the checks that has been completed
    // - validate if all check has been performed
    // -- If all checks has been performed, remove the spinner from page and trigger analytics
    // -- If there are checks that still runs on background, run loadCheckResults function to get the latest results every 500 miliseconds
    var displayCheckResults = function (processReport, healthCheckResults, $container) {
        "use strict";

        // Get the health check representations from process report
        var healthChecks = processReport.healthCheckRepresentations;
        // Get the health check statuses from check results
        var statuses = healthCheckResults.statuses;

        // Group the status by isHealthy attribute
        var groups = _.groupBy(statuses, function (item) {
            return item.isHealthy ? 'pass' : 'fail';
        });

        // If check results status length == health check representations length, it means that all checks run has been completed
        if (statuses.length == healthChecks.length) {
            // Remove the spinner container, we no longer need it on the page
            Helper.removeSpinner();

            // Trigger analytics event for each of healthcheck
            statuses.forEach(function (check) {
                Helper.sendAnalytics(check, '.' + (check.isHealthy ? 'pass' : 'fail') + '.done');
            });

            if (groups.fail) {
                Helper.displayBadge(groups.fail.length);
            }
        } else {
            // Else, we check the healthcheck background process again in 500 miliseconds
            setTimeout(function () {
                loadCheckResults(processReport, $container);
            }, 500);
        }

        if (groups.fail) {
            // We sort the failed results by priority, display the major error first before warn
            var failSorted = _.sortBy(groups.fail, Helper.getSeverity).reverse();
            // Then we group the failed checks by its tags
            var failGroups = Helper.groupByTags(failSorted);
            addResultsToTemplate("failed", failGroups);
        }

        if (groups.pass) {
            // We group the passed checks by its tags
            var healthyGroups = Helper.groupByTags(groups.pass);
            addResultsToTemplate("passed", healthyGroups);
        }
    };

    // Validates if a given check result in each groups has been displayed on the results container, else add the check results content to the container
    var addResultsToTemplate = function (passFail, checkGroups) {
        // Iterate over each of the health check groups
        checkGroups.forEach(function (group) {
            // Get the tag name, change the character casing to lower case, and replace space character with a "-"
            // The tag name will be used as the ID of certain elements inside the health check results container
            var tagName = group[0].tag.toLowerCase();
            tagName = tagName.split(" ").join("-");

            // Get either the passed or failed check container, depending on where this function was called
            var $checkContainer = AJS.$("#" + passFail + "-check-container");
            var $statusGroupList = $checkContainer.find(".health-status-group-list");

            // If the status group list doesn't exists, add it to the template
            if (!$statusGroupList.find("#" + tagName).length) {
                $statusGroupList.append(Stp.HealthCheck.Dynamic.Templates.healthStatusGroup({
                    tagName: tagName,
                    status: group[0]
                }));
            }

            // Iterate over each of the health check status in the group
            group.forEach(function (status) {
                var $statusList = $statusGroupList.find("#" + tagName);

                // Get the status container, if it doesn't exists we add it to the template
                if (!$statusList.find("#" + tagName).length) {
                    $statusList.append(Stp.HealthCheck.Dynamic.Templates.healthStatusList({
                        tagName: tagName
                    }));
                }

                // Get the status name, then we lower case it and replace space character with "-"
                // The status name will be used as ID for the health-status-item div
                var statusName = status.name.toLowerCase();
                statusName = statusName.split(" ").join("-");

                var $list = $statusList.find("#" + tagName);
                // If the check is not displayed on the page, we add it to the template
                if (!$list.find("#" + statusName).length) {
                    var status = AJS.$(Stp.HealthCheck.Dynamic.Templates.healthStatus({
                        statusName: statusName,
                        status: status
                    }));
                    AJS.$(status).find('.health-status-documentation-link').click(function (evt) {
                        Helper.sendAnalytics(this.dataset, '.clicked');
                    });
                    $list.append(status);
                }
            });

            // The passed/failed check container is hidden by default, hence if it's hidden here, we display it using the fadeIn() function
            if (!$checkContainer.is(":visible")) {
                $checkContainer.fadeIn("slow");
            }
        });
    };

    // This function is called for rendering the healthchecks when retrieving results using the older "/checkDetails" endpoint
    var buildResults = function (healthCheckResults, $container) {
        "use strict";

        // healthCheckResults.status is legacy and can be removed once the SHC plugin is updated.
        var statuses = healthCheckResults.statuses || healthCheckResults.status;

        statuses.forEach(function (status) {
            // The severity string value may in upper case for old versions of the SHC plugin.
            status.severity = status.severity.toLowerCase();
        });

        var groups = _.groupBy(statuses, function (item) {
            var passFail = item.isHealthy ? 'pass' : 'fail';
            Helper.sendAnalytics(item, '.' + passFail + '.done');
            return passFail;
        });

        var tagExists = false;
        statuses.forEach(function (item) {
            if (item.tag && item.tag != 'stp') {
                tagExists = true;
            }
        });

        if (groups.fail) {
            var failSorted = _.sortBy(groups.fail, Helper.getSeverity).reverse();
            var failGroups;

            if (tagExists) {
                failGroups = Helper.groupByTags(failSorted);
                $container.append(Stp.HealthCheck.Templates.unhealthy({
                    groupsOfStatuses: failGroups
                }));
            } else {
                failGroups = _.chain(failSorted).toArray().value();
                $container.append(Stp.HealthCheck.Templates.unhealthyLegacy({
                    groupsOfStatuses: failGroups
                }));
            }

            Helper.displayBadge(groups.fail.length);
        }

        if (groups.pass) {
            var healthyGroups;

            if (tagExists) {
                healthyGroups = Helper.groupByTags(groups.pass);
                $container.append(Stp.HealthCheck.Templates.healthy({
                    groupsOfStatuses: healthyGroups
                }));
            } else {
                healthyGroups = _.chain(groups.pass).toArray().value();
                $container.append(Stp.HealthCheck.Templates.healthyLegacy({
                    groupsOfStatuses: healthyGroups
                }));
            }
        }

        // Finally, display the health check status container using jQuery's fadeIn() function
        $container.fadeIn("slow");

        // Trigger the following event, which is listened by a script in Support Healthcheck Plugin to render the "Last Run" div
        AJS.trigger("healthContainerLoadedEvent");
    };

    return {
        initInstanceHealth: function () {
            // Get the health status container, and set the onclick event on the expandable child elements first
            var $container = AJS.$(".health-status-container");
            Helper.attachHealthcheckItemClickEvent($container);

            // populate the Instance health description events
            AJS.$('#instance-health-check-link').click(function (e) {
                AJS.trigger('analyticsEvent', {name: 'stp.instance.health.description'});
            });

            AJS.$('#instance-health-check-link').attr('href', 'https://confluence.atlassian.com/support/instance-health-790796828.html#InstanceHealth-WhatHealthChecksareavailable?');

            var checkProcess = AJS.$.ajax({
                type: "POST",
                url: Config.instanceHealthProcessUrl,
                contentType: "application/json",
                dataType: "json"
            });

            // If the request was successful, get the process Id and get the check results here
            checkProcess.done(function (processReport) {
                // Trigger the following event, which is listened by a script in Support Healthcheck Plugin to render the "Last Run" div
                AJS.trigger("healthContainerLoadedEvent");
                loadCheckResults(processReport, $container);
            });

            // If the call to trigger the check has failed, most likely it means that the endpoint /check/process doesn't exists
            // In this case, we fallback to the /checkDetails endpoint to retrieve the healthcheck results
            // In general, this is a fallback approach for backward compatibility
            checkProcess.fail(function () {
                var checkDetails = AJS.$.ajax({
                    url: Config.legacyCheckDetailsUrl,
                    dataType: "json"
                });

                // Display results if request was successful
                checkDetails.done(function (data) {
                    buildResults(data, $container);
                });

                // If an error occurred when making call to /checkDetails endpoint, we handle the error here
                checkDetails.fail(function (jqXHR) {
                    if (jqXHR.status === 503) {
                        try {
                            buildResults(AJS.$.parseJSON(jqXHR.responseText), $container);
                        } catch (e) {
                            Helper.handleError();
                        }
                    } else {
                        Helper.handleError();
                    }
                });
            });
        }
    };
});
