define('util/auth',
    ['backbone', 'util/ajax', 'util/app-data', 'util/events', 'util/router', 'util/scroll-tops'], // 'layout/login/login-view'
    function (Backbone, ajax, appData, events, router, Scrolling) {
        var contextPath = appData.get('context-path');

        var auth = {
            hasSSO: function () {
                return appData.getBoolean('has-sso');
            },

            getLoginDestination: function () {
                var location = window.location;
                var currentPath = location.href.substr((location.origin + contextPath).length).replace('#login/', '#');
                return contextPath + '/login.jsp?os_destination=' + encodeURIComponent(currentPath);
            },

            getLogoutDestination: function () {
                return contextPath + '/logout';
            },

            showLoginPage: function (replace) {
                if (this.hasSSO()) {
                    return window.location = this.getLoginDestination();
                } else {
                    var fragment = Backbone.history.fragment;
                    if (fragment.substr(0, 6) != 'login/') {
                        // replace is true by default unless otherwise specified
                        var replaceVal = typeof replace === "undefined" ? true : !!replace;
                        router.navigate('login/' + fragment, {
                            trigger: true,
                            replace: replaceVal
                        });
                    }
                }
            },

            showCleanLoginPage: function () {
                if (this.hasSSO()) {
                    return window.location = contextPath + '/login.jsp';
                } else {
                    router.navigate('login', {
                        trigger: true,
                        replace: false
                    });
                }
            },

            login: function (user) {
                this._updateUser(user);
                events.user.trigger('auth.login');
            },

            logout: function (options) {
                if (this.hasSSO()) {
                    return window.location = this.getLogoutDestination();
                } else {
                    this.triggerLogout();
                    options = options || {};
                    var willRedirect = options.showLoginPage !== false;
                    // We don't use a callback here because if the call fails with 401,
                    // the auth handler in util/ajax will show the login page anyway.
                    ajax.rest({
                        url: '/rest/auth/latest/session',
                        type: 'DELETE',
                        async: !willRedirect
                    });
                    if (options.showLoginPage !== false) {
                        this.showCleanLoginPage();
                    }
                }
            },

            /**
             * Clear the stored username and trigger the logout event.
             * Used for updating the UI when detecting a session timeout, but without showing the login page.
             */
            triggerLogout: function () {
                this._clearUser();
                Scrolling.resetAll();
                events.user.trigger('auth.logout');
            },

            _updateUser: function (user) {
                appData.put('remote-user', user.name);
                appData.put('user-data', user);
            },

            _clearUser: function () {
                appData.put('remote-user', '');
                appData.put('user-data', null);
            },

            on: function (event, handler) {
                events.user.on('auth.' + event, handler);
            },

            off: function (event, handler) {
                events.user.off('auth.' + event, handler);
            }
        };

        return auth;
    }
);
