define('feature/issues/actions-view',
[
    'zepto',
    'backbone',
    'feature/users/user-model',
    'util/ajax',
    'util/app-data',
    'util/misc',
    'util/router'
],
function (
    $,
    Backbone,
    UserModel,
    ajaxUtil,
    appData,
    misc,
    router
) {

    var MESSAGES = {
        0: AJS.I18n.getText('jira.mobile.error.network.failure.desc'),
        401: AJS.I18n.getText('jira.mobile.error.permission.violation.desc'), // This is what JIRA returns for permission violation
        403: AJS.I18n.getText('jira.mobile.error.permission.violation.desc'),
        404: AJS.I18n.getText('jira.mobile.action.error.notfound'), // This is the generic error case for JIRA's REST endpoints...
        'default': AJS.I18n.getText('jira.mobile.action.error.unknown')
    };

    return Backbone.View.extend({
        className: 'issue-actions no-flicker',

        initialize: function (options) {
            this.parent = options.parent;
            this.issueKey = options.model.get('key');
            this.isAndroid = $('body').hasClass('platform-android');
            this.errorEventsBound = false;
        },

        events: {
            'click .watch-button': 'watchIssue',
            'click .unwatch-button': 'unwatchIssue',
            'click .unvote-button': 'unvoteIssue',
            'click .vote-button': 'voteIssue',
            'click .self-assign-button': 'assignToMe',
            'click .user-assign-button': 'showAssignOverlay'
        },

        showOverlay: function() {
            $('.mob-overlay.actions').addClass('show');
            $('.action-menu-button').addClass('active');
            // Redraw is needed for android to show the full overlay
            // Redraw causes flickering in ios
            if (this.isAndroid) {
                misc.forceRepaint();
            }
        },

        hideOverlay: function() {
            $('.mob-overlay.actions').removeClass('show');
            $('.action-menu-button').removeClass('active');
        },

        toggleShowActions: function () {
            if (!this.$el.hasClass('expand')) {
                this.showOverlay();
                this.render();
            } else {
                this.hideOverlay();
            }
            this.$el.toggleClass('expand');
        },

        closeActionsIfOpen: function () {
            if (this.$el.hasClass('expand')) {
                this.$el.removeClass('expand');
                this.hideOverlay();
            }
        },

        voteIssue: function () {
            var url = '/rest/api/2/issue/' + this.issueKey + '/votes';
            var that = this;
            this._actionEventPrep();
            ajaxUtil.rest({
                type: 'POST',
                url: url,
                success: function (data, status, xhr) {
                    that.model.attributes.permissions.voterInfo.hasVoted = true;
                    var voteObj = that.model.attributes.fields.votes;
                    voteObj.votes++;
                    that.model.setField("votes", voteObj);
                    that._actionSuccessHandler();
                },
                error: function (xhr, errorType, error, data) {
                    that._actionErrorHandler(xhr, errorType, error, data, that.voteIssue);
                }
            });
        },

        unvoteIssue: function () {
            var url = '/rest/api/2/issue/' + this.issueKey + '/votes';
            var that = this;
            this._actionEventPrep();
            ajaxUtil.rest({
                type: 'DELETE',
                url: url,
                success: function (data, status, xhr) {
                    that.model.attributes.permissions.voterInfo.hasVoted = false;
                    var voteObj = that.model.attributes.fields.votes;
                    voteObj.votes--;
                    that.model.setField("votes", voteObj);
                    that._actionSuccessHandler();
                },
                error: function (xhr, errorType, error, data) {
                    that._actionErrorHandler(xhr, errorType, error, data, that.unvoteIssue);
                }
            });
        },

        watchIssue: function () {
            var url = '/rest/api/2/issue/' + this.issueKey + '/watchers';
            var data = appData.getUsername();
            var that = this;
            this._actionEventPrep();
            ajaxUtil.rest({
                type: 'POST',
                url: url,
                data: JSON.stringify(data),
                success: function (data, status, xhr) {
                    that.model.attributes.permissions.watcherInfo.isWatching = true;
                    var watchesObj = that.model.attributes.fields.watches;
                    watchesObj.watchCount++;
                    that.model.setField("watches", watchesObj);
                    that._actionSuccessHandler();
                },
                error: function (xhr, errorType, error, data) {
                    that._actionErrorHandler(xhr, errorType, error, data, that.watchIssue);
                }
            });
        },

        unwatchIssue: function () {
            var user = appData.getUsername();
            var url = '/rest/api/2/issue/' + this.issueKey + '/watchers?username=' + user;
            var that = this;
            this._actionEventPrep();
            ajaxUtil.rest({
                type: 'DELETE',
                url: url,
                success: function (data, status, xhr) {
                    that.model.attributes.permissions.watcherInfo.isWatching = false;
                    var watchesObj = that.model.attributes.fields.watches;
                    watchesObj.watchCount--;
                    that.model.setField("watches", watchesObj);
                    that._actionSuccessHandler();
                },
                error: function (xhr, errorType, error, data) {
                    that._actionErrorHandler(xhr, errorType, error, data, that.unwatchIssue);
                }
            });
        },

        showAssignOverlay: function () {
            var permissions = this.model.get('permissions');
            var searchUrl = '/rest/api/latest/user/assignable/search?issueKey=' + encodeURIComponent(this.model.get('key'));
            var unknownAvatarURL = appData.get('context-path') + '/secure/useravatar?ownerId=-1';

            var automaticAssignee = {
                name: '-1',
                displayName: AJS.I18n.getText('jira.mobile.issues.assignee.automatic'),
                avatarUrls: {
                    '48x48': unknownAvatarURL
                },
                isDefaultFallback: true
            };
            var unassignedAssignee = {
                name: '',
                displayName: AJS.I18n.getText('jira.mobile.issues.assignee.unassigned'),
                avatarUrls: {
                    '48x48': unknownAvatarURL
                }
            };

            var currentAssignee = this.model.get('fields').assignee;
            var specialAssignees = [automaticAssignee];
            if (permissions.unassignedEnabled && currentAssignee && currentAssignee.name !== unassignedAssignee.name) {
                specialAssignees.push(unassignedAssignee);
            }

            var defaultUser = currentAssignee || (permissions.unassignedEnabled ? unassignedAssignee : automaticAssignee);

            var PickerView = require('widget/user-picker/overlay-view');
            var options = {
                titleText: AJS.I18n.getText('jira.mobile.issues.assignto.title', this.model.get('key')),
                actionText: AJS.I18n.getText('jira.mobile.issues.assignto.submit'),
                cancelText: AJS.I18n.getText('jira.mobile.issues.assignto.cancel'),
                dataSrc: searchUrl,
                callback: _.bind(this.assignToUser, this),
                defaultUser: new UserModel(defaultUser),
                extraUsers: specialAssignees
            };

            var view = new PickerView(options);
            router.presenter.showOverlay('assignToUser', view);
            this.closeActionsIfOpen();
        },

        assignToMe: function () {
            this.assignToUser(appData.getUsername());
        },

        assignToUser: function (user) {
            if (!user) {
                return;
            }
            var url = '/assignee/' + this.issueKey;
            var username = user;
            if (user instanceof Backbone.Model) {
                username = user.get('name');
            }
            if (username === '') {
                username = null; // Null == Unassigned
            }
            var data = {
                name: username
            };
            var that = this;
            this._actionEventPrep();
            ajaxUtil.rest({
                type: 'PUT',
                url: url,
                data: JSON.stringify(data),
                success: function (data, status, xhr) {
                    that.model.attributes.permissions.isSelfAssignable = data.canAssignToSelf;
                    that.model.setField('assignee', data.assignee);
                    that._actionSuccessHandler();
                },
                error: function (xhr, errorType, error, data) {
                    that._actionErrorHandler(xhr, errorType, error, data, function () {
                        that.assignToUser(user);
                    });
                }
            });
        },

        bindErrorEvents: function () {
            // Manually bind some delegate events for the error box
            var that = this;
            this._getErrorContainer().on('click', '.error-cancel-button', function() {
                that.closeErrorBox();
            });
            this._getErrorContainer().on('click', '.retry-button', function(){
                that.doRetryAction();
            });
        },

        doRetryAction: function () {
            if (this.retryAction) {
                this.retryAction();
                this.retryAction = null;
            }
        },

        closeErrorBox: function () {
            this._getErrorContainer().empty();
        },

        // Actions should do this first
        _actionEventPrep: function () {
            // Close the action menu
            this.closeActionsIfOpen();
            // Start the spinner
            this.parent.showHeaderLoading();
        },

        _actionSuccessHandler: function() {
            // Stop/hide spinner
            this.parent.showSuccess();
            // Hide error message
            this.closeErrorBox();
            // Clear the retry method
            this.retryAction = null;
        },

        _actionErrorHandler: function (xhr, errorType, error, data, origin) {
            xhr.errorHandled = true;
            if (errorType === 'abort') {
                xhr.aborted = true;
                return;
            }
            if (!this.errorEventsBound) {
                this.bindErrorEvents();
                this.errorEventsBound = true;
            } else {
                this.closeErrorBox();
            }
            // Stop spinner
            this.parent.hideLoading();
            // Draw error box with error message
            var key = data && data.reasonCode;
            var msg;
            if (key) {
                msg = MESSAGES[key];
            } else {
                msg = MESSAGES[xhr.status] || MESSAGES['default'];
            }
            this._getErrorContainer().append(AtlassianMobile.Templates.JIRA.Issue.drawActionError({
                message: msg
            }));
            // make some mechanism for retrying the request
            this.retryAction = origin;
        },

        _getErrorContainer: function () {
            if (!this.errorContainer) {
                this.errorContainer = $('.action-error-container');
            }
            return this.errorContainer;
        },

        render: function () {
            this.$el.empty();
            var permissions = this.model.get('permissions');
            var canSelfAssign = !!this.model.get('names').assignee && permissions.isSelfAssignable;

            this.$el.append(AtlassianMobile.Templates.JIRA.Issue.drawActionMenu({
                watcherInfo: permissions.watcherInfo,
                voterInfo: permissions.voterInfo,
                canAssign: permissions.canAssign,
                canSelfAssign: canSelfAssign
            }));

            return this;
        }
    });
});