define('wiki-edit/UndoableTextarea', [
        'wiki-edit/UndoManager',
        'wiki-edit/KeyTester',
        'jquery'
], function(
    UndoManager,
    KeyTester,
    $
) {

    var CUT_PASTE_MODE = "cut-paste";
    var _lastEditMode = "";

    function _setEditMode(context, newMode) {
        if (newMode === CUT_PASTE_MODE || _lastEditMode != newMode) {
            if (_lastEditMode != "newline") {
                context.recordHistoryItem();
            }
            _lastEditMode = newMode;
        }
    }

    function _manipulateHistory(context, stateName) {
        if (context.undoManager["can" + stateName[0].toUpperCase() + stateName.substring(1)]()) {
            var state = context.undoManager[stateName]();
            context.element.value = state.value;
            context.element.selectionStart = state.selectionStart;
            context.element.selectionEnd = state.selectionEnd;
        }
    }

    var UndoableTextarea = function UndoableTextarea(element) {
        this.element = element;
        this.$el = $(element);
        this.undoManager = new UndoManager();
        this.undoManager.updateCurrent(this.getValue());
        this.undoManager.push(this.getValue());

        var handleKeyboardInput = (function handleChange(e) {
            var keyCode = e.keyCode;
            var keyCodeChar = String.fromCharCode(keyCode);

            if (keyCode) {
                if (!e.ctrlKey && !e.metaKey) {
                    this.undoManager.updateCurrent(this.getValue());
                    _setEditMode(this, KeyTester.getActionType(keyCode));
                } else if ((e.ctrlKey || e.metaKey) && !e.altKey) {
                    switch (keyCodeChar.toLowerCase()) {
                        case "y":
                            this.undoManager.updateCurrent(this.getValue(), true);
                            this.redo();
                            e.preventDefault();
                            break;

                        case "z":
                            this.undoManager.updateCurrent(this.getValue(), true);
                            if (!e.shiftKey) {
                                this.undo();
                            } else {
                                this.redo();
                            }
                            e.preventDefault();
                            break;
                    }
                }
            } else {
                this.undoManager.updateCurrent(this.getValue());
                _setEditMode(this, "other");
            }
        }).bind(this);

        this.$el.on("keydown", handleKeyboardInput);

        this.$el.on("paste cut", function handleCutAndPaste() {
            this.undoManager.updateCurrent(this.getValue());
            _setEditMode(this, CUT_PASTE_MODE);
        }.bind(this));
    };

    UndoableTextarea.prototype.getValue = function() {
        return {
            value: this.element.value,
            selectionStart: this.element.selectionStart,
            selectionEnd: this.element.selectionEnd
        };
    };

    UndoableTextarea.prototype.updateCurrent = function() {
        this.undoManager.updateCurrent(this.getValue());
    };

    UndoableTextarea.prototype.recordHistoryItem = function() {
        this.undoManager.push(this.getValue());
    };

    UndoableTextarea.prototype.undo = function() {
        _setEditMode(this, "undo");
        _manipulateHistory(this, "undo");
        this.element.scrollTop = this.element.scrollHeight;
    };

    UndoableTextarea.prototype.redo = function() {
        _manipulateHistory(this, "redo");
        this.element.scrollTop = this.element.scrollHeight;
    };

    return UndoableTextarea;
});