define('wiki-edit/SpeechRecognition', ['jira/util/formatter', 'jira/util/data/meta', 'aui/flag'], function (formatter, Meta, auiFlag) {
    function getIsFinal(results) {
        return results && results[0] && results[0].isFinal;
    }

    function getTranscript(results) {
        return results && results[0] && results[0][0] && results[0][0].transcript || '';
    }

    return {
        start: function (textareaInstance) {
            var self = this;
            var recognition = new webkitSpeechRecognition();
            recognition.continuous = false;
            recognition.interimResults = true;
            // if lang is converted into something that Chrome does not understand, it defaults to the system language
            recognition.lang = Meta.get("user-locale").replace("_", "-");
            // stops any recognition in progress to start a new session
            recognition.stop();
            recognition.start();

            recognition.onstart = function () {
                self.showTimedMessage({title: formatter.I18n.getText("jira.wiki.editor.operation.speech.start")}, "info", 10);
                self.selectionStart = textareaInstance.manipulationEngine.getSelection().start;
            };
            recognition.onresult = function (event) {
                self.dismissCurrentMessage();
                var isFinal = getIsFinal(event.results);
                var speechResult = getTranscript(event.results);
                // capitalise first letter
                speechResult = speechResult.charAt(0).toUpperCase() + speechResult.slice(1);

                if (isFinal) {
                    textareaInstance.undoRedoEl.recordHistoryItem();
                }
                textareaInstance.manipulationEngine.replaceSelectionWith.call(textareaInstance.manipulationEngine, speechResult, false);
                var selectionEnd = self.selectionStart + speechResult.length;
                if (isFinal) {
                    // clears the selection
                    textareaInstance.manipulationEngine.setSelection(selectionEnd);
                    textareaInstance.undoRedoEl.recordHistoryItem();
                    textareaInstance.$el.trigger("input");
                    recognition.stop();
                }
                else {
                    textareaInstance.manipulationEngine.setSelection(self.selectionStart, selectionEnd);
                }
            };
            recognition.onerror = function (event) {
                recognition.stop();

                // this is a generic error message in case no specific error is raised
                var message = {
                    title: formatter.I18n.getText("jira.wiki.editor.operation.speech.error.general"),
                    body: ''
                };
                switch (event.error) {
                    case "not-allowed":
                        message.title = formatter.I18n.getText("jira.wiki.editor.operation.speech.error.not-allowed");
                        message.body = formatter.I18n.getText("jira.wiki.editor.operation.speech.error.not-allowed.body");
                        break;
                    case "network":
                        message.title = formatter.I18n.getText("jira.wiki.editor.operation.speech.error.network");
                        message.body = formatter.I18n.getText("jira.wiki.editor.operation.speech.error.network.body");
                        break;
                    case "no-speech":
                        message.title = formatter.I18n.getText("jira.wiki.editor.operation.speech.error.no-speech");
                        message.body = formatter.I18n.getText("jira.wiki.editor.operation.speech.error.no-speech.body");
                        break;
                    // this only happens by user interaction, so there is no need to display a message
                    case "aborted":
                        return;
                }
                self.showTimedMessage(message, "warning", 5);
            };
            recognition.onend = function () {
                recognition.stop();
            }
        },

        /**
         * Shows a specific auiFlag and automatically hides it after the specified timeout
         *
         * @param message
         * @param flagType
         * @param duration in seconds
         */
        showTimedMessage: function (message, flagType, duration) {
            this.dismissCurrentMessage();
            this.speakFlag = auiFlag({
                type: flagType,
                title: message.title,
                body: message.body,
                persistent: true
            });

            clearTimeout(this.flagTimer);
            this.flagTimer = setTimeout(this.dismissCurrentMessage.bind(this), duration * 1000);
        },
        dismissCurrentMessage: function () {
            if (this.speakFlag && this.speakFlag.close) {
                this.speakFlag.close();
                this.speakFlag = null;
            }
        }
    }
});
