AJS.test.require(["com.atlassian.jira.jira-user-profile-plugin:user-details-resources"], function() {

    var jQuery = require('jquery');
    var _ = require('underscore');
    module("JIRA.UserProfile.ProfileDetails", {
        setup: function() {
            this.fixture = jQuery('#qunit-fixture');
            this.initProfilePage({
                baseUrl: "http://myhost",
            });
            this.ProfileDetails = require('jira/user-profile-plugin/profile-details');
        },

        initProfilePage: function(opts) {
            this.fixture.empty();
            this.defaultBaseUrl = opts.baseUrl;
            this.avatarSrc = (opts.avatarSrc) ? opts.avatarSrc : opts.baseUrl + "/avatar.jpg";

            this.createUserProfileHeader();
            this.createDetailsFragment();
        },

        /**
         * Creates the DOM for the user profile header.
         */
        createUserProfileHeader: function() {
            this.profileHeader = jQuery('<div class="aui-page-header"></div>').appendTo(this.fixture);
            this.headerAvatarContainer = jQuery('<div class="aui-avatar-large"></div>').appendTo(this.profileHeader);
            this.headerAvatarImg = jQuery(AJS.format('<img src="{0}"/>', this.avatarSrc)).appendTo(this.headerAvatarContainer);
        },

        /**
         * Creates the DOM for the details fragment.
         */
        createDetailsFragment: function() {
            this.detailsFragment = jQuery('<div></div>').appendTo(this.fixture);
            this.detailsAvatarImg = jQuery(AJS.format('<img class="avatar-image" src="{0}"/>', this.avatarSrc)).appendTo(this.detailsFragment);
            this.lozengeContainer = jQuery('<div class="gravatar-lozenge-container"></div>').appendTo(this.detailsFragment);
        },

        /**
         * @param {options} [options]
         * @param {string} [options.baseUrl] a baseUrl
         * @return JIRA.UserProfile.ProfileDetails
         */
        createProfileDetails: function(options) {
            return new this.ProfileDetails(_.defaults({}, options, {
                el: this.fixture,
                baseUrl: this.defaultBaseUrl
            }));
        }
    });

    test("ProfileDetails updates avatar in profile header after avatar change", function() {
        var profileDetails = this.createProfileDetails();

        var newAvatarSrc = "http://bogus/avatar";
        profileDetails.onAvatarChanged(newAvatarSrc);
        equal(this.headerAvatarImg.attr('src'), newAvatarSrc, "view should update avatar in page header");
    });

    test("ProfileDetails shows gravatar lozenge for external avatars on DOMReady", function() {
        this.initProfilePage({
            avatarSrc : 'http://www.gravatar.com/avatar/00000000000000000000000000000000'
        });
        var profileDetails = this.createProfileDetails();

        var lozenge = this.lozengeContainer.find('.aui-lozenge');
        equal(lozenge.length, 1, 'should show a lozenge');
        equal(lozenge.text(), "Gravatar", 'lozenge should say "Gravatar"');
    });

    test("ProfileDetails handles http URL with port in it", function() {
        var baseUrl = this.defaultBaseUrl + ":80";
        this.initProfilePage({ baseUrl: baseUrl});

        var lozenge = this.lozengeContainer.find('.aui-lozenge');
        equal(lozenge.length, 0, 'should not show lozenge for baseUrl: ' + baseUrl);
    });

    test("ProfileDetails handles https URL with port in it", function() {
        var baseUrl = "https://the_host:443";
        this.initProfilePage({ baseUrl: baseUrl });

        var lozenge = this.lozengeContainer.find('.aui-lozenge');
        equal(lozenge.length, 0, 'should not show lozenge for baseUrl: ' + baseUrl);
    });

    test("ProfileDetails does not show gravatar lozenge for internal avatars on DOMReady", function() {
        this.createProfileDetails();

        var lozenge = this.lozengeContainer.find('.aui-lozenge');
        equal(lozenge.length, 0, 'should not show a lozenge');
    });

    test("ProfileDetails shows gravatar lozenge on changing from internal avatar to external", function() {
        var profileDetails = this.createProfileDetails();
        equal(this.lozengeContainer.find('.aui-lozenge').length, 0, 'initially there should be no lozenge');

        profileDetails.onAvatarChanged("http://www.gravatar.com/avatar/00000000000000000000000000000000");

        var lozenge = this.lozengeContainer.find('.aui-lozenge');
        equal(lozenge.length, 1, 'should show a lozenge');
        equal(lozenge.text(), "Gravatar", 'lozenge should say "Gravatar"');
    });

    test("ProfileDetails hides gravatar lozenge on changing from gravatar to internal", function() {
        var baseUrl = 'http://otherhost';
        this.initProfilePage({
            avatarSrc : 'http://www.gravatar.com/avatar/00000000000000000000000000000000'
        });
        var profileDetails = this.createProfileDetails();

        equal(this.lozengeContainer.find('.aui-lozenge').length, 1, 'initially there should be a lozenge');

        profileDetails.onAvatarChanged(baseUrl + "/newimage.jpg");

        var lozenge = this.lozengeContainer.find('.aui-lozenge');
        equal(lozenge.length, 0, 'should hide the lozenge');
    });

    test("ProfileDetails updates avatar in profile header after avatar change to a relative URL", function() {
        var profileDetails = this.createProfileDetails();

        var newAvatarSrc = "/jira/secutre/someavatar";
        profileDetails.onAvatarChanged(newAvatarSrc);
        equal(this.headerAvatarImg.attr('src'), newAvatarSrc, "view should update avatar in page header");
        var lozenge = this.lozengeContainer.find('.aui-lozenge');
        equal(lozenge.length, 0, 'should not show a lozenge');
    });

    test("ProfileDetails removes gravatar lozenge on changing from external avatar to relative internal", function() {
        var profileDetails = this.createProfileDetails();
        equal(this.lozengeContainer.find('.aui-lozenge').length, 0, 'initially there should be no lozenge');

        profileDetails.onAvatarChanged("http://www.gravatar.com/avatar/00000000000000000000000000000000");

        var lozenge = this.lozengeContainer.find('.aui-lozenge');
        equal(lozenge.length, 1, 'should show a lozenge while pointing to an external avatar');
        equal(lozenge.text(), "Gravatar", 'lozenge should say "Gravatar"');

        profileDetails.onAvatarChanged("/jira/secutre/someavatar");
        var lozenge2 = this.lozengeContainer.find('.aui-lozenge');
        equal(lozenge2.length, 0, 'should not show a lozenge after changing to a relative url');
    });

    test("ProfileDetails removes gravatar lozenge on changing from external avatar to absolute internal", function() {
        var baseUrl = this.defaultBaseUrl;
        var profileDetails = this.createProfileDetails();
        equal(this.lozengeContainer.find('.aui-lozenge').length, 0, 'initially there should be no lozenge');

        profileDetails.onAvatarChanged("http://www.gravatar.com/avatar/00000000000000000000000000000000");

        var lozenge = this.lozengeContainer.find('.aui-lozenge');
        equal(lozenge.length, 1, 'should show a lozenge while pointing to an external avatar');
        equal(lozenge.text(), "Gravatar", 'lozenge should say "Gravatar"');

        profileDetails.onAvatarChanged(baseUrl + "/newimage.jpg");
        var lozenge2 = this.lozengeContainer.find('.aui-lozenge');
        equal(lozenge2.length, 0, 'should not show a lozenge after changing to a relative url');
    });
});
