AJS.test.require(['com.atlassian.jira.jira-tzdetect-plugin:tzdetect-lib'], function(){
    "use strict";

    window.__tzTesting = true;
    var $ = jQuery;

    var tzTestUtils = {
        defaultPrefs: {
            tzid: 'Australia/Eucla',
            tzname: '(GMT+8:45) Eucla',
            janOffset: 525,
            julyOffset: 525
        },
        curPrefs: {},
        setPref: function (key, value) {
            tzTestUtils.curPrefs[key] = value;
        },
        resetPrefs: function () {
            tzTestUtils.curPrefs = $.extend({}, tzTestUtils.defaultPrefs);
        },
        // Make sure we're not relying on translated strings
        i18n: function (string) {
            return '~~!' + string + '!~~';
        },
        // For mocking XHR responses
        generateZoneData: function (zones) {
            if (typeof zones === 'string') {
                zones = [zones];
            }
            var regionKeys = {};
            var regionData = [];
            var zoneData = [];
            var zoneName, regionName, cityName, split;
            for (var i = 0, ii = zones.length; i < ii; i++) {
                zoneName = zones[i];
                split = zoneName.split('/');
                regionName = split[0];
                cityName = split[1];
                if (!regionKeys[regionName]) {
                    regionKeys[regionName] = true;
                    regionData.push({key: regionName, displayName: tzTestUtils.i18n(regionName)});
                }
                zoneData.push({
                    regionKey: regionName,
                    city: cityName,
                    timeZoneId: zoneName,
                    displayName: tzTestUtils.i18n(regionName) + '/' + tzTestUtils.i18n(cityName),
                    gmtOffset: '(GMT+00:00)' // The exact offset doesn't matter for test data
                });
            }
            return {
                regions: regionData,
                zones: zoneData
            };
        }
    };

    module('tzdetect-banner', {
        setup: function () {
            tzTestUtils.resetPrefs();
            this.banner = JIRA.plugins.tzBanner;
            this.prefsStub = sinon.stub(this.banner, 'getPreferences').returns(tzTestUtils.curPrefs);
            $(this.banner.$tzFlag).remove();
            this.banner.$tzFlag = null;
        },
        teardown: function () {
            this.prefsStub.restore();
        }
    });


    /** Basic showing conditions **/

    test('banner is not re-shown if it already exists', function () {
        var stub = this.stub(this.banner, 'detectTimezoneOffsets');
        this.banner.init();
        equal(stub.callCount, 1, 'detectTimezoneOffsets() is called in blank state');

        this.banner.$tzFlag = $('<div/>');
        this.banner.init();
        equal(stub.callCount, 1, 'detectTimezoneOffsets() is not called when the banner exists');
    });

    test('banner is not shown when no tzid is found', function () {
        var stub = this.stub(this.banner, 'create');
        tzTestUtils.setPref('tzid', '');
        this.banner.init();
        equal(stub.callCount, 0, 'create() is not called');
    });

    test('banner is not shown when zone offsets match', function () {
        var stub = this.stub(this.banner, 'create');
        this.stub(this.banner, 'getTzOffset').returns(525);
        this.banner.init();
        equal(stub.callCount, 0, 'create() is not called');
    });

    test('banner is shown when zone offsets do not match', function () {
        var stub = this.stub(this.banner, 'create');
        this.stub(this.banner, 'getTzOffset').returns(600);
        this.banner.init();
        equal(stub.callCount, 1, 'create() is called');
    });

    test('banner silently does nothing if AJS.params is missing', function () {
        this.prefsStub.restore();
        var oldParams = AJS.params;
        AJS.params = {};
        var prefs = this.banner.getPreferences();
        var emptyPrefs = {tzid:undefined, tzname:undefined, janOffset:0, julyOffset:0};
        deepEqual(prefs, emptyPrefs, 'getPreferences() returns undefined for all keys');
        AJS.params = oldParams;
    });


    /** Clicking on the update link **/

    test('update for a single matching zone', function () {
        var stub = this.stub(this.banner, 'setUserTimeZone');
        var data = tzTestUtils.generateZoneData('Australia/Darwin');
        this.banner.handleZoneData(data);
        ok(stub.calledWith('Australia/Darwin'), 'A single zone sets the user pref');
    });

    test('update for multiple matching zones', function () {
        var stub = this.stub(this.banner, 'setUserTimeZone');
        var data = tzTestUtils.generateZoneData(['Australia/Sydney', 'Australia/Melbourne', 'Australia/Hobart']);
        this.banner.$link = $();
        this.banner.$dropdown = null;
        this.banner.handleZoneData(data);
        equal(stub.callCount, 0, 'User pref is not set');
        ok(this.banner.$dropdown instanceof $, 'Menu is created');
    });

    test('GMT offset zones are filtered out in menu', function () {
        var data = tzTestUtils.generateZoneData(['Australia/Perth', 'Asia/Singapore', 'Etc/GMT-8']);
        this.banner.$link = $();
        this.banner.$dropdown = null;
        this.banner.handleZoneData(data);
        equal(this.banner.$dropdown.find('a[data-tzid]').length, 2, 'GMT offset zone is ignored');
    });

    test('GMT offset and Antarctica zones are filtered out before getting to the menu', function () {
        var stub = this.stub(this.banner, 'setUserTimeZone');
        var data = tzTestUtils.generateZoneData(['Australia/Perth', 'Antarctica/Casey', 'Etc/GMT-8']);
        this.banner.$link = $();
        this.banner.$dropdown = null;
        this.banner.handleZoneData(data);
        ok(stub.calledWith('Australia/Perth'), 'Single zone after ignoring GMT offset sets user pref');
        equal(this.banner.$dropdown, null, 'Banner is not shown');
    });


    /** Tracking **/

    test('banner tracks analytics when shown', function () {
        var stub = this.stub(this.banner, 'track');
        this.stub(this.banner, 'getTzOffset').returns(600);
        this.banner.init();
        ok(stub.calledWith('shown'), 'Banner showing is tracked');
    });

    test('banner tracks analytics when update link is clicked', function () {
        var stub = this.stub(this.banner, 'track');
        this.stub(this.banner, 'redirect');
        this.banner.$link = $();

        var data = tzTestUtils.generateZoneData([]);
        this.banner.handleZoneData(data);
        ok(stub.calledWith('clicked.update', {matchingZoneCount: 0}), 'Update with no matching zones is tracked');
    });

    test('banner tracks analytics when zone is set', function () {
        var stub = this.stub(this.banner, 'track');
        this.stub(JIRA.SmartAjax, 'makeRequest');
        this.banner.setUserTimeZone('Australia/Eucla', 'qunit');
        ok(stub.calledWith('setzone', {zoneId: 'Australia/Eucla', source: 'qunit'}), 'Setting a zone is tracked with the zone ID');
    });

});

